/*
 * Decompiled with CFR 0.152.
 */
package qupath.ext.extensionmanager.gui;

import java.io.IOException;
import java.net.URI;
import java.nio.file.InvalidPathException;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.ResourceBundle;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import javafx.application.Platform;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.event.ActionEvent;
import javafx.fxml.FXML;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Alert;
import javafx.scene.control.Button;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.Label;
import javafx.scene.control.MenuItem;
import javafx.scene.control.OverrunStyle;
import javafx.scene.control.SelectionMode;
import javafx.scene.control.TableCell;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableRow;
import javafx.scene.control.TableView;
import javafx.scene.control.TextField;
import javafx.scene.control.Tooltip;
import javafx.scene.input.Clipboard;
import javafx.scene.input.ClipboardContent;
import javafx.stage.Stage;
import javafx.stage.Window;
import javafx.util.Callback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.ext.extensionmanager.core.ExtensionCatalogManager;
import qupath.ext.extensionmanager.core.catalog.Catalog;
import qupath.ext.extensionmanager.core.catalog.CatalogFetcher;
import qupath.ext.extensionmanager.core.savedentities.SavedCatalog;
import qupath.ext.extensionmanager.core.tools.GitHubRawLinkFinder;
import qupath.ext.extensionmanager.gui.ExtensionCatalogModel;
import qupath.ext.extensionmanager.gui.ProgressWindow;
import qupath.ext.extensionmanager.gui.UiUtils;
import qupath.fx.dialogs.Dialogs;

class CatalogManager
extends Stage {
    private static final Logger logger = LoggerFactory.getLogger(CatalogManager.class);
    private static final ResourceBundle resources = UiUtils.getResources();
    private static final String CATALOG_FILE_NAME = "catalog.json";
    private final ExtensionCatalogManager extensionCatalogManager;
    private final Runnable onInvalidExtensionDirectory;
    @FXML
    private TableView<SavedCatalog> catalogTable;
    @FXML
    private TableColumn<SavedCatalog, String> nameColumn;
    @FXML
    private TableColumn<SavedCatalog, URI> urlColumn;
    @FXML
    private TableColumn<SavedCatalog, String> descriptionColumn;
    @FXML
    private TableColumn<SavedCatalog, Button> removeColumn;
    @FXML
    private TextField catalogUrl;

    public CatalogManager(ExtensionCatalogManager extensionCatalogManager, ExtensionCatalogModel model, Runnable onInvalidExtensionDirectory) throws IOException {
        this.extensionCatalogManager = extensionCatalogManager;
        this.onInvalidExtensionDirectory = onInvalidExtensionDirectory;
        UiUtils.loadFXML((Object)this, CatalogManager.class.getResource("catalog_manager.fxml"));
        this.catalogTable.setItems(model.getCatalogs());
        this.setColumns();
        this.setDoubleClickHandler();
        this.setRightClickHandler();
    }

    @FXML
    private void onAddClicked(ActionEvent ignored) {
        ProgressWindow progressWindow;
        UiUtils.promptExtensionDirectory(this.extensionCatalogManager.getExtensionDirectoryPath(), this.onInvalidExtensionDirectory);
        String catalogUrl = this.catalogUrl.getText();
        if (catalogUrl == null || catalogUrl.isBlank()) {
            return;
        }
        ExecutorService executor = Executors.newSingleThreadExecutor();
        try {
            progressWindow = new ProgressWindow(MessageFormat.format(resources.getString("CatalogManager.fetching"), catalogUrl), executor::shutdownNow);
        }
        catch (IOException e) {
            logger.error("Error while creating progress window", (Throwable)e);
            executor.shutdown();
            return;
        }
        progressWindow.initOwner((Window)this);
        progressWindow.show();
        executor.execute(() -> {
            try {
                URI uri;
                Platform.runLater(() -> progressWindow.setStatus(MessageFormat.format(resources.getString("CatalogManager.attemptingToGetRawLink"), catalogUrl)));
                try {
                    uri = GitHubRawLinkFinder.getRawLinkOfFileInRepository(catalogUrl, CATALOG_FILE_NAME::equals).get();
                }
                catch (ExecutionException e) {
                    logger.debug("Attempt to get raw link of {} failed. Considering it to be a raw link.", (Object)catalogUrl, (Object)e);
                    uri = new URI(catalogUrl);
                }
                URI finalUri = uri;
                Platform.runLater(() -> {
                    progressWindow.setProgress(0.5f);
                    progressWindow.setStatus(MessageFormat.format(resources.getString("CatalogManager.fetchingCatalogLocatedAt"), finalUri.toString()));
                });
                Catalog catalog = CatalogFetcher.getCatalog(uri).get();
                Platform.runLater(() -> progressWindow.setProgress(1.0f));
                if (this.extensionCatalogManager.getCatalogs().stream().anyMatch(savedCatalog -> savedCatalog.name().equals(catalog.name()))) {
                    this.displayErrorMessage(resources.getString("CatalogManager.cannotAddCatalog"), MessageFormat.format(resources.getString("CatalogManager.catalogAlreadyExists"), catalog.name()));
                    return;
                }
                this.extensionCatalogManager.addCatalog(List.of(new SavedCatalog(catalog.name(), catalog.description(), new URI(catalogUrl), uri, true)));
            }
            catch (Exception e) {
                logger.debug("Error when fetching catalog at {}", (Object)catalogUrl, (Object)e);
                this.displayErrorMessage(resources.getString("CatalogManager.cannotAddCatalog"), e.getLocalizedMessage());
            }
            finally {
                Platform.runLater(() -> ((ProgressWindow)progressWindow).close());
            }
        });
        executor.shutdown();
    }

    private void setColumns() {
        this.nameColumn.setCellValueFactory(cellData -> new SimpleStringProperty(((SavedCatalog)cellData.getValue()).name()));
        this.urlColumn.setCellValueFactory(cellData -> new SimpleObjectProperty((Object)((SavedCatalog)cellData.getValue()).uri()));
        this.descriptionColumn.setCellValueFactory(cellData -> new SimpleStringProperty(((SavedCatalog)cellData.getValue()).description()));
        this.removeColumn.setCellValueFactory(cellData -> {
            Button button = new Button(resources.getString("CatalogManager.remove"));
            button.setDisable(!((SavedCatalog)cellData.getValue()).deletable());
            button.setOnAction(event -> {
                UiUtils.promptExtensionDirectory(this.extensionCatalogManager.getExtensionDirectoryPath(), this.onInvalidExtensionDirectory);
                this.deleteCatalogs(List.of((SavedCatalog)cellData.getValue()));
            });
            return new SimpleObjectProperty((Object)button);
        });
        this.nameColumn.setCellFactory(CatalogManager.getStringCellFactory());
        this.urlColumn.setCellFactory(column -> {
            TableCell<SavedCatalog, URI> tableCell = new TableCell<SavedCatalog, URI>(this){

                protected void updateItem(URI item, boolean empty) {
                    super.updateItem((Object)item, empty);
                    if (empty || item == null) {
                        this.setText(null);
                        this.setTooltip(null);
                    } else {
                        this.setTextOverrun(OverrunStyle.LEADING_ELLIPSIS);
                        this.setText(item.toString());
                        this.setTooltip(new Tooltip(item.toString()));
                    }
                }
            };
            tableCell.setAlignment(Pos.CENTER_LEFT);
            return tableCell;
        });
        this.descriptionColumn.setCellFactory(CatalogManager.getStringCellFactory());
        this.removeColumn.setCellFactory(column -> {
            TableCell<SavedCatalog, Button> tableCell = new TableCell<SavedCatalog, Button>(this){

                protected void updateItem(Button item, boolean empty) {
                    super.updateItem((Object)item, empty);
                    if (empty || item == null) {
                        this.setGraphic(null);
                    } else {
                        this.setGraphic((Node)item);
                    }
                }
            };
            tableCell.setAlignment(Pos.CENTER);
            return tableCell;
        });
    }

    private void setDoubleClickHandler() {
        this.catalogTable.setRowFactory(tv -> {
            TableRow row = new TableRow();
            row.setOnMouseClicked(event -> {
                if (event.getClickCount() == 2 && !row.isEmpty()) {
                    String url = ((SavedCatalog)row.getItem()).uri().toString();
                    UiUtils.openLinkInWebBrowser(url).exceptionally(error -> {
                        logger.error("Error when opening {} in browser", (Object)url, error);
                        Dialogs.showErrorMessage((String)resources.getString("CatalogManager.browserError"), (String)MessageFormat.format(resources.getString("CatalogManager.cannotOpen"), url, error.getLocalizedMessage()));
                        return null;
                    });
                }
            });
            return row;
        });
    }

    private void setRightClickHandler() {
        this.catalogTable.getSelectionModel().setSelectionMode(SelectionMode.MULTIPLE);
        ContextMenu menu = new ContextMenu();
        this.catalogTable.setContextMenu(menu);
        MenuItem copyItem = new MenuItem(resources.getString("CatalogManager.copyUrl"));
        copyItem.setOnAction(ignored -> {
            ClipboardContent content = new ClipboardContent();
            content.putString(((SavedCatalog)this.catalogTable.getSelectionModel().getSelectedItem()).uri().toString());
            Clipboard.getSystemClipboard().setContent((Map)content);
        });
        menu.getItems().add((Object)copyItem);
        MenuItem removeItem = new MenuItem(resources.getString("CatalogManager.remove"));
        removeItem.setOnAction(ignored -> {
            UiUtils.promptExtensionDirectory(this.extensionCatalogManager.getExtensionDirectoryPath(), this.onInvalidExtensionDirectory);
            List<String> nonDeletableCatalogs = this.catalogTable.getSelectionModel().getSelectedItems().stream().filter(savedCatalog -> !savedCatalog.deletable()).map(SavedCatalog::name).toList();
            if (!nonDeletableCatalogs.isEmpty()) {
                this.displayErrorMessage(resources.getString("CatalogManager.deleteCatalog"), MessageFormat.format(resources.getString("CatalogManager.cannotBeDeleted"), nonDeletableCatalogs.size() == 1 ? nonDeletableCatalogs.getFirst() : nonDeletableCatalogs.toString()));
                return;
            }
            this.deleteCatalogs((List<SavedCatalog>)this.catalogTable.getSelectionModel().getSelectedItems());
        });
        menu.getItems().add((Object)removeItem);
    }

    private void displayErrorMessage(String title, String text) {
        new Dialogs.Builder().alertType(Alert.AlertType.ERROR).title(title).content((Node)new Label(text)).owner((Window)this).show();
    }

    private void deleteCatalogs(List<SavedCatalog> catalogs) {
        List<SavedCatalog> catalogsToDelete = catalogs.stream().filter(SavedCatalog::deletable).toList();
        if (catalogsToDelete.isEmpty()) {
            return;
        }
        boolean deleteExtensions = Dialogs.showConfirmDialog((String)resources.getString("CatalogManager.deleteCatalog"), (String)MessageFormat.format(resources.getString("CatalogManager.deleteExtensions"), catalogsToDelete.stream().map(savedCatalog -> {
            try {
                return String.format("\"%s\"", this.extensionCatalogManager.getCatalogDirectory((SavedCatalog)savedCatalog));
            }
            catch (IOException | NullPointerException | SecurityException | InvalidPathException e) {
                logger.error("Cannot retrieve path of {}", (Object)savedCatalog.name(), (Object)e);
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.joining("\n"))));
        try {
            this.extensionCatalogManager.removeCatalogs(catalogsToDelete, deleteExtensions);
        }
        catch (IOException | NullPointerException | SecurityException e) {
            logger.error("Error when removing {}", catalogsToDelete.stream().map(SavedCatalog::name).toList(), (Object)e);
            this.displayErrorMessage(resources.getString("CatalogManager.deleteCatalog"), MessageFormat.format(resources.getString("CatalogManager.cannotRemoveSelectedCatalogs"), e.getLocalizedMessage()));
        }
    }

    private static Callback<TableColumn<SavedCatalog, String>, TableCell<SavedCatalog, String>> getStringCellFactory() {
        return column -> {
            TableCell<SavedCatalog, String> tableCell = new TableCell<SavedCatalog, String>(){

                protected void updateItem(String item, boolean empty) {
                    super.updateItem((Object)item, empty);
                    if (empty || item == null) {
                        this.setText(null);
                        this.setTooltip(null);
                    } else {
                        this.setText(item);
                        this.setTooltip(new Tooltip(item));
                    }
                }
            };
            tableCell.setAlignment(Pos.CENTER_LEFT);
            return tableCell;
        };
    }
}

