/*
 * Decompiled with CFR 0.152.
 */
package qupath.ext.extensionmanager.core.tools;

import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.ext.extensionmanager.core.tools.FileTools;

class RecursiveDirectoryWatcher
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(RecursiveDirectoryWatcher.class);
    private final Map<WatchKey, Path> keys = new HashMap<WatchKey, Path>();
    private final Set<Path> addedFiles = new HashSet<Path>();
    private final ExecutorService executor = Executors.newSingleThreadExecutor(runnable -> new Thread(runnable, String.format("%s-directory-watcher", directoryToWatch.getFileName())));
    private final WatchService watchService = FileSystems.getDefault().newWatchService();
    private final int depth;
    private final Predicate<Path> filesToFind;
    private final Predicate<Path> directoriesToSkip;
    private final Consumer<Path> onFileAdded;

    public RecursiveDirectoryWatcher(Path directoryToWatch, int depth, Predicate<Path> filesToFind, Predicate<Path> directoriesToSkip, Consumer<Path> onFileAdded, Consumer<Path> onFileDeleted) throws IOException {
        this.depth = depth;
        this.filesToFind = filesToFind;
        this.directoriesToSkip = directoriesToSkip;
        this.onFileAdded = onFileAdded;
        this.registerDirectory(directoryToWatch);
        this.executor.execute(() -> {
            try {
                WatchKey key;
                while ((key = this.watchService.take()) != null) {
                    Path parentFolder = this.keys.get(key);
                    if (parentFolder == null) {
                        throw new NullPointerException(String.format("%s is missing from %s", key, this.keys));
                    }
                    for (WatchEvent<?> event : key.pollEvents()) {
                        WatchEvent.Kind<?> kind = event.kind();
                        Path filename = (Path)event.context();
                        Path filePath = parentFolder.resolve(filename);
                        if (Files.isDirectory(filePath, new LinkOption[0]) && kind == StandardWatchEventKinds.ENTRY_CREATE) {
                            this.registerDirectory(filePath);
                        }
                        if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                            List<Path> filesToRemove = this.addedFiles.stream().filter(path -> FileTools.isFileParentOfAnotherFile(filePath.toFile(), path.toFile())).toList();
                            for (Path fileToRemove : filesToRemove) {
                                onFileDeleted.accept(fileToRemove);
                                this.addedFiles.remove(fileToRemove);
                            }
                        }
                        if (filesToFind.test(filePath)) {
                            if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                                logger.debug("File {} addition detected", (Object)filePath);
                                onFileAdded.accept(filePath);
                                this.addedFiles.add(filePath);
                                continue;
                            }
                            if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                                logger.debug("File {} deletion detected", (Object)filePath);
                                onFileDeleted.accept(filePath);
                                this.addedFiles.remove(filePath);
                                continue;
                            }
                            logger.debug("Unexpected event {} with file {}. Ignoring it", kind, (Object)filePath);
                            continue;
                        }
                        logger.debug("The file or directory {} doesn't match the predicate, so it won't be reported", (Object)filePath);
                    }
                    key.reset();
                }
            }
            catch (InterruptedException e) {
                logger.debug("Watching {} interrupted", (Object)directoryToWatch, (Object)e);
                Thread.currentThread().interrupt();
            }
            catch (ClosedWatchServiceException e) {
                logger.debug("Service watching {} closed", (Object)directoryToWatch, (Object)e);
            }
            catch (IOException | NullPointerException | SecurityException e) {
                logger.error("Error when watching directory {}", (Object)directoryToWatch, (Object)e);
            }
        });
    }

    private void registerDirectory(final Path directory) throws IOException {
        logger.debug("Registering directories in {} with depth {}", (Object)directory, (Object)this.depth);
        Files.walkFileTree(directory, EnumSet.noneOf(FileVisitOption.class), this.depth, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (RecursiveDirectoryWatcher.this.directoriesToSkip.test(dir)) {
                    logger.debug("Skipping {} as it doesn't match the predicate", (Object)dir);
                    return FileVisitResult.SKIP_SUBTREE;
                }
                logger.debug("Registering {}", (Object)dir);
                RecursiveDirectoryWatcher.this.keys.put(dir.register(RecursiveDirectoryWatcher.this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE), dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (RecursiveDirectoryWatcher.this.filesToFind.test(file)) {
                    logger.debug("File {} detected while visiting {}", (Object)file, (Object)directory);
                    RecursiveDirectoryWatcher.this.onFileAdded.accept(file);
                    RecursiveDirectoryWatcher.this.addedFiles.add(file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdownNow();
        this.executor.close();
        this.watchService.close();
    }
}

