/*
 * Decompiled with CFR 0.152.
 */
package qupath.ext.extensionmanager.core.tools;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.OptionalInt;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileDownloader {
    private static final Logger logger = LoggerFactory.getLogger(FileDownloader.class);
    private static final Duration REQUEST_TIMEOUT = Duration.ofSeconds(10L);
    private static final String CONTENT_LENGTH_ATTRIBUTE = "content-length";
    private static final int BUFFER_SIZE = 4096;

    private FileDownloader() {
        throw new AssertionError((Object)"This class is not instantiable.");
    }

    public static void downloadFile(URI uri, Path outputPath, Consumer<Float> onProgress) throws IOException, InterruptedException {
        if (!"http".equals(uri.getScheme()) && !"https".equals(uri.getScheme())) {
            throw new IllegalArgumentException(String.format("Unknown scheme %s in %s", uri.getScheme(), uri));
        }
        logger.debug("Sending request to {}", (Object)uri);
        try (HttpClient client = HttpClient.newBuilder().followRedirects(HttpClient.Redirect.ALWAYS).build();){
            HttpResponse<InputStream> response = client.send(HttpRequest.newBuilder().uri(uri).timeout(REQUEST_TIMEOUT).GET().build(), HttpResponse.BodyHandlers.ofInputStream());
            if (response.statusCode() != 200) {
                throw new RuntimeException(String.format("Request to %s failed with status code %d.", uri, response.statusCode()));
            }
            logger.debug("Got response from {} with status 200", (Object)uri);
            OptionalInt contentLength = FileDownloader.getContentLength(response);
            try (InputStream inputStream = response.body();
                 FileOutputStream fileOutputStream = new FileOutputStream(outputPath.toFile());){
                int bytesRead;
                byte[] buffer = new byte[4096];
                long bytesDownloaded = 0L;
                logger.debug("Starting downloading body of {}", (Object)uri);
                while ((bytesRead = inputStream.read(buffer)) > -1) {
                    logger.trace("{} bytes downloaded", (Object)bytesRead);
                    bytesDownloaded += (long)bytesRead;
                    if (contentLength.isPresent()) {
                        onProgress.accept(Float.valueOf((float)bytesDownloaded / (float)contentLength.getAsInt()));
                    }
                    fileOutputStream.write(buffer, 0, bytesRead);
                    if (!Thread.interrupted()) continue;
                    throw new InterruptedException(String.format("Download of %s interrupted", uri));
                }
                logger.debug("Download of {} ended successfully", (Object)uri);
            }
        }
    }

    private static OptionalInt getContentLength(HttpResponse<?> response) {
        if (!response.headers().map().containsKey(CONTENT_LENGTH_ATTRIBUTE)) {
            logger.debug("{} not found in {}. Cannot indicate progress of {} download", new Object[]{CONTENT_LENGTH_ATTRIBUTE, response.headers().map(), response.uri()});
            return OptionalInt.empty();
        }
        List<String> contentLengthEntry = response.headers().map().get(CONTENT_LENGTH_ATTRIBUTE);
        if (contentLengthEntry.isEmpty()) {
            logger.debug("{} empty in {}. Cannot indicate progress of {} download", new Object[]{CONTENT_LENGTH_ATTRIBUTE, response.headers().map(), response.uri()});
            return OptionalInt.empty();
        }
        String contentLengthText = contentLengthEntry.getFirst();
        try {
            int contentLength = Integer.parseInt(contentLengthText);
            logger.debug("Found content length of {} bytes in {}", (Object)contentLength, response.headers().map());
            return OptionalInt.of(contentLength);
        }
        catch (NumberFormatException e) {
            logger.debug("The {} header {} cannot be converted to a number. Cannot indicate progress of {} download", new Object[]{CONTENT_LENGTH_ATTRIBUTE, contentLengthText, response.uri(), e});
            return OptionalInt.empty();
        }
    }
}

