/*
 * Decompiled with CFR 0.152.
 */
package qupath.ext.extensionmanager.core;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.collections.ObservableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.ext.extensionmanager.core.catalog.Extension;
import qupath.ext.extensionmanager.core.savedentities.InstalledExtension;
import qupath.ext.extensionmanager.core.savedentities.Registry;
import qupath.ext.extensionmanager.core.savedentities.SavedCatalog;
import qupath.ext.extensionmanager.core.tools.FileTools;
import qupath.ext.extensionmanager.core.tools.FilesWatcher;

class ExtensionFolderManager
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ExtensionFolderManager.class);
    private static final String CATALOGS_FOLDER = "catalogs";
    private static final String REGISTRY_NAME = "registry.json";
    private static final Predicate<Path> isJar = path -> path.toString().toLowerCase().endsWith(".jar");
    private static final Gson gson = new Gson();
    private final ReadOnlyObjectProperty<Path> extensionDirectoryPath;
    private final FilesWatcher manuallyInstalledExtensionsWatcher;
    private final FilesWatcher catalogManagedInstalledExtensionsWatcher;

    public ExtensionFolderManager(ReadOnlyObjectProperty<Path> extensionDirectoryPath) {
        this.extensionDirectoryPath = extensionDirectoryPath;
        this.manuallyInstalledExtensionsWatcher = new FilesWatcher((ObservableValue<Path>)extensionDirectoryPath, isJar, path -> {
            try {
                return path.equals(((Path)extensionDirectoryPath.get()).resolve(CATALOGS_FOLDER));
            }
            catch (NullPointerException | InvalidPathException e) {
                logger.debug("Error when trying to assess if {} should be watched", path, (Object)e);
                return true;
            }
        });
        this.catalogManagedInstalledExtensionsWatcher = new FilesWatcher((ObservableValue<Path>)extensionDirectoryPath.map(path -> {
            if (path == null) {
                return null;
            }
            try {
                return this.getAndCreateCatalogsDirectory();
            }
            catch (IOException | NullPointerException | SecurityException | InvalidPathException e) {
                logger.debug("Error when getting catalog path from {}", path, (Object)e);
                return null;
            }
        }), isJar, path -> false);
    }

    @Override
    public void close() throws Exception {
        this.manuallyInstalledExtensionsWatcher.close();
        this.catalogManagedInstalledExtensionsWatcher.close();
    }

    public ReadOnlyObjectProperty<Path> getExtensionDirectoryPath() {
        return this.extensionDirectoryPath;
    }

    public synchronized void saveRegistry(Registry registry) throws IOException {
        try (FileWriter fileWriter = new FileWriter(this.getRegistryPath().toFile());
             BufferedWriter writer = new BufferedWriter(fileWriter);){
            writer.write(gson.toJson((Object)Objects.requireNonNull(registry)));
            logger.debug("Registry containing {} saved", registry.catalogs().stream().map(SavedCatalog::name).toList());
        }
    }

    public synchronized Registry getSavedRegistry() throws IOException {
        try (FileReader fileReader = new FileReader(this.getRegistryPath().toFile());){
            Registry registry;
            try (JsonReader jsonReader = new JsonReader((Reader)fileReader);){
                registry = Objects.requireNonNull((Registry)gson.fromJson(jsonReader, Registry.class));
            }
            return registry;
        }
    }

    public synchronized Path getCatalogDirectoryPath(SavedCatalog savedCatalog) throws IOException {
        return this.getAndCreateCatalogsDirectory().resolve(FileTools.stripInvalidFilenameCharacters(savedCatalog.name()));
    }

    public synchronized void deleteExtensionsFromCatalog(SavedCatalog savedCatalog) throws IOException {
        File catalogDirectory = this.getCatalogDirectoryPath(savedCatalog).toFile();
        FileTools.moveDirectoryToTrashOrDeleteRecursively(catalogDirectory);
        logger.debug("The extension files of {} located in {} have been deleted", (Object)savedCatalog.name(), (Object)catalogDirectory);
    }

    public synchronized Path getExtensionDirectoryPath(SavedCatalog savedCatalog, Extension extension) throws IOException {
        return this.getCatalogDirectoryPath(savedCatalog).resolve(FileTools.stripInvalidFilenameCharacters(extension.name()));
    }

    public synchronized Optional<InstalledExtension> getInstalledExtension(SavedCatalog savedCatalog, Extension extension) throws IOException {
        Path extensionPath = this.getExtensionDirectoryPath(savedCatalog, extension);
        Path versionPath = null;
        if (Files.isDirectory(extensionPath, new LinkOption[0])) {
            try (Stream<Path> stream = Files.list(extensionPath);){
                versionPath = stream.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).findAny().orElse(null);
            }
        }
        if (versionPath == null) {
            logger.debug("No folder found in {}. Guessing {} is not installed", (Object)extensionPath, (Object)extension.name());
            return Optional.empty();
        }
        Path mainJarFolderPath = Paths.get(versionPath.toString(), FileType.MAIN_JAR.name);
        if (!FileTools.isDirectoryNotEmpty(mainJarFolderPath)) {
            logger.debug("The folder at {} is not a non-empty directory. Guessing {} is not installed", (Object)mainJarFolderPath, (Object)extension.name());
            return Optional.empty();
        }
        logger.debug("{} detected at {}", (Object)extension.name(), (Object)mainJarFolderPath);
        Path optionalDependenciesFolderPath = Paths.get(versionPath.toString(), FileType.OPTIONAL_DEPENDENCIES.name);
        boolean optionalDependenciesInstalled = FileTools.isDirectoryNotEmpty(optionalDependenciesFolderPath);
        if (optionalDependenciesInstalled) {
            logger.debug("Optional dependencies of {} detected because {} is a non-empty directory", (Object)extension.name(), (Object)optionalDependenciesFolderPath);
        } else {
            logger.debug("Optional dependencies of {} not detected because {} is not a non-empty directory", (Object)extension.name(), (Object)optionalDependenciesFolderPath);
        }
        return Optional.of(new InstalledExtension(versionPath.toFile().getName(), optionalDependenciesInstalled));
    }

    public synchronized void deleteExtension(SavedCatalog savedCatalog, Extension extension) throws IOException {
        FileTools.moveDirectoryToTrashOrDeleteRecursively(this.getExtensionDirectoryPath(savedCatalog, extension).toFile());
        logger.debug("The extension files of {} belonging to {} have been deleted", (Object)extension.name(), (Object)savedCatalog.name());
    }

    public synchronized Path getExtensionPath(SavedCatalog savedCatalog, Extension extension, String releaseName, FileType fileType, boolean createDirectory) throws IOException {
        Path folderPath = Paths.get(this.getExtensionDirectoryPath(savedCatalog, extension).toString(), releaseName, fileType.name);
        if (createDirectory) {
            if (Files.isRegularFile(folderPath, new LinkOption[0])) {
                logger.debug("Deleting {} because it should be a directory", (Object)folderPath);
                Files.deleteIfExists(folderPath);
            }
            Files.createDirectories(folderPath, new FileAttribute[0]);
        }
        return folderPath;
    }

    public ObservableList<Path> getManuallyInstalledJars() {
        return this.manuallyInstalledExtensionsWatcher.getFiles();
    }

    public ObservableList<Path> getCatalogManagedInstalledJars() {
        return this.catalogManagedInstalledExtensionsWatcher.getFiles();
    }

    private Path getRegistryPath() throws IOException {
        return this.getAndCreateCatalogsDirectory().resolve(REGISTRY_NAME);
    }

    private Path getAndCreateCatalogsDirectory() throws IOException {
        Path catalogsFolder = ((Path)this.extensionDirectoryPath.get()).resolve(CATALOGS_FOLDER);
        if (Files.isRegularFile(catalogsFolder, new LinkOption[0])) {
            logger.debug("Deleting {} because it should be a directory", (Object)catalogsFolder);
            Files.deleteIfExists(catalogsFolder);
        }
        Files.createDirectories(catalogsFolder, new FileAttribute[0]);
        return catalogsFolder;
    }

    public static enum FileType {
        MAIN_JAR("main-jar"),
        JAVADOCS("javadocs-dependencies"),
        REQUIRED_DEPENDENCIES("required-dependencies"),
        OPTIONAL_DEPENDENCIES("optional-dependencies");

        private final String name;

        private FileType(String name) {
            this.name = name;
        }
    }
}

