/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.servers.openslide;

import com.google.gson.GsonBuilder;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.io.IOException;
import java.lang.ref.Cleaner;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.GeneralTools;
import qupath.lib.images.servers.AbstractTileableImageServer;
import qupath.lib.images.servers.ImageChannel;
import qupath.lib.images.servers.ImageServerBuilder;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.PixelType;
import qupath.lib.images.servers.TileRequest;
import qupath.lib.images.servers.openslide.OpenslideServerBuilder;
import qupath.lib.images.servers.openslide.jna.OpenSlide;
import qupath.lib.images.servers.openslide.jna.OpenSlideLoader;

public class OpenslideImageServer
extends AbstractTileableImageServer {
    private static final Logger logger = LoggerFactory.getLogger(OpenslideImageServer.class);
    private static final Cleaner cleaner = Cleaner.create();
    private final OpenSlideState state;
    private final Cleaner.Cleanable cleanable;
    private static boolean useBoundingBoxes = true;
    private ImageServerMetadata originalMetadata;
    private List<String> associatedImageList = null;
    private final OpenSlide osr;
    private Color backgroundColor;
    private int boundsX;
    private int boundsY;
    private int boundsWidth;
    private int boundsHeight;
    private URI uri;
    private String[] args;

    private static double readNumericPropertyOrDefault(Map<String, String> properties, String name, double defaultValue) {
        String value = properties.get(name);
        if (value == null) {
            logger.warn("Openslide: Property '{}' not available, will return default value {}", (Object)name, (Object)defaultValue);
            return defaultValue;
        }
        try {
            return Double.parseDouble(value);
        }
        catch (NumberFormatException e) {
            logger.error("Openslide: Could not parse property '{}' with value {} as a number, will return default value {}", new Object[]{name, value, defaultValue});
            return defaultValue;
        }
    }

    public OpenslideImageServer(URI uri, String ... args) throws IOException {
        String name;
        this.uri = uri;
        System.gc();
        Path filePath = GeneralTools.toPath((URI)uri);
        if (filePath != null && Files.exists(filePath, new LinkOption[0])) {
            this.osr = OpenSlideLoader.openImage(filePath.toRealPath(new LinkOption[0]).toString());
            name = filePath.getFileName().toString();
        } else {
            this.osr = OpenSlideLoader.openImage(uri.toString());
            name = null;
        }
        this.state = new OpenSlideState(this.osr);
        this.cleanable = cleaner.register((Object)this, this.state);
        int width = (int)this.osr.getLevel0Width();
        int height = (int)this.osr.getLevel0Height();
        Map<String, String> properties = this.osr.getProperties();
        boolean applyBounds = useBoundingBoxes;
        for (String arg : args) {
            if (!"--no-crop".equals(arg)) continue;
            applyBounds = false;
        }
        boolean isCropped = false;
        if (applyBounds && properties.keySet().containsAll(Arrays.asList("openslide.bounds-x", "openslide.bounds-y", "openslide.bounds-width", "openslide.bounds-height"))) {
            try {
                this.boundsX = Integer.parseInt(properties.get("openslide.bounds-x"));
                this.boundsY = Integer.parseInt(properties.get("openslide.bounds-y"));
                this.boundsWidth = Integer.parseInt(properties.get("openslide.bounds-width"));
                this.boundsHeight = Integer.parseInt(properties.get("openslide.bounds-height"));
                isCropped = this.boundsWidth != width && this.boundsHeight != height;
            }
            catch (Exception e) {
                this.boundsX = 0;
                this.boundsY = 0;
                this.boundsWidth = width;
                this.boundsHeight = height;
            }
        } else {
            this.boundsWidth = width;
            this.boundsHeight = height;
        }
        int tileWidth = (int)OpenslideImageServer.readNumericPropertyOrDefault(properties, "openslide.level[0].tile-width", 256.0);
        int tileHeight = (int)OpenslideImageServer.readNumericPropertyOrDefault(properties, "openslide.level[0].tile-height", 256.0);
        double pixelWidth = OpenslideImageServer.readNumericPropertyOrDefault(properties, "openslide.mpp-x", Double.NaN);
        double pixelHeight = OpenslideImageServer.readNumericPropertyOrDefault(properties, "openslide.mpp-y", Double.NaN);
        double magnification = OpenslideImageServer.readNumericPropertyOrDefault(properties, "openslide.objective-power", Double.NaN);
        if (pixelWidth <= 0.0 || pixelHeight <= 0.0 || Double.isInfinite(pixelWidth) || Double.isInfinite(pixelHeight)) {
            logger.warn("Invalid pixel sizes {} and {}, will use default", (Object)pixelWidth, (Object)pixelHeight);
            pixelWidth = Double.NaN;
            pixelHeight = Double.NaN;
        }
        int levelCount = this.osr.getLevelCount();
        ImageServerMetadata.ImageResolutionLevel.Builder resolutionBuilder = new ImageServerMetadata.ImageResolutionLevel.Builder(width, height);
        for (int i = 0; i < levelCount; ++i) {
            int w = (int)this.osr.getLevelWidth(i);
            int h = (int)this.osr.getLevelHeight(i);
            resolutionBuilder.addLevel(w, h);
        }
        List levels = resolutionBuilder.build();
        String path = uri.toString();
        if (isCropped) {
            ImageServerMetadata.ImageResolutionLevel.Builder resolutionBuilderCropped = new ImageServerMetadata.ImageResolutionLevel.Builder(this.boundsWidth, this.boundsHeight);
            for (ImageServerMetadata.ImageResolutionLevel level : levels) {
                resolutionBuilderCropped.addLevelByDownsample(level.getDownsample());
            }
            levels = resolutionBuilderCropped.build();
            path = String.format("%s [x=%d,y=%d,w=%d,h=%d]", path, this.boundsX, this.boundsY, this.boundsWidth, this.boundsHeight);
        }
        this.args = args;
        this.originalMetadata = new ImageServerMetadata.Builder(((Object)((Object)this)).getClass(), path, this.boundsWidth, this.boundsHeight).channels((Collection)ImageChannel.getDefaultRGBChannels()).name(name).rgb(true).pixelType(PixelType.UINT8).preferredTileSize(tileWidth, tileHeight).pixelSizeMicrons((Number)pixelWidth, (Number)pixelHeight).magnification(magnification).levels((Collection)levels).build();
        this.associatedImageList = Collections.unmodifiableList(this.osr.getAssociatedImages());
        try {
            Object bg = properties.get("openslide.background-color");
            if (bg != null) {
                if (!((String)bg).startsWith("#")) {
                    bg = "#" + (String)bg;
                }
                this.backgroundColor = Color.decode((String)bg);
            }
        }
        catch (Exception e) {
            this.backgroundColor = null;
            logger.debug("Unable to find background color: {}", (Object)e.getLocalizedMessage());
        }
        try {
            logger.debug("Test reading thumbnail with openslide: passed (" + ((BufferedImage)this.getDefaultThumbnail(0, 0)).toString() + ")");
        }
        catch (IOException e) {
            logger.error("Unable to read thumbnail using OpenSlide: {}", (Object)e.getLocalizedMessage());
            throw e;
        }
    }

    public Collection<URI> getURIs() {
        return Collections.singletonList(this.uri);
    }

    protected String createID() {
        return ((Object)((Object)this)).getClass().getName() + ": " + this.uri.toString();
    }

    public void close() {
        this.cleanable.clean();
    }

    public String dumpMetadata() {
        return new GsonBuilder().setPrettyPrinting().create().toJson(this.osr.getProperties());
    }

    public String getServerType() {
        return "OpenSlide";
    }

    public BufferedImage readTile(TileRequest tileRequest) throws IOException {
        int tileX = tileRequest.getImageX() + this.boundsX;
        int tileY = tileRequest.getImageY() + this.boundsY;
        int tileWidth = tileRequest.getTileWidth();
        int tileHeight = tileRequest.getTileHeight();
        BufferedImage img = new BufferedImage(tileWidth, tileHeight, 3);
        int[] data = ((DataBufferInt)img.getRaster().getDataBuffer()).getData();
        this.osr.paintRegionARGB(data, tileX, tileY, tileRequest.getLevel(), tileWidth, tileHeight);
        BufferedImage img2 = new BufferedImage(tileWidth, tileHeight, 1);
        Graphics2D g2d = img2.createGraphics();
        if (this.backgroundColor != null) {
            g2d.setColor(this.backgroundColor);
            g2d.fillRect(0, 0, tileWidth, tileHeight);
        }
        g2d.drawImage(img, 0, 0, tileWidth, tileHeight, null);
        g2d.dispose();
        return img2;
    }

    public List<String> getAssociatedImageList() {
        if (this.associatedImageList == null) {
            return Collections.emptyList();
        }
        return this.associatedImageList;
    }

    protected ImageServerBuilder.ServerBuilder<BufferedImage> createServerBuilder() {
        return ImageServerBuilder.DefaultImageServerBuilder.createInstance(OpenslideServerBuilder.class, (ImageServerMetadata)this.getMetadata(), (URI)this.uri, (String[])this.args);
    }

    public BufferedImage getAssociatedImage(String name) {
        try {
            return this.osr.getAssociatedImage(name);
        }
        catch (Exception e) {
            logger.error("Error requesting associated image " + name, (Throwable)e);
            throw new IllegalArgumentException("Unable to find sub-image with the name " + name);
        }
    }

    public ImageServerMetadata getOriginalMetadata() {
        return this.originalMetadata;
    }

    private static final class OpenSlideState
    implements Runnable {
        private final OpenSlide osr;

        private OpenSlideState(OpenSlide osr) {
            this.osr = osr;
        }

        @Override
        public void run() {
            logger.debug("Closing OpenSlide instance");
            this.osr.close();
        }
    }
}

