/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.roi;

import java.awt.Shape;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.geom.Point2;
import qupath.lib.regions.ImagePlane;
import qupath.lib.roi.PolygonROI;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;

public class ShapeSimplifier {
    private static final Logger logger = LoggerFactory.getLogger(ShapeSimplifier.class);

    public static void simplifyPolygonPoints(List<Point2> points, double altitudeThreshold) {
        PointWithArea pwa;
        double altitude;
        if (points.size() <= 1) {
            return;
        }
        ShapeSimplifier.removeDuplicates(points);
        if (points.size() <= 3) {
            return;
        }
        int n = points.size();
        MinimalPriorityQueue queue = new MinimalPriorityQueue(points.size() + 8);
        Point2 pPrevious = points.getLast();
        Point2 pCurrent = points.getFirst();
        PointWithArea pwaPrevious = null;
        PointWithArea pwaFirst = null;
        for (int i = 0; i < n; ++i) {
            Point2 pNext = points.get((i + 1) % n);
            PointWithArea pwa2 = new PointWithArea(pCurrent, ShapeSimplifier.calculateArea(pPrevious, pCurrent, pNext));
            pwa2.setPrevious(pwaPrevious);
            if (pwaPrevious != null) {
                pwaPrevious.setNext(pwa2);
            }
            queue.add(pwa2);
            pwaPrevious = pwa2;
            pPrevious = pCurrent;
            pCurrent = pNext;
            if (i == n - 1) {
                pwa2.setNext(pwaFirst);
                pwaFirst.setPrevious(pwa2);
                continue;
            }
            if (i != 0) continue;
            pwaFirst = pwa2;
        }
        double maxArea = 0.0;
        int minSize = Math.max(n / 100, 3);
        HashSet toRemove = HashSet.newHashSet(queue.size() / 4);
        while (queue.size() > minSize && !((altitude = (pwa = queue.poll()).getArea() * 2.0 / pwa.getNext().getPoint().distance(pwa.getPrevious().getPoint())) > altitudeThreshold)) {
            if (pwa.getArea() < maxArea) {
                pwa.setArea(maxArea);
            } else {
                maxArea = pwa.getArea();
            }
            toRemove.add(pwa.getPoint());
            pwaPrevious = pwa.getPrevious();
            PointWithArea pwaNext = pwa.getNext();
            queue.remove(pwaPrevious);
            queue.remove(pwaNext);
            pwaPrevious.setNext(pwaNext);
            pwaPrevious.updateArea();
            pwaNext.setPrevious(pwaPrevious);
            pwaNext.updateArea();
            queue.add(pwaPrevious);
            queue.add(pwaNext);
        }
        points.removeAll(toRemove);
    }

    public static ROI simplifyShape(ROI shapeROI, double altitudeThreshold) {
        Shape shape = RoiTools.getShape(shapeROI);
        Path2D path = shape instanceof Path2D ? (Path2D)shape : new Path2D.Float(shape);
        path = ShapeSimplifier.simplifyPath(path, altitudeThreshold);
        return RoiTools.getShapeROI(path, shapeROI.getImagePlane(), 0.5);
    }

    public static Path2D simplifyPath(Path2D path, double altitudeThreshold) {
        return ShapeSimplifier.simplifyPath(path, altitudeThreshold, -1, -1.0);
    }

    public static Path2D simplifyPath(Path2D path, double altitudeThreshold, int segmentPointThreshold, double discardBoundsLength) {
        ArrayList<Point2> points = new ArrayList<Point2>();
        PathIterator iter = path.getPathIterator(null, 0.5);
        Path2D.Float pathNew = new Path2D.Float();
        while (!iter.isDone()) {
            points.clear();
            ShapeSimplifier.getNextClosedSegment(iter, points);
            if (points.isEmpty()) break;
            if (discardBoundsLength > 0.0 && Double.isFinite(discardBoundsLength)) {
                double minX = Double.POSITIVE_INFINITY;
                double minY = Double.POSITIVE_INFINITY;
                double maxX = Double.NEGATIVE_INFINITY;
                double maxY = Double.NEGATIVE_INFINITY;
                for (Point2 p : points) {
                    double x = p.getX();
                    double y = p.getY();
                    minX = Math.min(minX, x);
                    minY = Math.min(minY, y);
                    maxX = Math.max(maxX, x);
                    maxY = Math.max(maxY, y);
                }
                if (maxX - minX < discardBoundsLength && maxY - minY < discardBoundsLength) {
                    logger.trace("Discarding small segment based on bounding box: {} x {} ({} points)", new Object[]{maxX - minX, maxY - minY, points.size()});
                    continue;
                }
            }
            boolean doSimplify = true;
            if (segmentPointThreshold >= 0) {
                ShapeSimplifier.removeDuplicates(points);
                boolean bl = doSimplify = points.size() >= segmentPointThreshold;
            }
            if (doSimplify) {
                ShapeSimplifier.simplifyPolygonPoints(points, altitudeThreshold);
            }
            boolean firstPoint = true;
            for (Point2 p : points) {
                double xx = p.getX();
                double yy = p.getY();
                if (firstPoint) {
                    ((Path2D)pathNew).moveTo(xx, yy);
                    firstPoint = false;
                    continue;
                }
                ((Path2D)pathNew).lineTo(xx, yy);
            }
            pathNew.closePath();
        }
        return pathNew;
    }

    public static List<Point2> smoothPoints(List<Point2> points) {
        ArrayList<Point2> points2 = new ArrayList<Point2>(points.size());
        for (int i = 0; i < points.size(); ++i) {
            Point2 p1 = points.get((i + points.size() - 1) % points.size());
            Point2 p2 = points.get(i);
            Point2 p3 = points.get((i + 1) % points.size());
            points2.add(new Point2((p1.getX() + p2.getX() + p3.getX()) / 3.0, (p1.getY() + p2.getY() + p3.getY()) / 3.0));
        }
        return points2;
    }

    public static PolygonROI simplifyPolygon(PolygonROI polygon, double altitudeThreshold) {
        List<Point2> points = polygon.getAllPoints();
        ShapeSimplifier.simplifyPolygonPoints(points, altitudeThreshold);
        return ROIs.createPolygonROI(points, ImagePlane.getPlaneWithChannel(polygon));
    }

    public static ROI simplifyROI(ROI roi, double altitudeThreshold) {
        ROI out;
        if (roi instanceof PolygonROI) {
            PolygonROI polygonROI = (PolygonROI)roi;
            out = ShapeSimplifier.simplifyPolygon(polygonROI, altitudeThreshold);
        } else {
            out = ShapeSimplifier.simplifyShape(roi, altitudeThreshold);
        }
        return out;
    }

    private static void removeDuplicates(List<Point2> points) {
        Iterator<Point2> iter = points.iterator();
        Point2 lastPoint = iter.next();
        while (iter.hasNext()) {
            Point2 nextPoint = iter.next();
            if (nextPoint.equals(lastPoint)) {
                iter.remove();
                continue;
            }
            lastPoint = nextPoint;
        }
        if (lastPoint.equals(points.getFirst())) {
            iter.remove();
        }
    }

    static double calculateArea(Point2 p1, Point2 p2, Point2 p3) {
        return Math.abs(0.5 * (p1.getX() * (p2.getY() - p3.getY()) + p2.getX() * (p3.getY() - p1.getY()) + p3.getX() * (p1.getY() - p2.getY())));
    }

    private static void getNextClosedSegment(PathIterator iter, List<Point2> points) {
        double[] seg = new double[6];
        while (!iter.isDone()) {
            switch (iter.currentSegment(seg)) {
                case 0: 
                case 1: {
                    points.add(new Point2(seg[0], seg[1]));
                    break;
                }
                case 4: {
                    iter.next();
                    return;
                }
                default: {
                    throw new RuntimeException("Invalid path iterator " + String.valueOf(iter) + " - only line connections are allowed");
                }
            }
            iter.next();
        }
    }

    private static class MinimalPriorityQueue {
        private final Comparator<PointWithArea> comparator = Comparator.reverseOrder();
        private final List<PointWithArea> list;
        private boolean initializing = true;

        private MinimalPriorityQueue(int capacity) {
            this.list = new ArrayList<PointWithArea>(capacity);
        }

        void add(PointWithArea pwa) {
            if (this.initializing) {
                this.list.add(pwa);
            } else {
                this.insert(pwa);
            }
        }

        void remove(PointWithArea pwa) {
            int ind = Collections.binarySearch(this.list, pwa, this.comparator);
            if (ind < 0) {
                throw new IllegalArgumentException("PointWithArea is not in the queue");
            }
            this.list.remove(ind);
        }

        void insert(PointWithArea pwa) {
            int ind = Collections.binarySearch(this.list, pwa, this.comparator);
            if (ind < 0) {
                ind = -ind - 1;
            } else if (this.list.get(ind) == pwa) {
                throw new IllegalArgumentException("PointWithArea is already in the queue");
            }
            this.list.add(ind, pwa);
        }

        int size() {
            return this.list.size();
        }

        PointWithArea poll() {
            if (this.initializing) {
                this.list.sort(this.comparator);
                this.initializing = false;
            }
            return this.list.isEmpty() ? null : this.list.removeLast();
        }
    }

    static class PointWithArea
    implements Comparable<PointWithArea> {
        private static final Comparator<PointWithArea> comparator = Comparator.comparingDouble(PointWithArea::getArea).reversed().thenComparingDouble(PointWithArea::getX).thenComparingDouble(PointWithArea::getY);
        private PointWithArea pPrevious;
        private PointWithArea pNext;
        private Point2 p;
        private double area;

        PointWithArea(Point2 p, double area) {
            this.p = p;
            this.area = area;
        }

        public void setArea(double area) {
            this.area = area;
        }

        public void updateArea() {
            this.area = ShapeSimplifier.calculateArea(this.pPrevious.getPoint(), this.p, this.pNext.getPoint());
        }

        public double getX() {
            return this.p.getX();
        }

        public double getY() {
            return this.p.getY();
        }

        public double getArea() {
            return this.area;
        }

        public Point2 getPoint() {
            return this.p;
        }

        public void setPrevious(PointWithArea pPrevious) {
            this.pPrevious = pPrevious;
        }

        public void setNext(PointWithArea pNext) {
            this.pNext = pNext;
        }

        public PointWithArea getPrevious() {
            return this.pPrevious;
        }

        public PointWithArea getNext() {
            return this.pNext;
        }

        @Override
        public int compareTo(PointWithArea p) {
            if (this.area < p.area) {
                return -1;
            }
            if (this.area > p.area) {
                return 1;
            }
            if (this.getX() < p.getX()) {
                return -1;
            }
            if (this.getX() > p.getX()) {
                return 1;
            }
            if (this.getY() < p.getY()) {
                return -1;
            }
            if (this.getY() > p.getY()) {
                return 1;
            }
            return 0;
        }

        public String toString() {
            return this.getX() + ", " + this.getY() + ", Area: " + this.getArea();
        }
    }
}

