/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.roi;

import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.prep.PreparedGeometry;
import org.locationtech.jts.geom.prep.PreparedGeometryFactory;
import org.locationtech.jts.geom.util.AffineTransformation;
import org.locationtech.jts.shape.random.RandomPointsBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.awt.common.AwtTools;
import qupath.lib.common.GeneralTools;
import qupath.lib.geom.ImmutableDimension;
import qupath.lib.geom.Point2;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.ImageRegion;
import qupath.lib.roi.DefaultMutableVertices;
import qupath.lib.roi.DefaultVertices;
import qupath.lib.roi.EllipseROI;
import qupath.lib.roi.GeometryROI;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.LineROI;
import qupath.lib.roi.MutableVertices;
import qupath.lib.roi.PolygonROI;
import qupath.lib.roi.PolylineROI;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.RectangleROI;
import qupath.lib.roi.Vertices;
import qupath.lib.roi.interfaces.ROI;

public class RoiTools {
    private static final Logger logger = LoggerFactory.getLogger(RoiTools.class);

    public static ROI combineROIs(ROI shape1, ROI shape2, CombineOp op) {
        logger.trace("Combining {} and {} with op {}", new Object[]{shape1, shape2, op});
        if (!RoiTools.sameImagePlane(shape1, shape2)) {
            throw new IllegalArgumentException("Cannot combine - shapes " + String.valueOf(shape1) + " and " + String.valueOf(shape2) + " do not share the same image plane");
        }
        Geometry area1 = shape1.getGeometry();
        Geometry area2 = shape2.getGeometry();
        switch (op.ordinal()) {
            case 0: {
                return GeometryTools.geometryToROI(area1.union(area2), shape1.getImagePlane());
            }
            case 2: {
                return GeometryTools.geometryToROI(GeometryTools.homogenizeGeometryCollection(area1.intersection(area2)), shape1.getImagePlane());
            }
            case 1: {
                return GeometryTools.geometryToROI(GeometryTools.homogenizeGeometryCollection(area1.difference(area2)), shape1.getImagePlane());
            }
        }
        throw new IllegalArgumentException("Unknown op " + String.valueOf((Object)op));
    }

    public static ROI union(Collection<? extends ROI> rois) {
        logger.trace("Calculating union of {} ROIs", (Object)rois.size());
        if (rois.isEmpty()) {
            return ROIs.createEmptyROI();
        }
        if (rois.size() == 1) {
            return rois.iterator().next();
        }
        ImagePlane plane = rois.iterator().next().getImagePlane();
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        for (ROI rOI : rois) {
            if (!rOI.getImagePlane().equals(plane)) {
                throw new IllegalArgumentException("Cannot merge ROIs - found plane " + String.valueOf(rOI.getImagePlane()) + " but expected " + String.valueOf(plane));
            }
            geometries.add(rOI.getGeometry());
        }
        return GeometryTools.geometryToROI(GeometryTools.union(geometries), plane);
    }

    public static ROI union(ROI ... rois) {
        return RoiTools.union(Arrays.asList(rois));
    }

    public static ROI intersection(Collection<? extends ROI> rois) {
        if (rois.isEmpty()) {
            return ROIs.createEmptyROI();
        }
        if (rois.size() == 1) {
            return rois.iterator().next();
        }
        ImagePlane plane = rois.iterator().next().getImagePlane();
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        for (ROI rOI : rois) {
            if (!rOI.getImagePlane().equals(plane)) {
                throw new IllegalArgumentException("Cannot merge ROIs - found plane " + String.valueOf(rOI.getImagePlane()) + " but expected " + String.valueOf(plane));
            }
            geometries.add(rOI.getGeometry());
        }
        Geometry first = (Geometry)geometries.removeFirst();
        for (Geometry geom : geometries) {
            first = first.intersection(geom);
        }
        return GeometryTools.geometryToROI(GeometryTools.homogenizeGeometryCollection(first), plane);
    }

    public static ROI intersection(ROI ... rois) {
        return RoiTools.intersection(Arrays.asList(rois));
    }

    public static double intersectionArea(ROI a, ROI b) {
        if (a.getZ() != b.getZ() || a.getT() != b.getT() || !a.isArea() || !b.isArea()) {
            return 0.0;
        }
        return GeometryTools.intersectionArea(a.getGeometry(), b.getGeometry());
    }

    public static double iou(ROI a, ROI b) {
        double intersection = RoiTools.intersectionArea(a, b);
        if (intersection == 0.0) {
            return 0.0;
        }
        return intersection / (a.getArea() + b.getArea() - intersection);
    }

    public static ROI difference(ROI roi1, ROI roi2) {
        ImagePlane plane = roi1.getImagePlane();
        if (!roi2.getImagePlane().equals(plane)) {
            throw new IllegalArgumentException("Cannot compute difference - found plane " + String.valueOf(roi2.getImagePlane()) + " but expected " + String.valueOf(plane));
        }
        Geometry geom = roi1.getGeometry().difference(roi2.getGeometry());
        return GeometryTools.geometryToROI(geom, plane);
    }

    public static ROI symDifference(ROI roi1, ROI roi2) {
        ImagePlane plane = roi1.getImagePlane();
        if (!roi2.getImagePlane().equals(plane)) {
            throw new IllegalArgumentException("Cannot compute symmetric difference - found plane " + String.valueOf(roi2.getImagePlane()) + " but expected " + String.valueOf(plane));
        }
        Geometry geom = roi1.getGeometry().symDifference(roi2.getGeometry());
        return GeometryTools.geometryToROI(geom, plane);
    }

    public static ROI subtract(ROI roiMain, ROI ... roisToSubtract) {
        return RoiTools.subtract(roiMain, Arrays.asList(roisToSubtract));
    }

    public static ROI subtract(ROI roiMain, Collection<? extends ROI> roisToSubtract) {
        if (roisToSubtract.isEmpty()) {
            return roiMain;
        }
        if (roisToSubtract.size() == 1) {
            return RoiTools.difference(roiMain, roisToSubtract.iterator().next());
        }
        ImageRegion region = ImageRegion.createInstance(roiMain);
        List<ROI> roisToSubtract2 = roisToSubtract.stream().filter(r -> region.intersects(r.getBoundsX(), r.getBoundsY(), r.getBoundsWidth(), r.getBoundsHeight())).toList();
        if (!roisToSubtract2.isEmpty()) {
            roiMain = RoiTools.difference(roiMain, RoiTools.union(roisToSubtract2));
        }
        return roiMain;
    }

    public static boolean intersectsRegion(ROI roi, ImageRegion region) {
        if (roi.getZ() != region.getZ() || roi.getT() != region.getT()) {
            return false;
        }
        if (!region.intersects(roi.getBoundsX(), roi.getBoundsY(), roi.getBoundsWidth(), roi.getBoundsHeight())) {
            return false;
        }
        if (roi instanceof RectangleROI) {
            return true;
        }
        return GeometryTools.regionToGeometry(region).intersects(roi.getGeometry());
    }

    public static ROI transformROI(ROI roi, AffineTransform transform) {
        Rectangle2D.Double bounds;
        Shape shape;
        logger.trace("Applying affine transform {} to ROI {}", (Object)transform, (Object)roi);
        if (roi == null || transform == null || transform.isIdentity()) {
            return roi;
        }
        if (roi instanceof EllipseROI && new Area(shape = transform.createTransformedShape(bounds = new Rectangle2D.Double(roi.getBoundsX(), roi.getBoundsY(), roi.getBoundsWidth(), roi.getBoundsHeight()))).isRectangular()) {
            bounds.setRect(shape.getBounds2D());
            return ROIs.createEllipseROI(bounds.x, bounds.y, bounds.width, bounds.height, roi.getImagePlane());
        }
        AffineTransformation t = GeometryTools.convertTransform(transform);
        Geometry geometry2 = t.transform(roi.getGeometry());
        return GeometryTools.geometryToROI(geometry2, roi.getImagePlane());
    }

    public static List<ROI> clipToROI(ROI parent, Collection<? extends ROI> rois) {
        logger.trace("Clipping {} ROIs to {}", (Object)rois.size(), (Object)parent);
        Geometry geom = parent.getGeometry();
        ArrayList<ROI> results = new ArrayList<ROI>();
        for (ROI rOI : rois) {
            if (!RoiTools.sameImagePlane(parent, rOI)) continue;
            Geometry g = rOI.getGeometry();
            if (geom.covers(g)) {
                results.add(rOI);
                continue;
            }
            g = geom.intersection(g);
            if ((g = GeometryTools.homogenizeGeometryCollection(g)).isEmpty()) continue;
            ROI r2 = GeometryTools.geometryToROI(g, rOI.getImagePlane());
            if (rOI.isArea() != r2.isArea() || rOI.isLine() != r2.isLine() || rOI.isPoint() != r2.isPoint()) continue;
            results.add(r2);
        }
        return results;
    }

    public static ROI fillHoles(ROI roi) {
        return RoiTools.removeSmallPieces(roi, 0.0, Double.POSITIVE_INFINITY);
    }

    public static ROI removeSmallPieces(ROI roi, double minAreaPixels, double minHoleAreaPixels) {
        Geometry geometry2;
        logger.trace("Removing small pieces from {} (min = {}, max = {})", new Object[]{roi, minAreaPixels, minHoleAreaPixels});
        if (roi instanceof RectangleROI || roi instanceof EllipseROI || roi instanceof LineROI || roi instanceof PolylineROI) {
            if (roi.getArea() < minAreaPixels) {
                return ROIs.createEmptyROI(roi.getImagePlane());
            }
            return roi;
        }
        Geometry geometry = roi.getGeometry();
        if (geometry == (geometry2 = GeometryTools.refineAreas(geometry, minAreaPixels, minHoleAreaPixels))) {
            return roi;
        }
        if (geometry2 == null) {
            return ROIs.createEmptyROI(roi.getImagePlane());
        }
        return GeometryTools.geometryToROI(geometry2, roi.getImagePlane());
    }

    public static double getCircularity(ROI roi) {
        return RoiTools.getCircularity(roi, 1.0, 1.0);
    }

    public static double getCircularity(ROI roi, double pixelWidth, double pixelHeight) {
        if (roi.isArea()) {
            double perim = roi.getScaledLength(pixelWidth, pixelHeight);
            return Math.PI * 4 * (roi.getScaledArea(pixelWidth, pixelHeight) / (perim * perim));
        }
        return Double.NaN;
    }

    public static ROI getShapeROI(Shape shape, ImagePlane plane, double flatness) {
        if (shape instanceof Rectangle2D) {
            Rectangle2D bounds = shape.getBounds2D();
            return ROIs.createRectangleROI(bounds.getX(), bounds.getY(), bounds.getWidth(), bounds.getHeight(), plane);
        }
        if (shape instanceof Ellipse2D) {
            Rectangle2D bounds = shape.getBounds2D();
            return ROIs.createEllipseROI(bounds.getX(), bounds.getY(), bounds.getWidth(), bounds.getHeight(), plane);
        }
        if (shape instanceof Line2D) {
            Line2D line = (Line2D)shape;
            return ROIs.createLineROI(line.getX1(), line.getY1(), line.getX2(), line.getY2(), plane);
        }
        boolean isClosed = false;
        ArrayList<Point2> points = null;
        if (!(shape instanceof Area)) {
            PathIterator iterator = shape.getPathIterator(null, flatness);
            double[] coords = new double[6];
            points = new ArrayList<Point2>();
            while (!iterator.isDone()) {
                int type = iterator.currentSegment(coords);
                if (type == 4) {
                    isClosed = true;
                    break;
                }
                points.add(new Point2(coords[0], coords[1]));
                iterator.next();
            }
        }
        if (isClosed) {
            Area area = shape instanceof Area ? (Area)shape : new Area(shape);
            return RoiTools.getShapeROI(area, plane, flatness);
        }
        if (points.size() == 2) {
            Point2 p1 = (Point2)points.get(0);
            Point2 p2 = (Point2)points.get(1);
            return ROIs.createLineROI(p1.getX(), p1.getY(), p2.getX(), p2.getY(), plane);
        }
        return ROIs.createPolylineROI((List<? extends Point2>)points, plane);
    }

    public static double getCentroidDistance(ROI roi1, ROI roi2) {
        return RoiTools.getCentroidDistance(roi1, roi2, 1.0, 1.0);
    }

    public static double getCentroidDistance(ROI roi1, ROI roi2, double pixelWidth, double pixelHeight) {
        double dx = (roi1.getCentroidX() - roi2.getCentroidX()) * pixelWidth;
        double dy = (roi1.getCentroidY() - roi2.getCentroidY()) * pixelHeight;
        return Math.sqrt(dx * dx + dy * dy);
    }

    public static double getBoundaryDistance(ROI roi1, ROI roi2) {
        return RoiTools.getBoundaryDistance(roi1, roi2, 1.0, 1.0);
    }

    public static double getBoundaryDistance(ROI roi1, ROI roi2, double pixelWidth, double pixelHeight) {
        if (pixelWidth == pixelHeight) {
            double pixelSize = pixelWidth;
            return roi1.getGeometry().distance(roi2.getGeometry()) * pixelSize;
        }
        AffineTransformation transform = AffineTransformation.scaleInstance((double)pixelWidth, (double)pixelHeight);
        Geometry g1 = transform.transform(roi1.getGeometry());
        Geometry g2 = transform.transform(roi2.getGeometry());
        return g1.distance(g2);
    }

    public static ROI getShapeROI(Area area, ImagePlane plane) {
        return RoiTools.getShapeROI(area, plane, -1.0);
    }

    public static Shape getShape(ROI roi) throws IllegalArgumentException {
        if (roi.isPoint()) {
            throw new IllegalArgumentException(String.valueOf(roi) + " cannot be converted to a shape!");
        }
        return roi.getShape();
    }

    public static Area getArea(ROI roi) {
        Shape shape = RoiTools.getShape(roi);
        if (shape instanceof Area) {
            return (Area)shape;
        }
        return new Area(shape);
    }

    public static List<ROI> makeTiles(ROI roi, int tileWidth, int tileHeight, boolean trimToROI) {
        Rectangle bounds = AwtTools.getBounds(roi);
        Area area = RoiTools.getArea(roi);
        ArrayList<ROI> tiles = new ArrayList<ROI>();
        for (int y = bounds.y; y < bounds.y + bounds.height; y += tileHeight) {
            for (int x = bounds.x; x < bounds.x + bounds.width; x += tileWidth) {
                ROI tile;
                int width = tileWidth;
                int height = tileHeight;
                Rectangle tileBounds = new Rectangle(x, y, width, height);
                if (area.contains(x, y, width, height)) {
                    tile = ROIs.createRectangleROI(x, y, width, height, roi.getImagePlane());
                } else if (!trimToROI) {
                    if (!area.contains((double)x + 0.5 * (double)width, (double)y + 0.5 * (double)height)) continue;
                    tile = ROIs.createRectangleROI(x, y, width, height, roi.getImagePlane());
                } else {
                    if (!area.intersects(x, y, width, height)) continue;
                    Area tileArea = new Area(tileBounds);
                    tileArea.intersect(area);
                    if (tileArea.isEmpty()) continue;
                    if (tileArea.isRectangular()) {
                        Rectangle2D bounds2 = tileArea.getBounds2D();
                        tile = ROIs.createRectangleROI(bounds2.getX(), bounds2.getY(), bounds2.getWidth(), bounds2.getHeight(), roi.getImagePlane());
                    } else {
                        tile = RoiTools.getShapeROI(tileArea, roi.getImagePlane());
                    }
                }
                tiles.add(tile);
            }
        }
        return tiles;
    }

    public static Collection<? extends ROI> computeTiledROIs(ROI parentROI, ImmutableDimension sizePreferred, ImmutableDimension sizeMax, boolean fixedSize, int overlap) {
        ROI pathArea = parentROI != null && parentROI.isArea() ? parentROI : null;
        Rectangle2D bounds = AwtTools.getBounds2D(parentROI);
        if (pathArea == null || bounds.getWidth() <= (double)sizeMax.width && bounds.getHeight() <= (double)sizeMax.height) {
            return Collections.singletonList(parentROI);
        }
        Geometry geometry = pathArea.getGeometry();
        PreparedGeometry prepared = null;
        double xMin = bounds.getMinX();
        double yMin = bounds.getMinY();
        int nx = (int)Math.ceil(bounds.getWidth() / (double)sizePreferred.width);
        int ny = (int)Math.ceil(bounds.getHeight() / (double)sizePreferred.height);
        double w = fixedSize ? (double)sizePreferred.width : (double)((int)Math.ceil(bounds.getWidth() / (double)nx));
        double h = fixedSize ? (double)sizePreferred.height : (double)((int)Math.ceil(bounds.getHeight() / (double)ny));
        xMin = (int)(bounds.getCenterX() - (double)nx * w * 0.5);
        yMin = (int)(bounds.getCenterY() - (double)ny * h * 0.5);
        boolean byRow = false;
        boolean byColumn = false;
        Map<Integer, Geometry> rowParents = null;
        Map<Integer, Geometry> columnParents = null;
        Envelope envelope = geometry.getEnvelopeInternal();
        if (ny > 1 && nx > 1 && geometry.getNumPoints() > 1000) {
            PreparedGeometry prepared2 = prepared = PreparedGeometryFactory.prepare((Geometry)geometry);
            Geometry empty = geometry.getFactory().createEmpty(2);
            byRow = nx > ny;
            byColumn = !byRow;
            double yMin2 = yMin;
            double xMin2 = xMin;
            if (byRow) {
                rowParents = IntStream.range(0, ny).parallel().mapToObj(yi -> yi).collect(Collectors.toMap(yi -> yi, yi -> {
                    double y = yMin2 + (double)yi.intValue() * h - (double)overlap;
                    Polygon row = GeometryTools.createRectangle(envelope.getMinX(), y, envelope.getMaxX() - envelope.getMinX(), h + (double)(overlap * 2));
                    if (!prepared2.intersects((Geometry)row)) {
                        return empty;
                    }
                    if (prepared2.covers((Geometry)row)) {
                        return row;
                    }
                    Geometry temp = RoiTools.intersect(geometry, (Geometry)row);
                    return temp == null ? geometry : temp;
                }));
            }
            if (byColumn) {
                columnParents = IntStream.range(0, nx).parallel().mapToObj(xi -> xi).collect(Collectors.toMap(xi -> xi, xi -> {
                    double x = xMin2 + (double)xi.intValue() * w - (double)overlap;
                    Polygon col = GeometryTools.createRectangle(x, envelope.getMinY(), w + (double)(overlap * 2), envelope.getMaxY() - envelope.getMinY());
                    if (!prepared2.intersects((Geometry)col)) {
                        return empty;
                    }
                    if (prepared2.covers((Geometry)col)) {
                        return col;
                    }
                    Geometry temp = RoiTools.intersect(geometry, (Geometry)col);
                    return temp == null ? geometry : temp;
                }));
            }
        }
        Geometry geometryLocal = geometry;
        LinkedHashMap<Polygon, Object> tileGeometries = new LinkedHashMap<Polygon, Object>();
        for (int yi2 = 0; yi2 < ny; ++yi2) {
            double y = yMin + (double)yi2 * h - (double)overlap;
            if (rowParents != null) {
                geometryLocal = rowParents.getOrDefault(yi2, geometry);
            }
            for (int xi2 = 0; xi2 < nx; ++xi2) {
                double x = xMin + (double)xi2 * w - (double)overlap;
                if (columnParents != null) {
                    geometryLocal = columnParents.getOrDefault(xi2, geometry);
                }
                if (geometryLocal.isEmpty()) continue;
                Polygon rect = GeometryTools.createRectangle(x, y, w + (double)(overlap * 2), h + (double)(overlap * 2));
                if (prepared != null) {
                    if (!prepared.intersects((Geometry)rect)) continue;
                    if (prepared.covers((Geometry)rect)) {
                        tileGeometries.put(rect, rect);
                        continue;
                    }
                }
                tileGeometries.put(rect, geometryLocal);
            }
        }
        ImagePlane plane = parentROI.getImagePlane();
        List<ROI> tileROIs = tileGeometries.entrySet().parallelStream().map(entry -> RoiTools.intersect((Geometry)entry.getKey(), (Geometry)entry.getValue())).filter(g -> g != null).map(g -> GeometryTools.geometryToROI(g, plane)).toList();
        if (tileROIs.size() < tileGeometries.size()) {
            logger.warn("Tiles lost during tiling: {}", (Object)(tileGeometries.size() - tileROIs.size()));
            logger.warn("You may be able to avoid tiling errors by calling 'Simplify shape' on any complex annotations first.");
        }
        return tileROIs.stream().filter(t -> !t.isEmpty() && t.isArea()).toList();
    }

    private static Geometry intersect(Geometry g1, Geometry g2) {
        if (g1 == g2) {
            return g1;
        }
        try {
            return GeometryTools.homogenizeGeometryCollection(g1.intersection(g2));
        }
        catch (Exception e) {
            logger.warn(e.getLocalizedMessage(), (Throwable)e);
            return null;
        }
    }

    public static ROI buffer(ROI roi, double distance) {
        return GeometryTools.geometryToROI(roi.getGeometry().buffer(distance), roi.getImagePlane());
    }

    public static List<ROI> splitROI(ROI roi) {
        if (roi instanceof RectangleROI || roi instanceof LineROI || roi instanceof EllipseROI) {
            return Collections.singletonList(roi);
        }
        ArrayList<ROI> list = new ArrayList<ROI>();
        ImagePlane plane = ImagePlane.getPlane(roi);
        if (roi.isPoint()) {
            if (roi.getNumPoints() <= 1) {
                return Collections.singletonList(roi);
            }
            for (Point2 p : roi.getAllPoints()) {
                list.add(ROIs.createPointsROI(p.getX(), p.getY(), plane));
            }
        } else {
            Geometry geometry = roi.getGeometry();
            if (geometry.getNumGeometries() == 1) {
                return Collections.singletonList(roi);
            }
            for (int i = 0; i < geometry.getNumGeometries(); ++i) {
                list.add(GeometryTools.geometryToROI(geometry.getGeometryN(i), plane));
            }
        }
        return list;
    }

    public static PolygonROI[][] splitAreaToPolygons(Area area, int c, int z, int t) {
        HashMap map = new HashMap();
        map.put(Boolean.TRUE, new ArrayList());
        map.put(Boolean.FALSE, new ArrayList());
        PathIterator iter = area.getPathIterator(null, 0.5);
        ImagePlane plane = ImagePlane.getPlaneWithChannel(c, z, t);
        ArrayList<Point2> points = new ArrayList<Point2>();
        double areaTempSigned = 0.0;
        double areaCached = 0.0;
        double[] seg = new double[6];
        double startX = Double.NaN;
        double startY = Double.NaN;
        double x0 = 0.0;
        double y0 = 0.0;
        double x1 = 0.0;
        double y1 = 0.0;
        boolean closed = false;
        block5: while (!iter.isDone()) {
            switch (iter.currentSegment(seg)) {
                case 0: {
                    startX = seg[0];
                    startY = seg[1];
                    x0 = startX;
                    y0 = startY;
                    iter.next();
                    areaCached += areaTempSigned;
                    areaTempSigned = 0.0;
                    points.clear();
                    points.add(new Point2(startX, startY));
                    closed = false;
                    continue block5;
                }
                case 4: {
                    x1 = startX;
                    y1 = startY;
                    closed = true;
                    break;
                }
                case 1: {
                    x1 = seg[0];
                    y1 = seg[1];
                    points.add(new Point2(x1, y1));
                    closed = false;
                    break;
                }
                default: {
                    throw new RuntimeException("Invalid area computation!");
                }
            }
            areaTempSigned += 0.5 * (x0 * y1 - x1 * y0);
            if (closed) {
                if (areaTempSigned < 0.0) {
                    ((List)map.get(Boolean.FALSE)).add(ROIs.createPolygonROI(points, plane));
                } else if (areaTempSigned > 0.0) {
                    ((List)map.get(Boolean.TRUE)).add(ROIs.createPolygonROI(points, plane));
                }
            }
            x0 = x1;
            y0 = y1;
            iter.next();
        }
        areaCached += areaTempSigned;
        PolygonROI[][] polyOutput = new PolygonROI[2][];
        if (areaCached < 0.0) {
            polyOutput[0] = (PolygonROI[])((List)map.get(Boolean.TRUE)).toArray(PolygonROI[]::new);
            polyOutput[1] = (PolygonROI[])((List)map.get(Boolean.FALSE)).toArray(PolygonROI[]::new);
        } else {
            polyOutput[0] = (PolygonROI[])((List)map.get(Boolean.FALSE)).toArray(PolygonROI[]::new);
            polyOutput[1] = (PolygonROI[])((List)map.get(Boolean.TRUE)).toArray(PolygonROI[]::new);
        }
        return polyOutput;
    }

    public static ROI createRandomRectangle(ImageRegion mask, double width, double height) throws IllegalArgumentException {
        return RoiTools.createRandomRectangle(mask, width, height, null);
    }

    public static ROI createRandomRectangle(ImageRegion mask, double width, double height, Random random) throws IllegalArgumentException {
        Objects.requireNonNull(mask, "Cannot create random rectangle - region mask must not be null");
        if ((double)mask.getWidth() < width || (double)mask.getHeight() < height) {
            throw new IllegalArgumentException("Cannot create random rectangle - region mask " + String.valueOf(mask) + " is too small to create a " + GeneralTools.formatNumber(width, 2) + " x " + GeneralTools.formatNumber(height, 2) + " region");
        }
        if (random == null) {
            random = new Random();
        }
        double x = width == (double)mask.getWidth() ? 0.0 : (double)mask.getMinX() + random.nextDouble() * ((double)mask.getWidth() - width);
        double y = height == (double)mask.getHeight() ? 0.0 : (double)mask.getMinY() + random.nextDouble() * ((double)mask.getHeight() - height);
        return ROIs.createRectangleROI(x, y, width, height, mask.getImagePlane());
    }

    public static ROI createRandomRectangle(ROI mask, double width, double height) throws IllegalArgumentException {
        return RoiTools.createRandomRectangle(mask, width, height, 1000, true, null);
    }

    public static ROI createRandomRectangle(ROI mask, double width, double height, int maxAttempts, boolean permitErosion, Random random) throws IllegalArgumentException {
        Objects.requireNonNull(mask, "Cannot create random rectangle - region mask must not be null");
        if (mask.getBoundsWidth() < width || mask.getBoundsHeight() < height || mask.getArea() < width * height) {
            throw new IllegalArgumentException("Cannot create random rectangle - region mask " + String.valueOf(mask) + " is too small to create a " + GeneralTools.formatNumber(width, 2) + " x " + GeneralTools.formatNumber(height, 2) + " region");
        }
        if (random == null) {
            random = new Random();
        }
        Geometry geometry = mask.getGeometry();
        PreparedGeometry prepared = PreparedGeometryFactory.prepare((Geometry)geometry);
        boolean success = false;
        double x = 0.0;
        double y = 0.0;
        for (int i = 0; i < maxAttempts; ++i) {
            x = width == mask.getBoundsWidth() ? mask.getBoundsX() : mask.getBoundsX() + random.nextDouble() * (mask.getBoundsWidth() - width);
            Polygon rect = GeometryTools.createRectangle(x, y = height == mask.getBoundsHeight() ? mask.getBoundsY() : mask.getBoundsY() + random.nextDouble() * (mask.getBoundsHeight() - height), width, height);
            if (!prepared.covers((Geometry)rect)) continue;
            success = true;
            break;
        }
        if (!success && permitErosion) {
            try {
                double erode = Math.sqrt(2.0) * Math.max(width / 2.0, height / 2.0);
                Geometry geometrySmaller = geometry.buffer(-erode);
                if (!geometrySmaller.isEmpty()) {
                    RandomPointsBuilder builder = new RandomPointsBuilder(geometrySmaller.getFactory());
                    builder.setExtent(geometrySmaller);
                    builder.setNumPoints(1);
                    Geometry points = builder.getGeometry();
                    Coordinate c = points.getCoordinate();
                    x = c.getX() - width / 2.0;
                    y = c.getY() - height / 2.0;
                    Polygon rect = GeometryTools.createRectangle(x, y, width, height);
                    if (prepared.covers((Geometry)rect)) {
                        if (GeneralTools.almostTheSame(width, height, 0.001)) {
                            logger.debug("Creating square region with RandomPointsBuilder");
                        } else {
                            logger.warn("Creating non-square region with RandomPointsBuilder - this will be constrained to the center of the ROI based on the largest side length (i.e. requested width or height)");
                        }
                        success = true;
                    } else {
                        logger.warn("Can't created random region - the one created with RandomPointsBuilder was not covered by the original ROI! This is unexpected...");
                    }
                }
            }
            catch (Exception e) {
                logger.warn(e.getLocalizedMessage(), (Throwable)e);
            }
        }
        if (!success) {
            logger.warn("Unable to find a large enough random region within the selected object after {} attempts, sorry", (Object)maxAttempts);
            return null;
        }
        return ROIs.createRectangleROI(x, y, width, height, mask.getImagePlane());
    }

    public static boolean areaContains(ROI pathROI, double x, double y) {
        return pathROI.isArea() && pathROI.contains(x, y);
    }

    public static boolean isShapeROI(ROI roi) {
        return roi != null && !roi.isPoint();
    }

    static ROI getShapeROI(Area area, ImagePlane plane, double flatness) {
        if (area.isRectangular()) {
            Rectangle2D bounds = area.getBounds2D();
            return ROIs.createRectangleROI(bounds.getX(), bounds.getY(), bounds.getWidth(), bounds.getHeight(), plane);
        }
        if (area.isSingular() && (area.isPolygonal() || flatness > 0.0)) {
            List<Point2> points;
            Path2D.Float path = new Path2D.Float(area);
            List<Point2> list = points = flatness > 0.0 ? RoiTools.getLinearPathPoints(path, path.getPathIterator(null, flatness)) : RoiTools.getLinearPathPoints(path, path.getPathIterator(null));
            if (points.size() > 2) {
                Point2 pStart = points.getFirst();
                Point2 pEnd = points.getLast();
                if (pEnd.equals(pStart)) {
                    points.removeLast();
                }
            }
            return ROIs.createPolygonROI(points, plane);
        }
        return new GeometryROI(GeometryTools.shapeToGeometry(area), plane);
    }

    static Collection<? extends ROI> computeTiledROIsLegacy(ROI parentROI, ImmutableDimension sizePreferred, ImmutableDimension sizeMax, boolean fixedSize, int overlap) {
        ROI pathArea = parentROI != null && parentROI.isArea() ? parentROI : null;
        Rectangle2D bounds = AwtTools.getBounds2D(parentROI);
        if (pathArea == null || bounds.getWidth() <= (double)sizeMax.width && bounds.getHeight() <= (double)sizeMax.height) {
            return Collections.singletonList(parentROI);
        }
        ArrayList<ROI> pathROIs = new ArrayList<ROI>();
        Geometry area = pathArea.getGeometry();
        double xMin = bounds.getMinX();
        double yMin = bounds.getMinY();
        int nx = (int)Math.ceil(bounds.getWidth() / (double)sizePreferred.width);
        int ny = (int)Math.ceil(bounds.getHeight() / (double)sizePreferred.height);
        double w = fixedSize ? (double)sizePreferred.width : (double)((int)Math.ceil(bounds.getWidth() / (double)nx));
        double h = fixedSize ? (double)sizePreferred.height : (double)((int)Math.ceil(bounds.getHeight() / (double)ny));
        xMin = (int)(bounds.getCenterX() - (double)nx * w * 0.5);
        yMin = (int)(bounds.getCenterY() - (double)ny * h * 0.5);
        ImagePlane plane = parentROI.getImagePlane();
        int skipCount = 0;
        for (int yi = 0; yi < ny; ++yi) {
            for (int xi = 0; xi < nx; ++xi) {
                double x = xMin + (double)xi * w - (double)overlap;
                double y = yMin + (double)yi * h - (double)overlap;
                ROI rect = ROIs.createRectangleROI(x, y, w + (double)(overlap * 2), h + (double)(overlap * 2), plane);
                Geometry boundsTile = rect.getGeometry();
                ROI pathROI = null;
                if (area.contains(boundsTile)) {
                    pathROI = rect;
                } else {
                    if (!area.intersects(boundsTile)) continue;
                    try {
                        Geometry areaTemp = GeometryTools.homogenizeGeometryCollection(boundsTile.intersection(area));
                        if (!areaTemp.isEmpty()) {
                            pathROI = GeometryTools.geometryToROI(areaTemp, plane);
                        }
                    }
                    catch (Exception e) {
                        logger.warn("Tile skipped because of error computing intersection: " + e.getLocalizedMessage(), (Throwable)e);
                        ++skipCount;
                    }
                }
                if (pathROI == null) continue;
                pathROIs.add(pathROI);
            }
        }
        if (skipCount > 0) {
            logger.warn("You may be able to avoid tiling errors by calling 'Simplify shape' on any complex annotations first.");
        }
        return pathROIs;
    }

    static PolygonROI[][] splitAreaToPolygons(ROI pathROI) {
        return RoiTools.splitAreaToPolygons(RoiTools.getArea(pathROI), pathROI.getC(), pathROI.getZ(), pathROI.getT());
    }

    static boolean sameImagePlane(ROI roi1, ROI roi2) {
        return roi1.getC() == roi2.getC() && roi1.getT() == roi2.getT() && roi1.getZ() == roi2.getZ();
    }

    static List<Point2> getLinearPathPoints(Path2D path, PathIterator iter) {
        ArrayList<Point2> points = new ArrayList<Point2>();
        double[] seg = new double[6];
        while (!iter.isDone()) {
            switch (iter.currentSegment(seg)) {
                case 0: 
                case 1: {
                    points.add(new Point2(seg[0], seg[1]));
                    break;
                }
                case 4: {
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid polygon " + String.valueOf(path) + " - only line connections are allowed");
                }
            }
            iter.next();
        }
        return points;
    }

    static List<Vertices> getVertices(Shape shape) {
        Path2D path = shape instanceof Path2D ? (Path2D)shape : new Path2D.Float(shape);
        PathIterator iter = path.getPathIterator(null, 0.5);
        ArrayList<Vertices> verticesList = new ArrayList<Vertices>();
        MutableVertices vertices = null;
        double[] seg = new double[6];
        while (!iter.isDone()) {
            switch (iter.currentSegment(seg)) {
                case 0: {
                    vertices = new DefaultMutableVertices(new DefaultVertices());
                }
                case 1: {
                    vertices.add(seg[0], seg[1]);
                    break;
                }
                case 4: {
                    vertices.close();
                    verticesList.add(vertices.getVertices());
                    break;
                }
                default: {
                    throw new RuntimeException("Invalid polygon " + String.valueOf(path) + " - only line connections are allowed");
                }
            }
            iter.next();
        }
        return verticesList;
    }

    static Point2 scalePoint(Point2 p, double scaleX, double scaleY, double originX, double originY) {
        return new Point2(RoiTools.scaleOrdinate(p.getX(), scaleX, originX), RoiTools.scaleOrdinate(p.getY(), scaleY, originY));
    }

    static double scaleOrdinate(double v, double scale, double origin) {
        v -= origin;
        v *= scale;
        return v += origin;
    }

    public static enum CombineOp {
        ADD,
        SUBTRACT,
        INTERSECT;

    }
}

