/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.projects;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import java.awt.Desktop;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.lang.ref.SoftReference;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.imageio.ImageIO;
import javax.swing.SwingUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.GeneralTools;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerBuilder;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.io.GsonTools;
import qupath.lib.io.PathIO;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.projects.Project;
import qupath.lib.projects.ProjectImageEntry;
import qupath.lib.projects.ResourceManager;

class DefaultProject
implements Project<BufferedImage> {
    public static final String IMAGE_ID = "PROJECT_ENTRY_ID";
    private static String ext = "qpproj";
    private static Logger logger = LoggerFactory.getLogger(DefaultProject.class);
    private final String LATEST_VERSION = GeneralTools.getVersion();
    private final Map<String, String> metadata;
    private String version = null;
    private final File dirBase;
    private File file;
    private String name = null;
    private URI previousURI;
    private List<PathClass> pathClasses = new ArrayList<PathClass>();
    private boolean maskNames = false;
    private List<DefaultProjectImageEntry> images = new ArrayList<DefaultProjectImageEntry>();
    private long creationTimestamp;
    private long modificationTimestamp;
    private AtomicLong counter = new AtomicLong(0L);

    DefaultProject(File file) {
        this.file = file;
        if (file.isDirectory()) {
            this.dirBase = file;
            this.file = DefaultProject.getUniqueFile(this.dirBase, "project", ext);
        } else {
            this.dirBase = file.getParentFile();
        }
        this.creationTimestamp = System.currentTimeMillis();
        this.modificationTimestamp = System.currentTimeMillis();
        this.metadata = Collections.synchronizedMap(DefaultProject.getStoredMetadata(this.dirBase.toPath()));
    }

    @Override
    public URI getPreviousURI() {
        return this.previousURI;
    }

    static synchronized File getUniqueFile(File dir, String name, String ext) {
        if (!((String)ext).startsWith(".")) {
            ext = "." + (String)ext;
        }
        File file = new File(dir, name + (String)ext);
        int count = 0;
        while (file.exists()) {
            file = new File(dir, name + "-" + ++count + (String)ext);
        }
        return file;
    }

    static DefaultProject loadFromFile(File file) throws IOException {
        DefaultProject project = new DefaultProject(file);
        project.loadProject();
        return project;
    }

    @Override
    public List<PathClass> getPathClasses() {
        return Collections.unmodifiableList(this.pathClasses);
    }

    @Override
    public boolean setPathClasses(Collection<? extends PathClass> pathClasses) {
        if (this.pathClasses.size() != pathClasses.size() || this.pathClasses.containsAll(pathClasses)) {
            this.pathClasses.clear();
            this.pathClasses.addAll(pathClasses);
        }
        if (this.file.exists()) {
            try {
                logger.debug("Writing PathClasses to project");
                this.writePathClasses(this.pathClasses);
            }
            catch (IOException e) {
                logger.warn("Unable to write classes to project", (Throwable)e);
            }
        }
        return true;
    }

    private boolean addImage(ProjectImageEntry<BufferedImage> entry) {
        if (entry instanceof DefaultProjectImageEntry) {
            return this.addImage((DefaultProjectImageEntry)entry);
        }
        try {
            return this.addImage(new DefaultProjectImageEntry(entry.getServerBuilder(), null, entry.getImageName(), entry.getDescription(), entry.getMetadataMap()));
        }
        catch (IOException e) {
            logger.error("Unable to add entry " + String.valueOf(entry), (Throwable)e);
            return false;
        }
    }

    private boolean addImage(DefaultProjectImageEntry entry) {
        return this.images.add(entry);
    }

    private File getFile() {
        return this.file;
    }

    @Override
    public Path getPath() {
        return this.getFile().toPath();
    }

    @Override
    public URI getURI() {
        return this.getFile().toURI();
    }

    private File getBaseDirectory() {
        return this.dirBase;
    }

    private Path getBasePath() {
        return this.getBaseDirectory().toPath();
    }

    private Path getClassifiersPath() {
        return Paths.get(this.getBasePath().toString(), "classifiers");
    }

    List<String> listFilenames(Path path, String ext) throws IOException {
        if (!Files.isDirectory(path, new LinkOption[0])) {
            return Collections.emptyList();
        }
        try (Stream<Path> stream = Files.list(path);){
            List<String> list = stream.filter(p -> Files.isRegularFile(p, new LinkOption[0]) && p.toString().endsWith(ext)).map(p -> this.nameWithoutExtension((Path)p, ext)).toList();
            return list;
        }
    }

    String nameWithoutExtension(Path path, String ext) {
        String name = path.getFileName().toString();
        if (name.endsWith(ext)) {
            return name.substring(0, name.length() - ext.length());
        }
        return name;
    }

    public int size() {
        return this.images.size();
    }

    @Override
    public boolean isEmpty() {
        return this.images.isEmpty();
    }

    @Override
    public ProjectImageEntry<BufferedImage> addImage(ImageServerBuilder.ServerBuilder<BufferedImage> builder) throws IOException {
        DefaultProjectImageEntry entry = new DefaultProjectImageEntry(builder, null, null, null, null);
        if (this.addImage(entry)) {
            return entry;
        }
        return null;
    }

    @Override
    public ProjectImageEntry<BufferedImage> addDuplicate(ProjectImageEntry<BufferedImage> entry, boolean copyData) throws IOException {
        DefaultProjectImageEntry entryNew = new DefaultProjectImageEntry(entry.getServerBuilder(), null, entry.getImageName(), entry.getDescription(), entry.getMetadataMap());
        if (this.addImage(entryNew)) {
            if (copyData) {
                entryNew.copyDataFromEntry(entry);
            } else {
                BufferedImage img = entry.getThumbnail();
                if (img != null) {
                    entryNew.setThumbnail(img);
                }
            }
            return entryNew;
        }
        throw new IOException("Unable to add duplicate of " + String.valueOf(entry));
    }

    @Override
    public ProjectImageEntry<BufferedImage> getEntry(ImageData<BufferedImage> imageData) {
        Object id = imageData.getProperty(IMAGE_ID);
        for (DefaultProjectImageEntry entry : this.images) {
            if (!entry.getFullProjectEntryID().equals(id)) continue;
            return entry;
        }
        return null;
    }

    @Override
    public void removeImage(ProjectImageEntry<?> entry, boolean removeAllData) {
        boolean couldRemove = this.images.remove(entry);
        if (couldRemove && removeAllData && entry instanceof DefaultProjectImageEntry) {
            DefaultProjectImageEntry defaultEntry = (DefaultProjectImageEntry)entry;
            defaultEntry.moveDataToTrash();
        }
    }

    @Override
    public void removeAllImages(Collection<ProjectImageEntry<BufferedImage>> entries, boolean removeAllData) {
        for (ProjectImageEntry<BufferedImage> entry : entries) {
            this.removeImage(entry, removeAllData);
        }
    }

    @Override
    public synchronized void syncChanges() throws IOException {
        this.writeProject(this.getFile());
        this.writePathClasses(this.pathClasses);
        DefaultProject.setStoredMetadata(this.getBasePath(), this.metadata);
    }

    @Override
    public boolean getMaskImageNames() {
        return this.maskNames;
    }

    @Override
    public void setMaskImageNames(boolean maskNames) {
        this.maskNames = maskNames;
    }

    @Override
    public List<ProjectImageEntry<BufferedImage>> getImageList() {
        return Collections.unmodifiableList(this.images);
    }

    @Override
    public String getName() {
        if (this.name != null) {
            return this.name;
        }
        if (this.dirBase == null || !this.dirBase.isDirectory()) {
            return "(Project directory missing)";
        }
        if (this.file != null && this.file.exists() && this.file != this.dirBase) {
            return this.dirBase.getName() + "/" + this.file.getName();
        }
        return this.dirBase.getName();
    }

    public String toString() {
        return "Project: " + Project.getNameFromURI(this.getURI());
    }

    @Override
    public long getCreationTimestamp() {
        return this.creationTimestamp;
    }

    @Override
    public long getModificationTimestamp() {
        return this.modificationTimestamp;
    }

    Path ensureDirectoryExists(Path path) throws IOException {
        if (!Files.isDirectory(path, new LinkOption[0])) {
            Files.createDirectories(path, new FileAttribute[0]);
        }
        return path;
    }

    synchronized <T> void writeProject(File fileProject) throws IOException {
        if (fileProject == null) {
            throw new IOException("No file found, cannot write project: " + String.valueOf(this));
        }
        Gson gson = GsonTools.getInstance(true);
        JsonObject builder = new JsonObject();
        builder.addProperty("version", this.LATEST_VERSION);
        builder.addProperty("createTimestamp", (Number)this.getCreationTimestamp());
        builder.addProperty("modifyTimestamp", (Number)this.getModificationTimestamp());
        builder.addProperty("uri", fileProject.toURI().toString());
        builder.addProperty("lastID", (Number)this.counter.get());
        builder.add("images", gson.toJsonTree(this.images));
        Path pathProject = fileProject.toPath();
        Path pathTempNew = new File(fileProject.getAbsolutePath() + ".tmp").toPath();
        logger.debug("Writing project to {}", (Object)pathTempNew);
        try (BufferedWriter writer = Files.newBufferedWriter(pathTempNew, StandardCharsets.UTF_8, new OpenOption[0]);){
            gson.toJson((JsonElement)builder, (Appendable)writer);
        }
        if (fileProject.exists()) {
            Path pathBackup = new File(fileProject.getAbsolutePath() + ".backup").toPath();
            logger.debug("Backing up existing project to {}", (Object)pathBackup);
            Files.move(pathProject, pathBackup, StandardCopyOption.REPLACE_EXISTING);
        }
        logger.debug("Renaming project to {}", (Object)pathProject);
        Files.move(pathTempNew, pathProject, StandardCopyOption.REPLACE_EXISTING);
    }

    void loadProject() throws IOException {
        File fileProject = this.getFile();
        try (BufferedReader fileReader = Files.newBufferedReader(fileProject.toPath(), StandardCharsets.UTF_8);){
            Gson gson = GsonTools.getInstance();
            JsonObject element = (JsonObject)gson.fromJson((Reader)fileReader, JsonObject.class);
            this.creationTimestamp = element.get("createTimestamp").getAsLong();
            this.modificationTimestamp = element.get("modifyTimestamp").getAsLong();
            if (element.has("uri")) {
                try {
                    this.previousURI = new URI(element.get("uri").getAsString());
                }
                catch (URISyntaxException e) {
                    logger.warn("Error parsing previous URI: " + e.getLocalizedMessage(), (Throwable)e);
                }
            } else {
                logger.debug("No previous URI found in project");
            }
            if (element.has("version")) {
                this.version = element.get("version").getAsString();
            }
            if (Arrays.asList("v0.2.0-m2", "v0.2.0-m1").contains(this.version)) {
                throw new IOException("Older projects written with " + this.version + " are not compatible with this version of QuPath, sorry!");
            }
            if (this.version == null && !element.has("lastID")) {
                throw new IOException("QuPath project is missing a version number and last ID (was it written with an old version?)");
            }
            long lastID = 0L;
            List images = element.has("images") ? (List)gson.fromJson(element.get("images"), new TypeToken<ArrayList<DefaultProjectImageEntry>>(this){}.getType()) : Collections.emptyList();
            for (DefaultProjectImageEntry entry : images) {
                this.addImage(new DefaultProjectImageEntry(entry));
                lastID = Math.max(lastID, entry.entryID);
            }
            if (element.has("lastID")) {
                lastID = Math.max(lastID, element.get("lastID").getAsLong());
            }
            this.counter.set(lastID);
            this.pathClasses.addAll(this.loadPathClasses());
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    void writePathClasses(Collection<PathClass> pathClasses) throws IOException {
        Path path = Paths.get(this.ensureDirectoryExists(this.getClassifiersPath()).toString(), "classes.json");
        if (pathClasses == null || pathClasses.isEmpty()) {
            Files.deleteIfExists(path);
            return;
        }
        JsonArray pathClassArray = new JsonArray();
        for (PathClass pathClass : pathClasses) {
            if (pathClass == PathClass.NULL_CLASS) continue;
            JsonObject jsonEntry = new JsonObject();
            jsonEntry.addProperty("name", pathClass.toString());
            jsonEntry.addProperty("color", (Number)pathClass.getColor());
            pathClassArray.add((JsonElement)jsonEntry);
        }
        JsonObject element = new JsonObject();
        element.add("pathClasses", (JsonElement)pathClassArray);
        try (BufferedWriter writer = Files.newBufferedWriter(path, StandardCharsets.UTF_8, new OpenOption[0]);){
            GsonTools.getInstance(true).toJson((JsonElement)element, (Appendable)writer);
        }
    }

    Collection<PathClass> loadPathClasses() throws IOException {
        Path path = Paths.get(this.ensureDirectoryExists(this.getClassifiersPath()).toString(), "classes.json");
        if (!Files.isRegularFile(path, new LinkOption[0])) {
            return Collections.emptyList();
        }
        Gson gson = GsonTools.getInstance();
        try (BufferedReader reader = Files.newBufferedReader(path, StandardCharsets.UTF_8);){
            JsonObject element = (JsonObject)gson.fromJson((Reader)reader, JsonObject.class);
            JsonElement pathClassesElement = element.get("pathClasses");
            if (pathClassesElement != null && pathClassesElement.isJsonArray()) {
                JsonArray pathClassesArray = pathClassesElement.getAsJsonArray();
                ArrayList<PathClass> pathClasses = new ArrayList<PathClass>();
                for (int i = 0; i < pathClassesArray.size(); ++i) {
                    JsonObject pathClassObject = pathClassesArray.get(i).getAsJsonObject();
                    if (!pathClassObject.has("name")) continue;
                    String name = pathClassObject.get("name").getAsString();
                    if (PathClass.NULL_CLASS.toString().equals(name)) {
                        logger.debug("Skipping PathClass '{}' - shares a name with the null class", (Object)name);
                        continue;
                    }
                    Integer color = null;
                    if (pathClassObject.has("color") && !pathClassObject.get("color").isJsonNull()) {
                        color = pathClassObject.get("color").getAsInt();
                    }
                    PathClass pathClass = PathClass.fromString(name, color);
                    if (color != null) {
                        pathClass.setColor(color);
                    }
                    pathClasses.add(pathClass);
                }
                ArrayList<PathClass> arrayList = pathClasses;
                return arrayList;
            }
            List<PathClass> list = Collections.emptyList();
            return list;
        }
    }

    @Override
    public String getVersion() {
        return this.version;
    }

    @Override
    public <S, R extends S> ResourceManager.Manager<R> getResources(String location, Class<S> cls, String ext) {
        Path path = Paths.get(this.getBasePath().toString(), location);
        if ((ext = ext.toLowerCase().strip()).startsWith(".")) {
            ext = ext.substring(1);
        }
        switch (ext) {
            case "json": {
                return new ResourceManager.JsonFileResourceManager<S>(path, cls);
            }
        }
        if (String.class.equals(cls)) {
            return new ResourceManager.StringFileResourceManager(path, ext);
        }
        return null;
    }

    @Override
    public Project<BufferedImage> createSubProject(String name, Collection<ProjectImageEntry<BufferedImage>> entries) {
        if (!((String)name).endsWith(ext)) {
            name = ((String)name).endsWith(".") ? (String)name + ext : (String)name + "." + ext;
        }
        File file = new File(this.getBaseDirectory(), (String)name);
        DefaultProject project = new DefaultProject(file);
        boolean changes = false;
        for (ProjectImageEntry<BufferedImage> entry : entries) {
            changes = project.addImage(entry) | changes;
        }
        return project;
    }

    @Override
    public Map<String, String> getMetadata() {
        return this.metadata;
    }

    private static Map<String, String> getStoredMetadata(Path projectPath) {
        LinkedHashMap<String, String> metadata = new LinkedHashMap<String, String>();
        Path metadataPath = DefaultProject.getMetadataPath(projectPath);
        if (Files.exists(metadataPath, new LinkOption[0])) {
            try (BufferedReader reader = Files.newBufferedReader(metadataPath, StandardCharsets.UTF_8);){
                metadata.putAll((Map)GsonTools.getInstance().fromJson((Reader)reader, new TypeToken<Map<String, String>>(){}.getType()));
            }
            catch (IOException e) {
                logger.error("Error while retrieving project metadata", (Throwable)e);
            }
        }
        return metadata;
    }

    private static void setStoredMetadata(Path projectPath, Map<String, String> metadata) {
        Path metadataPath = DefaultProject.getMetadataPath(projectPath);
        try {
            Files.createDirectories(metadataPath.getParent(), new FileAttribute[0]);
            try (BufferedWriter writer = Files.newBufferedWriter(metadataPath, StandardCharsets.UTF_8, new OpenOption[0]);){
                GsonTools.getInstance().toJson(metadata, (Appendable)writer);
            }
        }
        catch (IOException e) {
            logger.error("Error while saving project metadata", (Throwable)e);
        }
    }

    private static Path getMetadataPath(Path projectPath) {
        return Paths.get(projectPath.toString(), "data", "metadata.json");
    }

    class DefaultProjectImageEntry
    implements ProjectImageEntry<BufferedImage> {
        private ImageServerBuilder.ServerBuilder<BufferedImage> serverBuilder;
        private long entryID;
        private String randomizedName = UUID.randomUUID().toString();
        private String imageName;
        private String description;
        private Map<String, String> metadata = Collections.synchronizedMap(new LinkedHashMap());
        private final Set<String> tags = Collections.synchronizedSet(new LinkedHashSet());
        private transient SoftReference<BufferedImage> cachedThumbnail;
        private transient ResourceManager.ImageResourceManager<BufferedImage> imageManager = null;

        DefaultProjectImageEntry(ImageServerBuilder.ServerBuilder<BufferedImage> builder) throws IOException {
            this(builder, null, null, null, null);
        }

        DefaultProjectImageEntry(ImageServerBuilder.ServerBuilder<BufferedImage> builder, Long entryID, String imageName, String description, Map<String, String> metadataMap) throws IOException {
            this.serverBuilder = builder;
            this.entryID = entryID == null ? DefaultProject.this.counter.incrementAndGet() : entryID.longValue();
            this.imageName = imageName == null ? "Image " + entryID : imageName;
            if (description != null) {
                this.setDescription(description);
            }
            if (metadataMap != null) {
                this.metadata.putAll(metadataMap);
            }
            this.writeServerBuilder();
        }

        DefaultProjectImageEntry(DefaultProjectImageEntry entry) {
            this.serverBuilder = entry.serverBuilder;
            this.entryID = entry.entryID;
            this.imageName = entry.imageName;
            this.description = entry.description;
            if (entry.metadata != null) {
                this.metadata.putAll(entry.metadata);
            }
            if (entry.tags != null) {
                this.tags.addAll(entry.tags);
            }
        }

        void copyDataFromEntry(ProjectImageEntry<BufferedImage> entry) throws IOException {
            if (entry instanceof DefaultProjectImageEntry) {
                this.copyDataFromEntry((DefaultProjectImageEntry)entry);
            } else {
                BufferedImage imgThumbnail;
                if (entry.hasImageData()) {
                    this.saveImageData(entry.readImageData());
                }
                if (this.getThumbnail() == null && (imgThumbnail = entry.getThumbnail()) != null) {
                    this.setThumbnail(imgThumbnail);
                }
            }
        }

        void copyDataFromEntry(DefaultProjectImageEntry entry) throws IOException {
            this.getEntryPath(true);
            if (Files.exists(entry.getImageDataPath(), new LinkOption[0])) {
                Files.copy(entry.getImageDataPath(), this.getImageDataPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            if (Files.exists(entry.getDataSummaryPath(), new LinkOption[0])) {
                Files.copy(entry.getDataSummaryPath(), this.getDataSummaryPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            if (Files.exists(entry.getServerPath(), new LinkOption[0])) {
                Files.copy(entry.getServerPath(), this.getServerPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            if (this.getThumbnail() == null && Files.exists(entry.getThumbnailPath(), new LinkOption[0])) {
                Files.copy(entry.getThumbnailPath(), this.getThumbnailPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }

        @Override
        public synchronized ResourceManager.Manager<ImageServer<BufferedImage>> getImages() {
            if (this.imageManager == null) {
                this.imageManager = new ResourceManager.ImageResourceManager<BufferedImage>(DefaultProject.this.getPath(), BufferedImage.class);
            }
            return this.imageManager;
        }

        private String getFullProjectEntryID() {
            return DefaultProject.this.file.getAbsolutePath() + "::" + this.getID();
        }

        @Override
        public String getID() {
            return Long.toString(this.entryID);
        }

        @Override
        public Collection<URI> getURIs() throws IOException {
            if (this.serverBuilder == null) {
                return Collections.emptyList();
            }
            return this.serverBuilder.getURIs();
        }

        @Override
        public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
            boolean changes;
            ImageServerBuilder.ServerBuilder<BufferedImage> builderBefore = this.serverBuilder;
            this.serverBuilder = this.serverBuilder.updateURIs(replacements);
            boolean bl = changes = builderBefore != this.serverBuilder;
            if (changes) {
                this.writeServerBuilder();
            }
            return changes;
        }

        String getUniqueName() {
            return Long.toString(this.entryID);
        }

        @Override
        public String getImageName() {
            if (DefaultProject.this.maskNames) {
                return this.randomizedName;
            }
            return this.imageName;
        }

        @Override
        public String getOriginalImageName() {
            return this.imageName;
        }

        public String toString() {
            Object s = this.getImageName();
            if (!this.metadata.isEmpty()) {
                s = (String)s + " - " + this.getMetadataSummaryString();
            }
            return s;
        }

        @Override
        public void setImageName(String name) {
            this.imageName = name;
        }

        @Override
        public String getDescription() {
            return this.description;
        }

        @Override
        public void setDescription(String description) {
            this.description = description;
        }

        @Override
        public Map<String, String> getMetadata() {
            return this.metadata;
        }

        @Override
        public ImageServerBuilder.ServerBuilder<BufferedImage> getServerBuilder() {
            return this.serverBuilder;
        }

        private Path getEntryPath(boolean create) throws IOException {
            Path path = this.getEntryPath();
            if (create && !Files.exists(path, new LinkOption[0])) {
                Files.createDirectories(path, new FileAttribute[0]);
            }
            return path;
        }

        @Override
        public Path getEntryPath() {
            return Paths.get(DefaultProject.this.getBasePath().toString(), "data", this.getUniqueName());
        }

        private Path getImageDataPath() {
            return Paths.get(this.getEntryPath().toString(), "data.qpdata");
        }

        private Path getBackupImageDataPath() {
            return Paths.get(this.getEntryPath().toString(), "data.qpdata.bkp");
        }

        private Path getDataSummaryPath() {
            return Paths.get(this.getEntryPath().toString(), "summary.json");
        }

        private Path getServerPath() {
            return Paths.get(this.getEntryPath().toString(), "server.json");
        }

        private Path getThumbnailPath() {
            return Paths.get(this.getEntryPath().toString(), "thumbnail.jpg");
        }

        @Override
        public synchronized ImageData<BufferedImage> readImageData() throws IOException {
            Path pathBackup;
            Path path = this.getImageDataPath();
            ImageData<BufferedImage> imageData = null;
            if (Files.exists(path, new LinkOption[0])) {
                try {
                    imageData = PathIO.readImageData(path, this.getServerBuilder());
                }
                catch (Exception e) {
                    logger.error("Error reading image data from {}", (Object)path, (Object)e);
                }
            }
            if (imageData == null && Files.exists(pathBackup = this.getBackupImageDataPath(), new LinkOption[0])) {
                try {
                    imageData = PathIO.readImageData(pathBackup, this.getServerBuilder());
                    logger.warn("Restored previous ImageData from {}", (Object)pathBackup);
                }
                catch (IOException e) {
                    logger.error("Error reading backup image data from {}", (Object)pathBackup, (Object)e);
                }
            }
            if (imageData == null) {
                imageData = new ImageData<BufferedImage>(this.getServerBuilder(), new PathObjectHierarchy(), ImageData.ImageType.UNSET);
            }
            imageData.setProperty(DefaultProject.IMAGE_ID, this.getFullProjectEntryID());
            imageData.setChanged(false);
            imageData.updateServerMetadata(new ImageServerMetadata.Builder(imageData.getServerMetadata()).name(this.getImageName()).build());
            return imageData;
        }

        @Override
        public synchronized void saveImageData(ImageData<BufferedImage> imageData) throws IOException {
            String id;
            this.getEntryPath(true);
            Path pathData = this.getImageDataPath();
            Path pathBackup = this.getBackupImageDataPath();
            if (Files.exists(pathData, new LinkOption[0])) {
                Files.move(pathData, pathBackup, StandardCopyOption.REPLACE_EXISTING);
            }
            if (!Objects.equals(id = this.getFullProjectEntryID(), imageData.getProperty(DefaultProject.IMAGE_ID))) {
                logger.warn("Updating ID property to {}", (Object)id);
                imageData.setProperty(DefaultProject.IMAGE_ID, id);
            }
            long timestamp = 0L;
            try (OutputStream stream = Files.newOutputStream(pathData, new OpenOption[0]);){
                logger.debug("Saving image data to {}", (Object)pathData);
                PathIO.writeImageData(stream, imageData);
                imageData.setLastSavedPath(pathData.toString(), true);
                timestamp = Files.getLastModifiedTime(pathData, new LinkOption[0]).toMillis();
                if (Files.exists(pathBackup, new LinkOption[0])) {
                    Files.delete(pathBackup);
                }
            }
            catch (IOException e) {
                if (Files.exists(pathBackup, new LinkOption[0])) {
                    logger.warn("Exception writing image file - attempting to restore {} from backup", (Object)pathData);
                    Files.move(pathBackup, pathData, StandardCopyOption.REPLACE_EXISTING);
                }
                throw e;
            }
            ImageServerBuilder.ServerBuilder<BufferedImage> currentServerBuilder = imageData.getServerBuilder();
            if (currentServerBuilder != null && !currentServerBuilder.equals(this.serverBuilder)) {
                this.serverBuilder = currentServerBuilder;
                this.writeServerBuilder();
            }
            Path pathSummary = this.getDataSummaryPath();
            try (BufferedWriter out = Files.newBufferedWriter(pathSummary, StandardCharsets.UTF_8, new OpenOption[0]);){
                GsonTools.getInstance().toJson((Object)new ImageDataSummary(imageData, timestamp), (Appendable)out);
            }
        }

        private void writeServerBuilder() throws IOException {
            this.getEntryPath(true);
            Path pathServer = this.getServerPath();
            try (BufferedWriter out = Files.newBufferedWriter(pathServer, StandardCharsets.UTF_8, new OpenOption[0]);){
                GsonTools.getInstance(true).toJson(this.serverBuilder, ImageServerBuilder.ServerBuilder.class, (Appendable)out);
            }
            catch (Exception e) {
                logger.warn("Unable to write server to {}", (Object)pathServer);
                Files.deleteIfExists(pathServer);
            }
        }

        @Override
        public boolean hasImageData() {
            return Files.exists(this.getImageDataPath(), new LinkOption[0]);
        }

        @Override
        public synchronized PathObjectHierarchy readHierarchy() throws IOException {
            Path path = this.getImageDataPath();
            if (Files.exists(path, new LinkOption[0])) {
                try (InputStream stream = Files.newInputStream(path, new OpenOption[0]);){
                    PathObjectHierarchy pathObjectHierarchy = PathIO.readHierarchy(stream);
                    return pathObjectHierarchy;
                }
            }
            return new PathObjectHierarchy();
        }

        @Override
        public String getSummary() {
            File file;
            StringBuilder sb = new StringBuilder();
            sb.append(this.getImageName()).append("\n");
            sb.append("ID:\t").append(this.getID()).append("\n\n");
            if (!this.getMetadataMap().isEmpty()) {
                for (Map.Entry<String, String> mapEntry : this.getMetadataMap().entrySet()) {
                    sb.append(mapEntry.getKey()).append(":\t").append(mapEntry.getValue()).append("\n");
                }
                sb.append("\n");
            }
            if ((file = this.getImageDataPath().toFile()) != null && file.exists()) {
                double sizeMB = (double)file.length() / 1024.0 / 1024.0;
                sb.append(String.format("Data file:\t%.2f MB", sizeMB)).append("\n");
            } else {
                sb.append("No data file");
            }
            return sb.toString();
        }

        @Override
        public synchronized BufferedImage getThumbnail() throws IOException {
            Path path;
            boolean cached;
            long startTime = System.nanoTime();
            BufferedImage thumbnail = this.cachedThumbnail == null ? null : this.cachedThumbnail.get();
            boolean bl = cached = thumbnail != null;
            if (thumbnail == null && Files.exists(path = this.getThumbnailPath(), new LinkOption[0])) {
                try (InputStream stream = Files.newInputStream(path, new OpenOption[0]);){
                    thumbnail = ImageIO.read(stream);
                    this.cachedThumbnail = new SoftReference<BufferedImage>(thumbnail);
                }
            }
            long endTime = System.nanoTime();
            if (cached) {
                logger.trace("Thumbnail accessed from cache in {} ms", (Object)((endTime - startTime) / 1000000L));
            } else {
                logger.trace("Thumbnail read in {} ms", (Object)((endTime - startTime) / 1000000L));
            }
            return thumbnail;
        }

        @Override
        public synchronized void setThumbnail(BufferedImage img) throws IOException {
            this.resetCachedThumbnail();
            this.getEntryPath(true);
            Path path = this.getThumbnailPath();
            if (img == null) {
                if (Files.exists(path, new LinkOption[0])) {
                    logger.debug("Deleting thumbnail for {}", (Object)path);
                    Files.delete(path);
                }
            } else {
                try (OutputStream stream = Files.newOutputStream(path, new OpenOption[0]);){
                    logger.debug("Writing thumbnail to {}", (Object)path);
                    ImageIO.write((RenderedImage)img, "JPEG", stream);
                }
            }
        }

        @Override
        public Set<String> getTags() {
            return this.tags;
        }

        private synchronized void resetCachedThumbnail() {
            logger.trace("Resetting cached thumbnail for {}", (Object)this.getID());
            this.cachedThumbnail = null;
        }

        synchronized void moveDataToTrash() {
            Desktop desktop;
            Path path = this.getEntryPath();
            if (!Files.exists(path, new LinkOption[0])) {
                return;
            }
            if (Desktop.isDesktopSupported() && (desktop = Desktop.getDesktop()).isSupported(Desktop.Action.MOVE_TO_TRASH)) {
                if (SwingUtilities.isEventDispatchThread()) {
                    if (desktop.moveToTrash(path.toFile())) {
                        return;
                    }
                } else {
                    SwingUtilities.invokeLater(() -> {
                        if (!desktop.moveToTrash(path.toFile())) {
                            logger.warn("Unable to move {} to trash - please delete manually if required", (Object)path);
                        }
                    });
                    return;
                }
            }
            logger.warn("Unable to move {} to trash - please delete manually if required", (Object)path);
        }
    }

    static class HierarchySummary {
        private int nObjects;
        private Integer nTMACores;
        private Map<String, Long> objectTypeCounts;
        private Map<String, Long> annotationClassificationCounts;
        private Map<String, Long> detectionClassificationCounts;

        HierarchySummary(PathObjectHierarchy hierarchy) {
            Collection<PathObject> pathObjects = hierarchy.getObjects(null, null);
            this.nObjects = pathObjects.size();
            this.objectTypeCounts = pathObjects.stream().collect(Collectors.groupingBy(p -> PathObjectTools.getSuitableName(p.getClass(), true), Collectors.counting()));
            this.annotationClassificationCounts = pathObjects.stream().filter(p -> p.isAnnotation()).collect(Collectors.groupingBy(p -> HierarchySummary.pathClassToString(p.getPathClass()), Collectors.counting()));
            this.detectionClassificationCounts = pathObjects.stream().filter(p -> p.isDetection()).collect(Collectors.groupingBy(p -> HierarchySummary.pathClassToString(p.getPathClass()), Collectors.counting()));
        }

        static String pathClassToString(PathClass pathClass) {
            return pathClass == null ? "Unclassified" : pathClass.toString();
        }
    }

    static class ServerSummary {
        private int width;
        private int height;
        private int sizeC;
        private int sizeZ;
        private int sizeT;

        ServerSummary(ImageServer<?> server) {
            this.width = server.getWidth();
            this.height = server.getHeight();
            this.sizeC = server.nChannels();
            this.sizeZ = server.nZSlices();
            this.sizeT = server.nTimepoints();
        }

        ServerSummary(ImageServerMetadata metadata) {
            this.width = metadata.getWidth();
            this.height = metadata.getHeight();
            this.sizeC = metadata.getChannels().size();
            this.sizeZ = metadata.getSizeZ();
            this.sizeT = metadata.getSizeZ();
        }
    }

    static class ImageDataSummary {
        private long timestamp;
        private ImageData.ImageType imageType;
        private ServerSummary server;
        private HierarchySummary hierarchy;

        ImageDataSummary(ImageData<?> imageData, long timestamp) {
            this.imageType = imageData.getImageType();
            this.server = imageData.isLoaded() ? new ServerSummary(imageData.getServer()) : new ServerSummary(imageData.getServerMetadata());
            this.timestamp = timestamp;
            this.hierarchy = new HierarchySummary(imageData.getHierarchy());
        }

        public String toString() {
            return GsonTools.getInstance().toJson((Object)this);
        }
    }
}

