/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.io;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.GeneralTools;
import qupath.lib.io.UriResource;
import qupath.lib.projects.Project;

public class UriUpdater<T extends UriResource> {
    private static final Logger logger = LoggerFactory.getLogger(UriUpdater.class);
    private Collection<T> resources;
    private Collection<SingleUriItem> items;
    private Map<SingleUriItem, SingleUriItem> replacements;
    private int maxDepth = 2;

    public static <T extends UriResource> UriUpdater<T> create(T resource) throws IOException {
        return UriUpdater.create(Collections.singleton(resource));
    }

    public static <T extends UriResource> UriUpdater<T> create(Collection<T> resources) throws IOException {
        return UriUpdater.create(new ArrayList<T>(resources), new ArrayList<SingleUriItem>(), new HashMap<SingleUriItem, SingleUriItem>());
    }

    public static <T extends UriResource> UriUpdater<T> create(Collection<T> resources, Collection<SingleUriItem> items, Map<SingleUriItem, SingleUriItem> replacements) throws IOException {
        return new UriUpdater<T>(resources, items, replacements);
    }

    public static UriResource wrap(URI ... uris) {
        return new DefaultUriResource(uris);
    }

    public static UriResource wrap(Collection<URI> uris) {
        return UriUpdater.wrap((URI[])uris.toArray(URI[]::new));
    }

    public static URI locateFile(URI uri, int searchDepth, Path ... searchPaths) throws IOException {
        UriResource resource = UriUpdater.wrap(uri);
        UriUpdater<UriResource> updater = UriUpdater.create(resource);
        int nMissing = updater.countMissing();
        if (nMissing == 0) {
            return uri;
        }
        updater.searchDepth(searchDepth);
        for (Path p : searchPaths) {
            if (p == null) continue;
            if (!Files.isDirectory(p, new LinkOption[0])) {
                p = p.getParent();
            }
            updater.searchPath(p);
            updater.applyReplacements();
            nMissing = updater.countMissing();
            if (nMissing == 0) break;
        }
        return resource.getURIs().iterator().next();
    }

    public static String locateFile(String path, int searchDepth, Path ... searchPaths) throws IOException {
        try {
            URI uri = GeneralTools.toURI(path);
            URI uri2 = UriUpdater.locateFile(uri, searchDepth, searchPaths);
            if (uri2 == null || Objects.equals(uri, uri2)) {
                return path;
            }
            return Paths.get(uri2).toAbsolutePath().toString();
        }
        catch (URISyntaxException e) {
            throw new IOException(e);
        }
    }

    public static int fixUris(UriResource resource, Project<?> project) {
        int n = 0;
        try {
            Collection<URI> uris = resource.getURIs();
            if (uris.isEmpty()) {
                return n;
            }
            UriUpdater<UriResource> updater = UriUpdater.create(resource);
            int nMissing = updater.countMissing();
            if (nMissing == 0) {
                return n;
            }
            if (project != null) {
                updater.relative(project.getPreviousURI(), project.getURI());
                n += updater.applyReplacements();
                if (updater.countMissing() == 0) {
                    return n;
                }
                Path path = project.getPath();
                if (!Files.isDirectory(path, new LinkOption[0])) {
                    path = path.getParent();
                }
                updater.searchDepth(4);
                updater.searchPath(path);
                n += updater.applyReplacements();
                if (updater.countMissing() == 0) {
                    return n;
                }
            }
        }
        catch (IOException e) {
            logger.warn("Error fixing URIs for '{}'", (Object)resource);
            logger.debug(e.getLocalizedMessage(), (Throwable)e);
        }
        return n;
    }

    private UriUpdater(Collection<T> resources, Collection<SingleUriItem> items, Map<SingleUriItem, SingleUriItem> replacements) throws IOException {
        this.resources = resources;
        this.items = items;
        this.replacements = replacements;
        this.updateItems();
        this.replacements.clear();
    }

    public UriUpdater<T> searchDepth(int maxDepth) {
        this.maxDepth = maxDepth;
        return this;
    }

    public UriUpdater<T> relative(URI uriOriginal, URI uriCurrent) {
        Path pathOriginal = uriOriginal == null ? null : GeneralTools.toPath(uriOriginal);
        Path pathCurrent = uriCurrent == null ? null : GeneralTools.toPath(uriCurrent);
        return this.relative(pathOriginal, pathCurrent);
    }

    public UriUpdater<T> relative(Path pathOriginal, Path pathCurrent) {
        if (pathOriginal != null && pathCurrent != null) {
            int n = UriUpdater.updateReplacementsRelative(this.items, pathOriginal, pathCurrent, this.replacements);
            if (n > 0) {
                logger.debug("Updated {} URI(s) with relative paths", (Object)n);
            } else {
                logger.trace("Updated {} URIs with relative paths", (Object)n);
            }
        }
        return this;
    }

    public UriUpdater<T> searchPath(Path path) {
        try {
            int n = UriUpdater.searchDirectoriesRecursive(path.toFile(), this.items, this.maxDepth, this.replacements);
            logger.debug("Updated {} URI(s) with relative paths", (Object)n);
            return this;
        }
        catch (Exception e) {
            logger.error("Error searching {}", (Object)path);
            logger.error(e.getLocalizedMessage(), (Throwable)e);
            return this;
        }
    }

    public UriUpdater<T> makeReplacement(URI originalItem, URI updatedItem) {
        SingleUriItem item2;
        SingleUriItem item1 = new SingleUriItem(originalItem);
        SingleUriItem singleUriItem = item2 = updatedItem == null ? null : new SingleUriItem(updatedItem);
        if (item2 == null || Objects.equals(item1, item2)) {
            this.replacements.remove(item1);
        } else {
            this.replacements.put(item1, item2);
        }
        return this;
    }

    public Map<URI, URI> getReplacements() {
        return Collections.unmodifiableMap(this.replacements.entrySet().stream().filter(s -> s.getValue() != null).collect(Collectors.toMap(s -> ((SingleUriItem)s.getKey()).getURI(), s -> ((SingleUriItem)s.getValue()).getURI())));
    }

    public int applyReplacements() throws IOException {
        if (this.replacements.isEmpty()) {
            return 0;
        }
        int n = UriUpdater.replaceItems(this.resources, this.replacements);
        this.updateItems();
        this.replacements.clear();
        return n;
    }

    private void updateItems() throws IOException {
        List<SingleUriItem> newItems = UriUpdater.getItems(this.resources);
        if (!newItems.equals(this.items)) {
            this.items.clear();
            this.items.addAll(newItems);
        }
    }

    public int countMissing() {
        return this.getMissingItems().size();
    }

    public int countReplacements() {
        return (int)this.replacements.entrySet().stream().filter(e -> ((SingleUriItem)e.getKey()).getStatus() == UriStatus.MISSING && e.getValue() != null && ((SingleUriItem)e.getValue()).getStatus() != UriStatus.MISSING).count();
    }

    public Collection<SingleUriItem> getMissingItems() {
        return this.getItems(UriStatus.MISSING);
    }

    public Collection<SingleUriItem> getItems(UriStatus status) {
        if (status == null) {
            return Collections.unmodifiableCollection(this.items);
        }
        return this.items.stream().filter(i -> i.getStatus() == status).toList();
    }

    private static int updateReplacementsRelative(Collection<SingleUriItem> items, Path pathPrevious, Path pathBase, Map<SingleUriItem, SingleUriItem> replacements) {
        if (pathBase != null && Files.exists(pathBase, new LinkOption[0]) && !Files.isDirectory(pathBase, new LinkOption[0])) {
            pathBase = pathBase.getParent();
        }
        if (pathPrevious != null && Files.exists(pathPrevious, new LinkOption[0]) && !Files.isDirectory(pathPrevious, new LinkOption[0])) {
            pathPrevious = pathPrevious.getParent();
        }
        boolean tryRelative = pathBase != null && pathPrevious != null && !pathBase.equals(pathPrevious);
        int count = 0;
        for (SingleUriItem item : items) {
            if (item.getStatus() != UriStatus.MISSING || replacements.getOrDefault(item, null) != null) continue;
            Path pathItem = item.getPath();
            try {
                Path pathRelative;
                if (!tryRelative || pathItem == null || pathPrevious == null || !Objects.equals(pathItem.getRoot(), pathPrevious.getRoot()) || !Files.exists(pathRelative = pathBase.resolve(pathPrevious.relativize(pathItem)), new LinkOption[0])) continue;
                URI uri2 = pathRelative.normalize().toUri().normalize();
                replacements.put(item, new SingleUriItem(uri2));
                ++count;
            }
            catch (Exception e) {
                logger.warn("Error relativizing paths: {}", (Object)e.getLocalizedMessage());
                logger.debug(e.getLocalizedMessage(), (Throwable)e);
            }
        }
        return count;
    }

    private static int replaceItems(Collection<? extends UriResource> uriResources, Map<SingleUriItem, SingleUriItem> replacements) throws IOException {
        Map<URI, URI> map = replacements.entrySet().stream().collect(Collectors.toMap(e -> ((SingleUriItem)e.getKey()).getURI(), e -> ((SingleUriItem)e.getValue()).getURI()));
        return UriUpdater.replace(uriResources, map);
    }

    private static int replace(Collection<? extends UriResource> uriResources, Map<URI, URI> replacements) throws IOException {
        int count = 0;
        for (UriResource uriResource : uriResources) {
            if (!uriResource.updateURIs(replacements)) continue;
            ++count;
        }
        return count;
    }

    private static List<SingleUriItem> getItems(Collection<? extends UriResource> uriResources) throws IOException {
        LinkedHashSet<URI> imageUris = new LinkedHashSet<URI>();
        for (UriResource uriResource : uriResources) {
            imageUris.addAll(uriResource.getURIs());
        }
        return imageUris.stream().map(u -> new SingleUriItem((URI)u)).toList();
    }

    private static int searchDirectoriesRecursive(File dir, Collection<SingleUriItem> allItems, int maxDepth, Map<SingleUriItem, SingleUriItem> replacements) {
        Map<String, List<SingleUriItem>> missing = allItems.stream().filter(p -> p.getStatus() == UriStatus.MISSING && (p.getPath() != null || p.getURI().getPath() != null) && replacements.getOrDefault(p, null) == null).collect(Collectors.groupingBy(p -> {
            Path path = p.getPath() == null ? Paths.get(p.getURI().getPath(), new String[0]) : p.getPath();
            return path.getFileName().toString();
        }));
        int sizeBefore = replacements.size();
        UriUpdater.searchDirectoriesRecursive(dir, missing, maxDepth, replacements);
        return replacements.size() - sizeBefore;
    }

    private static void searchDirectoriesRecursive(File dir, Map<String, List<SingleUriItem>> missing, int maxDepth, Map<SingleUriItem, SingleUriItem> replacements) {
        if (dir == null || !dir.canRead() || !dir.isDirectory() || missing.isEmpty() || maxDepth <= 0) {
            return;
        }
        ArrayList<File> subdirs = new ArrayList<File>();
        logger.debug("Searching {}", (Object)dir);
        File[] list = dir.listFiles();
        if (list == null) {
            return;
        }
        for (File f : list) {
            if (f == null || f.isHidden()) continue;
            if (f.isFile()) {
                String name = f.getName();
                List<SingleUriItem> myList = missing.remove(name);
                if (myList != null) {
                    for (SingleUriItem item : myList) {
                        replacements.put(item, new SingleUriItem(f.toURI()));
                    }
                }
                if (!missing.isEmpty()) continue;
                return;
            }
            if (!f.isDirectory()) continue;
            subdirs.add(f);
        }
        for (File subdir : subdirs) {
            UriUpdater.searchDirectoriesRecursive(subdir, missing, maxDepth - 1, replacements);
        }
    }

    static class DefaultUriResource
    implements UriResource {
        private List<URI> uris;

        DefaultUriResource(URI ... uris) {
            this.uris = Arrays.asList(uris);
        }

        @Override
        public Collection<URI> getURIs() throws IOException {
            return Collections.unmodifiableList(this.uris);
        }

        @Override
        public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
            boolean changes = false;
            for (int i = 0; i < this.uris.size(); ++i) {
                URI replace;
                URI uri = this.uris.get(i);
                if (Objects.equals(uri, replace = replacements.get(uri))) continue;
                this.uris.set(i, replace);
                changes = true;
            }
            return changes;
        }
    }

    public static class SingleUriItem {
        private URI uri;
        private Path path;

        private SingleUriItem(URI uri) {
            this.uri = uri;
            this.path = GeneralTools.toPath(uri);
        }

        public UriStatus getStatus() {
            if (this.path == null && !"file".equals(this.uri.getScheme())) {
                return UriStatus.UNKNOWN;
            }
            if (this.path != null && Files.exists(this.path, new LinkOption[0])) {
                return UriStatus.EXISTS;
            }
            return UriStatus.MISSING;
        }

        public URI getURI() {
            return this.uri;
        }

        public Path getPath() {
            return this.path;
        }

        public String toString() {
            if (this.path == null) {
                return this.uri.toString();
            }
            return this.path.toString();
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.path == null ? 0 : this.path.hashCode());
            result = 31 * result + (this.uri == null ? 0 : this.uri.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            SingleUriItem other = (SingleUriItem)obj;
            if (this.path == null ? other.path != null : !this.path.equals(other.path)) {
                return false;
            }
            return !(this.uri == null ? other.uri != null : !this.uri.equals(other.uri));
        }
    }

    public static enum UriStatus {
        EXISTS,
        MISSING,
        UNKNOWN;

    }
}

