/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.io;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.Strictness;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Objects;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.precision.GeometryPrecisionReducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.GeneralTools;
import qupath.lib.io.GsonTools;
import qupath.lib.regions.ImagePlane;
import qupath.lib.roi.EllipseROI;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.interfaces.ROI;

class ROITypeAdapters {
    private static final Logger logger = LoggerFactory.getLogger(ROITypeAdapters.class);
    static ROITypeAdapter ROI_ADAPTER_INSTANCE = new ROITypeAdapter();
    static GeometryTypeAdapter GEOMETRY_ADAPTER_INSTANCE = new GeometryTypeAdapter();
    private static final Gson gson = new GsonBuilder().setStrictness(Strictness.LENIENT).create();

    ROITypeAdapters() {
    }

    static Geometry parseGeometry(JsonObject obj, GeometryFactory factory) throws IllegalArgumentException {
        if (!obj.has("type")) {
            return null;
        }
        String type = obj.get("type").getAsString();
        if (obj.has("coordinates")) {
            JsonArray coordinates = obj.getAsJsonArray("coordinates").getAsJsonArray();
            GeometryFactory parsingFactory = ROITypeAdapters.getGeometryFactoryFloat(factory == null ? GeometryTools.getDefaultFactory() : factory);
            Point geom = switch (type) {
                case "Point" -> ROITypeAdapters.parsePoint(coordinates, parsingFactory);
                case "MultiPoint" -> ROITypeAdapters.parseMultiPoint(coordinates, parsingFactory);
                case "LineString" -> ROITypeAdapters.parseLineString(coordinates, parsingFactory);
                case "MultiLineString" -> ROITypeAdapters.parseMultiLineString(coordinates, parsingFactory);
                case "Polygon" -> ROITypeAdapters.parsePolygon(coordinates, parsingFactory);
                case "MultiPolygon" -> ROITypeAdapters.parseMultiPolygon(coordinates, parsingFactory);
                case "GeometryCollection" -> ROITypeAdapters.parseGeometryCollection(obj, parsingFactory);
                default -> throw new IllegalArgumentException("No Geometry type found for object " + String.valueOf(obj));
            };
            if (factory != null && !Objects.equals(parsingFactory.getPrecisionModel(), factory.getPrecisionModel())) {
                boolean isEmpty = geom.isEmpty();
                geom = GeometryPrecisionReducer.reduce((Geometry)geom, (PrecisionModel)factory.getPrecisionModel());
                geom = factory.createGeometry((Geometry)geom);
                if (!isEmpty && geom.isEmpty()) {
                    logger.warn("Precision reduction resulted in empty geometry!");
                    if (geom.getFactory() != factory) {
                        geom = factory.createEmpty(geom.getDimension());
                    }
                }
            }
            return geom;
        }
        throw new IllegalArgumentException("Json object does not contain coordinates: " + String.valueOf(obj));
    }

    private static GeometryFactory getGeometryFactoryFloat(GeometryFactory preferredFactory) {
        if (preferredFactory.getPrecisionModel().isFloating()) {
            return preferredFactory;
        }
        return new GeometryFactory(new PrecisionModel(PrecisionModel.FLOATING), preferredFactory.getSRID(), preferredFactory.getCoordinateSequenceFactory());
    }

    static Coordinate parseCoordinate(JsonArray array) {
        double x = array.get(0).getAsDouble();
        double y = array.get(1).getAsDouble();
        if (array.size() == 2) {
            return new Coordinate(x, y);
        }
        double z = array.get(2).getAsDouble();
        return new Coordinate(x, y, z);
    }

    static Coordinate[] parseCoordinateArray(JsonArray array) {
        Coordinate[] coordinates = new Coordinate[array.size()];
        for (int i = 0; i < array.size(); ++i) {
            coordinates[i] = ROITypeAdapters.parseCoordinate(array.get(i).getAsJsonArray());
        }
        return coordinates;
    }

    static Point parsePoint(JsonArray coord, GeometryFactory factory) {
        return factory.createPoint(ROITypeAdapters.parseCoordinate(coord));
    }

    static MultiPoint parseMultiPoint(JsonArray coords, GeometryFactory factory) {
        return factory.createMultiPointFromCoords(ROITypeAdapters.parseCoordinateArray(coords));
    }

    static LineString parseLineString(JsonArray coords, GeometryFactory factory) {
        return factory.createLineString(ROITypeAdapters.parseCoordinateArray(coords));
    }

    static MultiLineString parseMultiLineString(JsonArray coords, GeometryFactory factory) {
        LineString[] lineStrings = new LineString[coords.size()];
        for (int i = 0; i < coords.size(); ++i) {
            JsonArray array = coords.get(i).getAsJsonArray();
            lineStrings[i] = factory.createLineString(ROITypeAdapters.parseCoordinateArray(array));
        }
        return factory.createMultiLineString(lineStrings);
    }

    static Polygon parsePolygon(JsonArray coords, GeometryFactory factory) {
        int n = coords.size();
        if (n == 0) {
            return factory.createPolygon();
        }
        JsonArray array = coords.get(0).getAsJsonArray();
        LinearRing shell = factory.createLinearRing(ROITypeAdapters.parseCoordinateArray(array));
        if (n == 1) {
            return factory.createPolygon(shell);
        }
        LinearRing[] holes = new LinearRing[n - 1];
        for (int i = 1; i < n; ++i) {
            array = coords.get(i).getAsJsonArray();
            holes[i - 1] = factory.createLinearRing(ROITypeAdapters.parseCoordinateArray(array));
        }
        return factory.createPolygon(shell, holes);
    }

    static MultiPolygon parseMultiPolygon(JsonArray coords, GeometryFactory factory) {
        int n = coords.size();
        Polygon[] polygons = new Polygon[n];
        for (int i = 0; i < n; ++i) {
            polygons[i] = ROITypeAdapters.parsePolygon(coords.get(i).getAsJsonArray(), factory);
        }
        return factory.createMultiPolygon(polygons);
    }

    static GeometryCollection parseGeometryCollection(JsonObject obj, GeometryFactory factory) {
        JsonArray array = obj.get("geometries").getAsJsonArray();
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        for (int i = 0; i < array.size(); ++i) {
            geometries.add(ROITypeAdapters.parseGeometry(array.get(i).getAsJsonObject(), factory));
        }
        return new GeometryCollection((Geometry[])geometries.toArray(Geometry[]::new), factory);
    }

    static void writeGeometry(Geometry geometry, JsonWriter out, int nDecimals) throws IOException {
        String type = geometry.getGeometryType();
        out.name("type");
        out.value(type);
        if ("GeometryCollection".equals(geometry.getGeometryType())) {
            out.beginArray();
            for (int i = 0; i < geometry.getNumGeometries(); ++i) {
                ROITypeAdapters.writeGeometry(geometry.getGeometryN(i), out, nDecimals);
            }
            out.endArray();
        } else {
            out.name("coordinates");
            ROITypeAdapters.writeCoordinates(geometry, out, nDecimals);
        }
    }

    static void writeCoordinates(Geometry geometry, JsonWriter out, int nDecimals) throws IOException {
        if (geometry instanceof Point) {
            ROITypeAdapters.writeCoordinates((Point)geometry, out, nDecimals);
        } else if (geometry instanceof MultiPoint) {
            ROITypeAdapters.writeCoordinates((MultiPoint)geometry, out, nDecimals);
        } else if (geometry instanceof LineString) {
            ROITypeAdapters.writeCoordinates((LineString)geometry, out, nDecimals);
        } else if (geometry instanceof MultiLineString) {
            ROITypeAdapters.writeCoordinates((Geometry)((MultiLineString)geometry), out, nDecimals);
        } else if (geometry instanceof Polygon) {
            ROITypeAdapters.writeCoordinates((Polygon)geometry, out, nDecimals);
        } else if (geometry instanceof MultiPolygon) {
            ROITypeAdapters.writeCoordinates((MultiPolygon)geometry, out, nDecimals);
        } else {
            throw new IllegalArgumentException("Unable to write coordinates for geometry type " + geometry.getGeometryType());
        }
    }

    static void writeCoordinates(Point point, JsonWriter out, int nDecimals) throws IOException {
        out.jsonValue(ROITypeAdapters.coordinateToString(point.getCoordinate(), nDecimals));
    }

    static void writeCoordinates(MultiPoint multiPoint, JsonWriter out, int nDecimals) throws IOException {
        Coordinate[] coords = multiPoint.getCoordinates();
        out.beginArray();
        for (Coordinate c : coords) {
            out.jsonValue(ROITypeAdapters.coordinateToString(c, nDecimals));
        }
        out.endArray();
    }

    static void writeCoordinates(LineString lineString, JsonWriter out, int nDecimals) throws IOException {
        Coordinate[] coords = lineString.getCoordinates();
        out.beginArray();
        for (Coordinate c : coords) {
            out.jsonValue(ROITypeAdapters.coordinateToString(c, nDecimals));
        }
        out.endArray();
    }

    static void writeCoordinates(Polygon polygon, JsonWriter out, int nDecimals) throws IOException {
        out.beginArray();
        ROITypeAdapters.writeCoordinates((LineString)polygon.getExteriorRing(), out, nDecimals);
        for (int i = 0; i < polygon.getNumInteriorRing(); ++i) {
            ROITypeAdapters.writeCoordinates((LineString)polygon.getInteriorRingN(i), out, nDecimals);
        }
        out.endArray();
    }

    static void writeCoordinates(MultiPolygon multiPolygon, JsonWriter out, int nDecimals) throws IOException {
        out.beginArray();
        for (int i = 0; i < multiPolygon.getNumGeometries(); ++i) {
            ROITypeAdapters.writeCoordinates(multiPolygon.getGeometryN(i), out, nDecimals);
        }
        out.endArray();
    }

    static void writeMultiPoint(MultiPoint multiPoint, JsonWriter out, int nDecimals) throws IOException {
        Coordinate[] coords;
        out.name("type");
        out.value("MultiPoint");
        out.name("coordinates");
        out.beginArray();
        for (Coordinate c : coords = multiPoint.getCoordinates()) {
            out.jsonValue(ROITypeAdapters.coordinateToString(c, nDecimals));
        }
        out.endArray();
    }

    static void writeLineString(LineString lineString, JsonWriter out, int nDecimals) throws IOException {
        out.name("type");
        out.value("LineString");
        out.name("coordinates");
        out.beginArray();
        CoordinateSequence coords = lineString.getCoordinateSequence();
        for (int i = 0; i < coords.size(); ++i) {
            out.jsonValue(ROITypeAdapters.coordinateToString(coords.getX(i), coords.getY(i), nDecimals));
        }
        out.endArray();
    }

    static String coordinateToString(Coordinate coord, int nDecimals) {
        return ROITypeAdapters.coordinateToString(coord.x, coord.y, nDecimals);
    }

    static String coordinateToString(double x, double y, int nDecimals) {
        return "[" + GeneralTools.formatNumber(Locale.US, x, nDecimals) + ", " + GeneralTools.formatNumber(Locale.US, y, nDecimals) + "]";
    }

    static class ROITypeAdapter
    extends TypeAdapter<ROI> {
        private int numDecimalPlaces = 2;
        private static final GeometryFactory factory = GeometryTools.getDefaultFactory();

        ROITypeAdapter() {
        }

        public void write(JsonWriter out, ROI roi) throws IOException {
            out.beginObject();
            if (roi != null) {
                Geometry geometry = roi.getGeometry();
                ROITypeAdapters.writeGeometry(geometry, out, this.numDecimalPlaces);
                ImagePlane plane = roi.getImagePlane();
                if (!ImagePlane.getDefaultPlane().equals(plane)) {
                    out.name("plane");
                    GsonTools.ImagePlaneTypeAdapter.INSTANCE.write(out, plane);
                }
                if (roi instanceof EllipseROI) {
                    out.name("isEllipse");
                    out.value(Boolean.TRUE);
                }
            }
            out.endObject();
        }

        public ROI read(JsonReader in) throws IOException {
            JsonObject obj = (JsonObject)gson.fromJson(in, JsonObject.class);
            Geometry geometry = ROITypeAdapters.parseGeometry(obj, factory);
            if (geometry == null) {
                return null;
            }
            ImagePlane plane = obj.has("plane") ? (ImagePlane)GsonTools.ImagePlaneTypeAdapter.INSTANCE.fromJsonTree(obj.get("plane")) : ImagePlane.getDefaultPlane();
            if (obj.has("isEllipse") && obj.get("isEllipse").getAsBoolean()) {
                Envelope envelope = geometry.getEnvelopeInternal();
                return ROIs.createEllipseROI(envelope.getMinX(), envelope.getMinY(), envelope.getWidth(), envelope.getHeight(), plane);
            }
            return GeometryTools.geometryToROI(geometry, plane);
        }
    }

    static class GeometryTypeAdapter
    extends TypeAdapter<Geometry> {
        private int numDecimalPlaces = 2;

        GeometryTypeAdapter() {
        }

        public void write(JsonWriter out, Geometry geometry) throws IOException {
            out.beginObject();
            ROITypeAdapters.writeGeometry(geometry, out, this.numDecimalPlaces);
            out.endObject();
        }

        public Geometry read(JsonReader in) throws IOException {
            JsonObject obj = (JsonObject)gson.fromJson(in, JsonObject.class);
            return ROITypeAdapters.parseGeometry(obj, GeometryTools.getDefaultFactory());
        }
    }
}

