/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.servers;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.lang.invoke.CallSite;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;
import org.locationtech.jts.geom.Geometry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.images.servers.AbstractTileableImageServer;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerBuilder;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.ImageServers;
import qupath.lib.images.servers.TileRequest;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.GeometryTools;

public class SparseImageServer
extends AbstractTileableImageServer {
    private static final Logger logger = LoggerFactory.getLogger(SparseImageServer.class);
    private final ImageServerMetadata metadata;
    private String path;
    private SparseImageServerManager manager;
    private transient ColorModel colorModel;
    private int originZ = 0;
    private int originT = 0;
    private int originX = 0;
    private int originY = 0;

    SparseImageServer(List<SparseImageServerManagerRegion> regions, String path) throws IOException {
        this(SparseImageServer.createManager(regions), path);
    }

    private SparseImageServer(SparseImageServerManager manager, String path) throws IOException {
        logger.trace("Creating SparseImageServer with path {}", path);
        this.manager = manager;
        ImageServerMetadata metadata = null;
        int x1 = Integer.MAX_VALUE;
        int y1 = Integer.MAX_VALUE;
        int x2 = -2147483647;
        int y2 = -2147483647;
        ArrayList<CallSite> paths = new ArrayList<CallSite>();
        int minZ = Integer.MAX_VALUE;
        int maxZ = -2147483647;
        int minT = Integer.MAX_VALUE;
        int maxT = -2147483647;
        for (ImageRegion region : manager.getRegions()) {
            if (region.getX() < x1) {
                x1 = region.getX();
            }
            if (region.getY() < y1) {
                y1 = region.getY();
            }
            if (region.getX() + region.getWidth() > x2) {
                x2 = region.getX() + region.getWidth();
            }
            if (region.getY() + region.getHeight() > y2) {
                y2 = region.getY() + region.getHeight();
            }
            if (region.getZ() < minZ) {
                minZ = region.getZ();
            }
            if (region.getZ() > maxZ) {
                maxZ = region.getZ();
            }
            if (region.getT() < minT) {
                minT = region.getT();
            }
            if (region.getT() > maxT) {
                maxT = region.getT();
            }
            ImageServer<BufferedImage> server = null;
            if (metadata != null && path != null) continue;
            server = manager.getServer(region, 1.0);
            if (metadata == null) {
                metadata = server.getMetadata();
                this.colorModel = server.getDefaultThumbnail(0, 0).getColorModel();
            }
            if (path != null) continue;
            paths.add((CallSite)((Object)(region.toString() + " (" + server.getPath() + ")")));
        }
        if (path == null) {
            path = this.getClass().getName() + ": " + String.join((CharSequence)", ", paths);
        }
        this.path = path;
        this.originX = x1;
        this.originY = y1;
        int width = x2 - x1;
        int height = y2 - y1;
        this.originZ = minZ;
        this.originT = minT;
        this.metadata = new ImageServerMetadata.Builder(metadata).name("Sparse image (" + manager.getRegions().size() + " regions)").sizeZ(maxZ - minZ + 1).sizeT(maxT - minT + 1).width(width).height(height).preferredTileSize(1024, 1024).levelsFromDownsamples(manager.getAvailableDownsamples()).build();
    }

    @Override
    protected ColorModel getDefaultColorModel() {
        return this.colorModel;
    }

    @Override
    public Collection<URI> getURIs() {
        LinkedHashSet<URI> uris = new LinkedHashSet<URI>();
        for (ImageServerBuilder.ServerBuilder<BufferedImage> builder : this.manager.serverMap.keySet()) {
            uris.addAll(builder.getURIs());
        }
        return uris;
    }

    @Override
    protected String createID() {
        return this.path;
    }

    public SparseImageServerManager getManager() {
        return this.manager;
    }

    @Override
    public String getServerType() {
        return "Sparse image server";
    }

    @Override
    public ImageServerMetadata getOriginalMetadata() {
        return this.metadata;
    }

    @Override
    protected ImageServerBuilder.ServerBuilder<BufferedImage> createServerBuilder() {
        ArrayList<SparseImageServerManagerRegion> resolutions = new ArrayList<SparseImageServerManagerRegion>();
        for (Map.Entry<ImageRegion, List<SparseImageServerManagerResolution>> entry : this.manager.regionMap.entrySet()) {
            resolutions.add(new SparseImageServerManagerRegion(entry.getKey(), entry.getValue()));
        }
        return new ImageServers.SparseImageServerBuilder(this.getMetadata(), resolutions, this.getPath());
    }

    @Override
    public void close() throws Exception {
        this.manager.close();
        super.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected BufferedImage readTile(TileRequest tileRequest) throws IOException {
        WritableRaster raster = null;
        for (ImageRegion subRegion : this.manager.getRegions()) {
            if (subRegion.getZ() != tileRequest.getZ() + this.originZ || subRegion.getT() != tileRequest.getT() + this.originT) continue;
            double downsample = tileRequest.getRegionRequest().getDownsample();
            if (!subRegion.intersects(tileRequest.getImageX() + this.originX, tileRequest.getImageY() + this.originY, tileRequest.getImageWidth(), tileRequest.getImageHeight())) continue;
            ImageServer<BufferedImage> serverTemp = this.manager.getServer(subRegion, downsample);
            int x1 = Math.max(tileRequest.getImageX() + this.originX, subRegion.getX());
            int y1 = Math.max(tileRequest.getImageY() + this.originY, subRegion.getY());
            int x2 = Math.min(tileRequest.getImageX() + this.originX + tileRequest.getImageWidth(), subRegion.getX() + subRegion.getWidth());
            int y2 = Math.min(tileRequest.getImageY() + this.originY + tileRequest.getImageHeight(), subRegion.getY() + subRegion.getHeight());
            int xr = x1 - subRegion.getX();
            int yr = y1 - subRegion.getY();
            int xr2 = x2 - subRegion.getX();
            int yr2 = y2 - subRegion.getY();
            double requestDownsample = downsample;
            RegionRequest requestTemp = RegionRequest.createInstance(serverTemp.getPath(), requestDownsample, xr, yr, xr2 - xr, yr2 - yr, tileRequest.getZ() + this.originZ, tileRequest.getT() + this.originT);
            BufferedImage imgTemp = null;
            ImageServer<BufferedImage> imageServer = serverTemp;
            synchronized (imageServer) {
                imgTemp = serverTemp.readRegion(requestTemp);
            }
            if (imgTemp == null) continue;
            if (raster == null) {
                raster = imgTemp.getRaster().createCompatibleWritableRaster(tileRequest.getTileWidth(), tileRequest.getTileHeight());
            }
            int x = (int)Math.round((double)(x1 - tileRequest.getImageX() - this.originX) / downsample);
            int y = (int)Math.round((double)(y1 - tileRequest.getImageY() - this.originY) / downsample);
            int w = Math.min(imgTemp.getWidth(), raster.getWidth() - x);
            int h = Math.min(imgTemp.getHeight(), raster.getHeight() - y);
            raster.setDataElements(x, y, w, h, imgTemp.getRaster().getDataElements(0, 0, w, h, null));
        }
        if (raster == null) {
            return this.getEmptyTile(tileRequest.getTileWidth(), tileRequest.getTileHeight(), true);
        }
        return new BufferedImage(this.colorModel, raster, false, null);
    }

    List<SparseImageServerManagerRegion> getRegions() {
        ArrayList<SparseImageServerManagerRegion> regions = new ArrayList<SparseImageServerManagerRegion>();
        for (Map.Entry<ImageRegion, List<SparseImageServerManagerResolution>> entry : this.manager.regionMap.entrySet()) {
            regions.add(new SparseImageServerManagerRegion(entry.getKey(), entry.getValue()));
        }
        return regions;
    }

    public static List<SparseImageServer> splitConnectedRegions(SparseImageServer server, double distancePixels, boolean permitBoundsOverlap) throws IOException {
        int i;
        if (distancePixels < 0.0) {
            throw new IllegalArgumentException("Minimum pixel distance must be >= 0");
        }
        List<SparseImageServerManagerRegion> regions = server.getRegions();
        LinkedHashMap<SparseImageServerManagerRegion, Geometry> map = new LinkedHashMap<SparseImageServerManagerRegion, Geometry>();
        for (SparseImageServerManagerRegion region : regions) {
            map.put(region, GeometryTools.regionToGeometry(region.getRegion()));
        }
        Geometry allGeometries = GeometryTools.union(map.values());
        if (distancePixels > 0.0) {
            allGeometries = allGeometries.buffer(distancePixels);
        }
        if (allGeometries.getNumGeometries() > 1 && !permitBoundsOverlap) {
            ArrayList<Geometry> bounds = new ArrayList<Geometry>();
            for (i = 0; i < allGeometries.getNumGeometries(); ++i) {
                bounds.add(allGeometries.getGeometryN(i).getEnvelope());
            }
            allGeometries = GeometryTools.union(bounds);
        }
        if (allGeometries.getNumGeometries() == 1) {
            return Collections.singletonList(server);
        }
        LinkedHashMap splitMap = new LinkedHashMap();
        for (i = 0; i < allGeometries.getNumGeometries(); ++i) {
            splitMap.put(allGeometries.getGeometryN(i), new ArrayList());
        }
        for (Map.Entry entry : map.entrySet()) {
            SparseImageServerManagerRegion region = (SparseImageServerManagerRegion)entry.getKey();
            Geometry geom = (Geometry)entry.getValue();
            for (Map.Entry entrySplit : splitMap.entrySet()) {
                if (!((Geometry)entrySplit.getKey()).covers(geom)) continue;
                ((List)entrySplit.getValue()).add(region);
            }
        }
        ArrayList<SparseImageServer> servers = new ArrayList<SparseImageServer>();
        for (List list : splitMap.values()) {
            servers.add(new SparseImageServer(list, UUID.randomUUID().toString()));
        }
        return servers;
    }

    static SparseImageServerManager createManager(List<SparseImageServerManagerRegion> regions) {
        SparseImageServerManager manager = new SparseImageServerManager();
        for (SparseImageServerManagerRegion region : regions) {
            for (SparseImageServerManagerResolution resolution : region.resolutions) {
                manager.addRegionServer(region.region, resolution.getDownsample(), resolution.getServerBuilder());
            }
        }
        return manager;
    }

    public static class SparseImageServerManager
    implements AutoCloseable {
        private Map<ImageRegion, List<SparseImageServerManagerResolution>> regionMap = new LinkedHashMap<ImageRegion, List<SparseImageServerManagerResolution>>();
        private Set<Double> downsamples = new TreeSet<Double>();
        private transient List<ImageRegion> regionList;
        private transient Map<ImageServerBuilder.ServerBuilder<BufferedImage>, ImageServer<BufferedImage>> serverMap = new HashMap<ImageServerBuilder.ServerBuilder<BufferedImage>, ImageServer<BufferedImage>>();

        private synchronized void addRegionServer(ImageRegion region, double downsample, ImageServerBuilder.ServerBuilder<BufferedImage> serverBuilder) {
            int ind;
            this.resetCaches();
            List<SparseImageServerManagerResolution> resolutions = this.regionMap.get(region);
            if (resolutions == null) {
                resolutions = new ArrayList<SparseImageServerManagerResolution>();
                this.regionMap.put(region, resolutions);
            }
            this.downsamples.add(downsample);
            for (ind = 0; ind < resolutions.size() && downsample > resolutions.get(ind).getDownsample(); ++ind) {
            }
            resolutions.add(ind, new SparseImageServerManagerResolution(serverBuilder, downsample));
        }

        private synchronized void addRegionServer(ImageRegion region, double downsample, ImageServer<BufferedImage> server) {
            ImageServerBuilder.ServerBuilder<BufferedImage> builder = server.getBuilder();
            if (!this.serverMap.containsKey(builder)) {
                this.serverMap.put(builder, server);
            }
            this.addRegionServer(region, downsample, builder);
        }

        private void resetCaches() {
            this.regionList = null;
        }

        public synchronized Collection<ImageRegion> getRegions() {
            if (this.regionList == null) {
                this.regionList = new ArrayList<ImageRegion>(this.regionMap.keySet());
                this.regionList = Collections.unmodifiableList(this.regionList);
            }
            return this.regionList;
        }

        public synchronized ImageServer<BufferedImage> getServer(ImageRegion region, double downsample) throws IOException {
            int level;
            List<SparseImageServerManagerResolution> resolutions = this.regionMap.get(region);
            if (resolutions == null || resolutions.isEmpty()) {
                return null;
            }
            for (level = resolutions.size() - 1; level > 0 && resolutions.get(level).getDownsample() > downsample; --level) {
            }
            ImageServerBuilder.ServerBuilder<BufferedImage> builder = resolutions.get(level).getServerBuilder();
            ImageServer<BufferedImage> server = this.serverMap.get(builder);
            if (server == null) {
                try {
                    server = builder.build();
                }
                catch (IOException e) {
                    throw e;
                }
                catch (Exception e) {
                    throw new IOException(e);
                }
                this.serverMap.put(builder, server);
            }
            return server;
        }

        @Override
        public void close() throws Exception {
            for (ImageServer<BufferedImage> server : this.serverMap.values()) {
                server.close();
            }
        }

        double[] getAvailableDownsamples() {
            return this.downsamples.stream().mapToDouble(d -> d).toArray();
        }
    }

    static class SparseImageServerManagerRegion {
        private ImageRegion region;
        private List<SparseImageServerManagerResolution> resolutions;

        SparseImageServerManagerRegion(ImageRegion region, List<SparseImageServerManagerResolution> resolutions) {
            this.region = region;
            this.resolutions = resolutions;
        }

        public ImageRegion getRegion() {
            return this.region;
        }

        public List<SparseImageServerManagerResolution> getResolutions() {
            return Collections.unmodifiableList(this.resolutions);
        }
    }

    static class SparseImageServerManagerResolution {
        private final double downsample;
        private final ImageServerBuilder.ServerBuilder<BufferedImage> serverBuilder;

        SparseImageServerManagerResolution(ImageServerBuilder.ServerBuilder<BufferedImage> serverBuilder, double downsample) {
            this.serverBuilder = serverBuilder;
            this.downsample = downsample;
        }

        ImageServerBuilder.ServerBuilder<BufferedImage> getServerBuilder() {
            return this.serverBuilder;
        }

        double getDownsample() {
            return this.downsample;
        }
    }

    public static class Builder {
        private SparseImageServerManager manager = new SparseImageServerManager();

        public synchronized Builder jsonRegion(ImageRegion region, double downsample, ImageServerBuilder.ServerBuilder<BufferedImage> builder) {
            this.manager.addRegionServer(region, downsample, builder);
            return this;
        }

        public synchronized Builder serverRegion(ImageRegion region, double downsample, ImageServer<BufferedImage> server) {
            this.manager.addRegionServer(region, downsample, server);
            return this;
        }

        public SparseImageServer build() throws IOException {
            return new SparseImageServer(this.manager, null);
        }
    }
}

