/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.LongStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.GeneralTools;

public class Timeit {
    private static final Logger logger = LoggerFactory.getLogger(Timeit.class);
    private static final String DEFAULT_END_NAME = "END";
    private TimeUnit unit;
    private int maxDecimals = 3;
    private boolean isStarted = false;
    private boolean summarizeCheckpoints = false;
    private List<Checkpoint> checkpoints = Collections.synchronizedList(new ArrayList());

    public Timeit start() throws UnsupportedOperationException {
        return this.start(null);
    }

    public Timeit start(String name) throws UnsupportedOperationException {
        if (this.isStarted) {
            throw new UnsupportedOperationException("Timeit has already been started!");
        }
        return this.checkpoint(name);
    }

    public Timeit checkpointAndRun(Runnable runnable) throws UnsupportedOperationException {
        return this.checkpointAndRun(null, runnable, 1);
    }

    public Timeit checkpointAndRun(String name, Runnable runnable) throws UnsupportedOperationException {
        return this.checkpointAndRun(name, runnable, 1);
    }

    public Timeit checkpointAndRun(String name, Runnable runnable, int nIterations) throws UnsupportedOperationException {
        if (nIterations <= 0) {
            return this;
        }
        for (int i = 0; i < nIterations; ++i) {
            String name2 = null;
            if (name != null) {
                name2 = nIterations == 1 ? name : name + " (" + (i + 1) + ")";
            }
            this.checkpoint(name2);
            runnable.run();
        }
        return this;
    }

    public Timeit checkpoint() throws UnsupportedOperationException {
        return this.checkpoint(null);
    }

    public Timeit checkpoint(String name) throws UnsupportedOperationException {
        if (!this.isStarted && !this.checkpoints.isEmpty()) {
            throw new UnsupportedOperationException("Timeit has already been stopped!");
        }
        if (name == null) {
            name = "Checkpoint " + (this.checkpoints.size() + 1);
        }
        if (!this.isStarted) {
            this.isStarted = true;
            logger.debug("Timeit now started with checkpoint {}", name);
        }
        this.checkpoints.add(new Checkpoint((String)name));
        if (DEFAULT_END_NAME.equals(name)) {
            this.isStarted = false;
            logger.debug("Timeit now stopped");
        }
        return this;
    }

    public Timeit stop() throws UnsupportedOperationException {
        if (!this.isStarted) {
            throw new UnsupportedOperationException("Timeit has already been stopped!");
        }
        return this.checkpoint(DEFAULT_END_NAME);
    }

    private Timeit setUnit(TimeUnit unit) {
        this.unit = unit;
        return this;
    }

    public Timeit autoUnits() {
        return this.setUnit(null);
    }

    public Timeit nanoseconds() {
        return this.setUnit(TimeUnit.NANOSECONDS);
    }

    public Timeit milliseconds() {
        return this.setUnit(TimeUnit.MILLISECONDS);
    }

    public Timeit microseconds() {
        return this.setUnit(TimeUnit.MICROSECONDS);
    }

    public Timeit seconds() {
        return this.setUnit(TimeUnit.SECONDS);
    }

    public Timeit minutes() {
        return this.setUnit(TimeUnit.MINUTES);
    }

    public Timeit summarizeCheckpoints() {
        return this.summarizeCheckpoints(true);
    }

    public Timeit summarizeCheckpoints(boolean summarize) {
        this.summarizeCheckpoints = summarize;
        return this;
    }

    private long getStartTime(List<Checkpoint> checkpoints) {
        return checkpoints.get(0).getNanoseconds();
    }

    private static Checkpoint getEndCheckpoint(List<Checkpoint> checkpoints) {
        if (checkpoints.isEmpty()) {
            return null;
        }
        Checkpoint lastCheckpoint = checkpoints.get(checkpoints.size() - 1);
        if (Objects.equals(DEFAULT_END_NAME, lastCheckpoint.getName())) {
            return lastCheckpoint;
        }
        return null;
    }

    public int getMaxDecimalPlaces() {
        return this.maxDecimals;
    }

    public Timeit maxDecimalPlaces(int maxDP) {
        this.maxDecimals = maxDP;
        return this;
    }

    public List<Checkpoint> getCheckpoints() {
        return new ArrayList<Checkpoint>(this.checkpoints);
    }

    public String toString() {
        if (this.checkpoints.isEmpty()) {
            return "No TimeIt information available";
        }
        ArrayList<Checkpoint> checkpoints = new ArrayList<Checkpoint>(this.checkpoints);
        long startTime = this.getStartTime(checkpoints);
        Checkpoint endCheckpoint = Timeit.getEndCheckpoint(checkpoints);
        long endTime = endCheckpoint == null ? System.nanoTime() : endCheckpoint.getNanoseconds();
        long totalDuration = endTime - startTime;
        TimeUnit timeUnit = this.unit;
        if (timeUnit == null) {
            timeUnit = Timeit.chooseAutoTimeUnit(totalDuration);
        }
        if (checkpoints.size() == 1) {
            return "Time since start\t" + Timeit.getDurationString(totalDuration, timeUnit, this.maxDecimals);
        }
        StringBuilder sb = new StringBuilder();
        long[] durations = new long[checkpoints.size() - 1];
        if (checkpoints.size() > 2 || endCheckpoint == null) {
            for (int i = 0; i < checkpoints.size() - 1; ++i) {
                long duration;
                long start = checkpoints.get((int)i).nano;
                long end = checkpoints.get((int)(i + 1)).nano;
                durations[i] = duration = end - start;
                sb.append(checkpoints.get((int)i).name);
                sb.append("\t");
                sb.append(Timeit.getDurationString(duration, timeUnit, this.maxDecimals));
                sb.append("\n");
            }
        }
        sb.append("Total duration\t" + Timeit.getDurationString(totalDuration, timeUnit, this.maxDecimals));
        if (this.summarizeCheckpoints) {
            double average = LongStream.of(durations).summaryStatistics().getAverage();
            sb.append("\nAverage per checkpoint: " + Timeit.getDurationString(Math.round(average), timeUnit, this.maxDecimals));
        }
        return sb.toString();
    }

    private static TimeUnit chooseAutoTimeUnit(long nanos) {
        TimeUnit t = TimeUnit.NANOSECONDS;
        if (t.toMinutes(nanos) >= 10L) {
            return TimeUnit.MINUTES;
        }
        if (t.toSeconds(nanos) >= 1L) {
            return TimeUnit.SECONDS;
        }
        if (t.toMillis(nanos) >= 10L) {
            return TimeUnit.MILLISECONDS;
        }
        if (t.toMicros(nanos) >= 10L) {
            return TimeUnit.MICROSECONDS;
        }
        return TimeUnit.NANOSECONDS;
    }

    private static String getDurationString(long nanos, TimeUnit unit, int maxDecimals) {
        switch (unit) {
            case MICROSECONDS: {
                return Timeit.toMicrosString(nanos, 0);
            }
            case MILLISECONDS: {
                return Timeit.toMillisString(nanos, 0);
            }
            case MINUTES: {
                return Timeit.toMinutesString(nanos, maxDecimals);
            }
            case NANOSECONDS: {
                return Timeit.toNanoString(nanos, 0);
            }
            case SECONDS: {
                return Timeit.toSecondsString(nanos, maxDecimals);
            }
        }
        throw new UnsupportedOperationException("Unsupported time unit " + String.valueOf((Object)unit));
    }

    private static String toMinutesString(long nanos, int nDecimals) {
        long seconds = TimeUnit.NANOSECONDS.toSeconds(nanos);
        return GeneralTools.formatNumber((double)seconds / 60.0, nDecimals) + " minutes";
    }

    private static String toSecondsString(long nanos, int nDecimals) {
        return GeneralTools.formatNumber((double)nanos / 1.0E9, nDecimals) + " s";
    }

    private static String toMillisString(long nanos, int nDecimals) {
        return GeneralTools.formatNumber((double)nanos / 1000000.0, nDecimals) + " ms";
    }

    private static String toMicrosString(long nanos, int nDecimals) {
        return GeneralTools.formatNumber((double)nanos / 1000.0, nDecimals) + " \u00b5s";
    }

    private static String toNanoString(long nanos, int nDecimals) {
        return nanos + " ns";
    }

    public static void main(String[] args) {
        try {
            Timeit timeit = new Timeit().microseconds().checkpointAndRun("Greeting", () -> {
                logger.info("Hello!");
                try {
                    Thread.sleep(10L);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }, 10).summarizeCheckpoints().stop();
            Thread.sleep(100L);
            logger.info("{}", (Object)timeit);
        }
        catch (Exception e) {
            logger.error(e.getMessage(), (Throwable)e);
        }
    }

    public static class Checkpoint {
        private String name;
        private long nano;

        private Checkpoint(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            this.nano = System.nanoTime();
        }

        public String getName() {
            return this.name;
        }

        public long getNanoseconds() {
            return this.nano;
        }

        public String toString() {
            return "Checkpoint [name=" + this.name + ", nano=" + this.nano + "]";
        }
    }
}

