/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.analysis.images;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.Predicate;
import org.locationtech.jts.dissolve.LineDissolver;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.simplify.DouglasPeuckerSimplifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.images.CoordinatePair;
import qupath.lib.analysis.images.IntPoint;

class ContourTracingUtils {
    private static final Logger logger = LoggerFactory.getLogger(ContourTracingUtils.class);

    ContourTracingUtils() {
    }

    static Collection<CoordinatePair> removeDuplicatesCompletely(Collection<CoordinatePair> lines) {
        CoordinatePair lastPair = null;
        ArrayList<CoordinatePair> pairs = new ArrayList<CoordinatePair>();
        boolean duplicate = false;
        Iterator iterator = lines.stream().sorted().iterator();
        while (iterator.hasNext()) {
            CoordinatePair line = (CoordinatePair)iterator.next();
            if (Objects.equals(lastPair, line)) {
                duplicate = true;
            } else {
                if (!duplicate && lastPair != null) {
                    pairs.add(lastPair);
                }
                duplicate = false;
            }
            lastPair = line;
        }
        if (!duplicate) {
            pairs.add(lastPair);
        }
        return pairs;
    }

    static Geometry linesFromPairsLegacy(GeometryFactory factory, Collection<CoordinatePair> pairs, double xOrigin, double yOrigin, double scale) {
        LineDissolver dissolver = new LineDissolver();
        for (CoordinatePair p : pairs) {
            dissolver.add((Geometry)ContourTracingUtils.createLineString(p, factory, xOrigin, yOrigin, scale));
        }
        Geometry lineStrings = dissolver.getResult();
        return DouglasPeuckerSimplifier.simplify((Geometry)lineStrings, (double)0.0);
    }

    private static LineString createLineString(CoordinatePair pair, GeometryFactory factory, double xOrigin, double yOrigin, double scale) {
        PrecisionModel pm = factory.getPrecisionModel();
        IntPoint c1 = pair.c1();
        IntPoint c2 = pair.c2();
        double x1 = pm.makePrecise(xOrigin + (double)c1.getX() * scale);
        double x2 = pm.makePrecise(xOrigin + (double)c2.getX() * scale);
        double y1 = pm.makePrecise(yOrigin + (double)c1.getY() * scale);
        double y2 = pm.makePrecise(yOrigin + (double)c2.getY() * scale);
        return factory.createLineString(new Coordinate[]{new Coordinate(x1, y1), new Coordinate(x2, y2)});
    }

    private static MinimalCoordinateSet findNonMergeableCoordinates(Collection<CoordinatePair> pairList) {
        ArrayList<IntPoint> allCoordinates = new ArrayList<IntPoint>(pairList.size() * 2);
        for (CoordinatePair p : pairList) {
            allCoordinates.add(p.c1());
            allCoordinates.add(p.c2());
        }
        allCoordinates.sort(null);
        int count = 0;
        IntPoint currentCoord = null;
        long startTime = System.currentTimeMillis();
        MinimalCoordinateSet nonMergeable = new MinimalCoordinateSet(allCoordinates.size() / 10);
        for (IntPoint c : allCoordinates) {
            if (Objects.equals(currentCoord, c)) {
                ++count;
                continue;
            }
            if (count != 2 && currentCoord != null) {
                nonMergeable.add(currentCoord);
            }
            currentCoord = c;
            count = 1;
        }
        nonMergeable.rebuild();
        long endTime = System.currentTimeMillis();
        logger.trace("Time to find non-mergeable coordinates: " + (endTime - startTime) + " ms");
        return nonMergeable;
    }

    static Geometry linesFromPairsFast(GeometryFactory factory, Collection<CoordinatePair> pairs, double xOrigin, double yOrigin, double scale) throws InterruptedException {
        IntPoint c2;
        IntPoint c1;
        List<CoordinatePair> pairList = pairs.stream().sorted().toList();
        MinimalCoordinateSet nonMergeable = ContourTracingUtils.findNonMergeableCoordinates(pairs);
        TreeMap<Integer, List> horizontal = new TreeMap<Integer, List>();
        TreeMap<Integer, List> vertical = new TreeMap<Integer, List>();
        for (CoordinatePair p : pairList) {
            if (p.isHorizontal()) {
                horizontal.computeIfAbsent(p.c1().getY(), y -> new ArrayList()).add(p);
                continue;
            }
            if (!p.isVertical()) continue;
            vertical.computeIfAbsent(p.c2().getX(), x -> new ArrayList()).add(p);
        }
        TreeMap<Integer, List> firstDirection = horizontal.size() <= vertical.size() ? horizontal : vertical;
        TreeMap<Integer, List> secondDirection = firstDirection == horizontal ? vertical : horizontal;
        ArrayList<List<Object>> lines = new ArrayList<List<Object>>(pairs.size() / 10);
        MinimalCoordinateMap<List> mergeable = new MinimalCoordinateMap<List>(Math.max(100, (int)Math.sqrt(pairList.size())));
        for (Map.Entry entry : firstDirection.entrySet()) {
            List list = (List)entry.getValue();
            for (List<IntPoint> list2 : ContourTracingUtils.buildLineStrings(list, nonMergeable::contains)) {
                c1 = list2.getFirst();
                c2 = list2.getLast();
                boolean isMergeable = false;
                if (!nonMergeable.contains(c1)) {
                    if (mergeable.put(c1, list2) != null) {
                        throw new RuntimeException("Unexpected mergeable already exists");
                    }
                    isMergeable = true;
                }
                if (!nonMergeable.contains(c2)) {
                    if (mergeable.put(c2, list2) != null) {
                        throw new RuntimeException("Unexpected mergeable already exists");
                    }
                    isMergeable = true;
                }
                if (isMergeable) continue;
                lines.add(list2);
            }
        }
        ArrayDeque<List<IntPoint>> queued = new ArrayDeque<List<IntPoint>>();
        for (Map.Entry entry : secondDirection.entrySet()) {
            List list = (List)entry.getValue();
            queued.addAll(ContourTracingUtils.buildLineStrings(list, nonMergeable::contains));
            while (!queued.isEmpty()) {
                List existing;
                boolean c2Mergeable;
                List list3 = (List)queued.pop();
                if (ContourTracingUtils.isClosed(list3)) {
                    lines.add(list3);
                    continue;
                }
                if (Thread.interrupted()) {
                    throw new InterruptedException("Contour tracing interrupted!");
                }
                c1 = (IntPoint)list3.getFirst();
                c2 = (IntPoint)list3.getLast();
                boolean c1Mergeable = !nonMergeable.contains(c1);
                boolean bl = c2Mergeable = !nonMergeable.contains(c2);
                if (c1Mergeable && (existing = (List)mergeable.remove(c1)) != null) {
                    if (c1.equals(existing.getFirst())) {
                        mergeable.remove((IntPoint)existing.getLast());
                    } else {
                        mergeable.remove((IntPoint)existing.getFirst());
                    }
                    queued.add(ContourTracingUtils.mergeLines(existing, list3));
                    continue;
                }
                if (c2Mergeable && (existing = (List)mergeable.remove(c2)) != null) {
                    if (c2.equals(existing.getFirst())) {
                        mergeable.remove((IntPoint)existing.getLast());
                    } else {
                        mergeable.remove((IntPoint)existing.getFirst());
                    }
                    queued.add(ContourTracingUtils.mergeLines(existing, list3));
                    continue;
                }
                if (c1Mergeable || c2Mergeable) {
                    if (c1Mergeable) {
                        mergeable.put(c1, list3);
                    }
                    if (!c2Mergeable) continue;
                    mergeable.put(c2, list3);
                    continue;
                }
                lines.add(list3);
            }
        }
        if (!mergeable.isEmpty()) {
            logger.warn("Remaining mergeable lines: {}", (Object)mergeable.size());
        }
        lines.addAll(mergeable.values());
        ArrayList<LineString> arrayList = new ArrayList<LineString>();
        PrecisionModel precisionModel = factory.getPrecisionModel();
        for (List<IntPoint> list : lines) {
            Coordinate[] coords = new Coordinate[list.size()];
            for (int i = 0; i < list.size(); ++i) {
                IntPoint c = list.get(i);
                double x2 = precisionModel.makePrecise(xOrigin + (double)c.getX() * scale);
                double y2 = precisionModel.makePrecise(yOrigin + (double)c.getY() * scale);
                coords[i] = new Coordinate(x2, y2);
            }
            arrayList.add(factory.createLineString(coords));
        }
        return factory.buildGeometry(arrayList);
    }

    private static boolean isClosed(List<IntPoint> coords) {
        return coords.size() > 2 && coords.getFirst().equals(coords.getLast());
    }

    private static List<IntPoint> mergeLines(List<IntPoint> l1, List<IntPoint> l2) {
        IntPoint c1Start = l1.getFirst();
        IntPoint c1End = l1.getLast();
        IntPoint c2Start = l2.getFirst();
        IntPoint c2End = l2.getLast();
        if (c1End.equals(c2Start)) {
            return ContourTracingUtils.concat(l1, l2);
        }
        if (c1Start.equals(c2End)) {
            return ContourTracingUtils.concat(l2, l1);
        }
        if (c1Start.equals(c2Start)) {
            return ContourTracingUtils.concat((List<IntPoint>)l1.reversed(), l2);
        }
        if (c1End.equals(c2End)) {
            return ContourTracingUtils.concat(l1, (List<IntPoint>)l2.reversed());
        }
        throw new IllegalArgumentException("Lines are not mergeable");
    }

    private static List<IntPoint> concat(List<IntPoint> l1, List<IntPoint> l2) {
        int n = l1.size() + l2.size() - 1;
        if (l1 instanceof ArrayList) {
            ArrayList list = (ArrayList)l1;
            list.addAll(l2);
            return list;
        }
        if (l2 instanceof ArrayList) {
            ArrayList list = (ArrayList)l2;
            list.addAll(0, l1);
            return list;
        }
        ArrayList<IntPoint> list = new ArrayList<IntPoint>(n);
        list.addAll(l1);
        list.addAll(l2.subList(1, l2.size()));
        return list;
    }

    private static List<List<IntPoint>> buildLineStrings(List<CoordinatePair> pairs, Predicate<IntPoint> counter) {
        if (pairs.isEmpty()) {
            return List.of();
        }
        ArrayList<List<IntPoint>> lines = new ArrayList<List<IntPoint>>();
        IntPoint firstCoord = pairs.getFirst().c1();
        IntPoint secondCoord = pairs.getFirst().c2();
        for (int i = 1; i < pairs.size(); ++i) {
            CoordinatePair p = pairs.get(i);
            if (!secondCoord.equals(p.c1()) || counter.test(secondCoord)) {
                lines.add(ContourTracingUtils.createLineString(firstCoord, secondCoord));
                firstCoord = p.c1();
                secondCoord = p.c2();
                continue;
            }
            secondCoord = p.c2();
        }
        lines.add(ContourTracingUtils.createLineString(firstCoord, secondCoord));
        return lines;
    }

    private static List<IntPoint> createLineString(IntPoint c1, IntPoint c2) {
        return List.of(c1, c2);
    }

    private static class MinimalCoordinateSet {
        private Set<IntPoint> set;
        private IntPoint lastContained;
        private IntPoint lastNotContained;
        private long[] values;

        MinimalCoordinateSet(int numElements) {
            this.set = HashSet.newHashSet(numElements);
        }

        boolean add(IntPoint p) {
            this.values = null;
            return this.set.add(p);
        }

        void rebuild() {
            if (this.set.size() > 1000000) {
                this.values = this.set.stream().mapToLong(IntPoint::value).sorted().toArray();
            }
        }

        boolean contains(IntPoint p) {
            if (Objects.equals(this.lastContained, p)) {
                return true;
            }
            if (Objects.equals(this.lastNotContained, p)) {
                return false;
            }
            if (this.values != null) {
                return Arrays.binarySearch(this.values, p.value()) >= 0;
            }
            if (this.set.contains(p)) {
                this.lastContained = p;
                return true;
            }
            this.lastNotContained = p;
            return false;
        }
    }

    private static class MinimalCoordinateMap<T> {
        private final int numMappings;
        private final Map<Integer, Map<Integer, T>> map;

        MinimalCoordinateMap(int numMappings) {
            this.numMappings = numMappings;
            this.map = HashMap.newHashMap(numMappings);
        }

        T put(IntPoint c, T val) {
            return this.map.computeIfAbsent(c.getX(), x -> HashMap.newHashMap(this.numMappings)).put(c.getY(), val);
        }

        T remove(IntPoint c) {
            Map temp = this.map.getOrDefault(c.getX(), null);
            if (temp != null) {
                Object value = temp.remove(c.getY());
                if (temp.isEmpty()) {
                    this.map.remove(c.getX());
                }
                return (T)value;
            }
            return null;
        }

        T get(IntPoint c) {
            return this.getOrDefault(c, null);
        }

        T getOrDefault(IntPoint c, T defaultValue) {
            return (T)this.map.getOrDefault(c.getX(), Collections.emptyMap()).getOrDefault(c.getY(), defaultValue);
        }

        Collection<T> values() {
            if (this.isEmpty()) {
                return Collections.emptyList();
            }
            return this.map.values().stream().flatMap(m -> m.values().stream()).toList();
        }

        int size() {
            return this.map.values().stream().mapToInt(Map::size).sum();
        }

        boolean isEmpty() {
            return this.map.values().stream().allMatch(Map::isEmpty);
        }
    }
}

