/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.analysis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.locationtech.jts.algorithm.distance.DistanceToPoint;
import org.locationtech.jts.algorithm.distance.PointPairDistance;
import org.locationtech.jts.algorithm.locate.IndexedPointInAreaLocator;
import org.locationtech.jts.algorithm.locate.PointOnGeometryLocator;
import org.locationtech.jts.algorithm.locate.SimplePointInAreaLocator;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Lineal;
import org.locationtech.jts.geom.Polygonal;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.geom.Puntal;
import org.locationtech.jts.geom.util.AffineTransformation;
import org.locationtech.jts.geom.util.GeometryCombiner;
import org.locationtech.jts.index.strtree.ItemBoundable;
import org.locationtech.jts.index.strtree.ItemDistance;
import org.locationtech.jts.index.strtree.STRtree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.objects.classes.PathClassTools;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.interfaces.ROI;

public class DistanceTools {
    private static final Logger logger = LoggerFactory.getLogger(DistanceTools.class);

    public static void detectionToAnnotationDistances(ImageData<?> imageData, boolean splitClassNames) {
        DistanceTools.detectionToAnnotationDistances(imageData, splitClassNames, false);
    }

    public static void detectionToAnnotationDistancesSigned(ImageData<?> imageData, boolean splitClassNames) {
        DistanceTools.detectionToAnnotationDistances(imageData, splitClassNames, true);
    }

    public static void detectionToAnnotationDistances(ImageData<?> imageData, boolean splitClassNames, boolean signedDistances) {
        ImageServer<?> server = imageData.getServer();
        PathObjectHierarchy hierarchy = imageData.getHierarchy();
        Collection<PathObject> annotations = hierarchy.getAnnotationObjects();
        Collection<PathObject> detections = hierarchy.getCellObjects();
        if (detections.isEmpty()) {
            detections = hierarchy.getDetectionObjects();
        }
        if (hierarchy.getTMAGrid() != null) {
            logger.warn("Detection to annotation distances command currently ignores TMA grid information!");
        }
        Set pathClasses = annotations.stream().map(p -> p.getPathClass()).filter(p -> p != null && p.isValid() && !PathClassTools.isIgnoredClass(p)).collect(Collectors.toSet());
        PixelCalibration cal = server.getPixelCalibration();
        String distanceString = signedDistances ? "Signed distance" : "Distance";
        String xUnit = cal.getPixelWidthUnit();
        String yUnit = cal.getPixelHeightUnit();
        double pixelWidth = cal.getPixelWidth().doubleValue();
        double pixelHeight = cal.getPixelHeight().doubleValue();
        if (!xUnit.equals(yUnit)) {
            throw new IllegalArgumentException("Pixel width & height units do not match! Width " + xUnit + ", height " + yUnit);
        }
        String unit = xUnit;
        for (PathClass pathClass : pathClasses) {
            if (splitClassNames) {
                List<String> names = PathClassTools.splitNames(pathClass);
                for (String name : names) {
                    logger.debug("Computing distances for {}", (Object)pathClass);
                    List<PathObject> filteredAnnotations = annotations.stream().filter(a -> PathClassTools.containsName(a.getPathClass(), name)).toList();
                    if (filteredAnnotations.isEmpty()) continue;
                    String measurementName = distanceString + " to annotation with " + name + " " + unit;
                    DistanceTools.centroidToBoundsDistance2D(detections, filteredAnnotations, pixelWidth, pixelHeight, measurementName, signedDistances);
                }
                continue;
            }
            logger.debug("Computing distances for {}", (Object)pathClass);
            List<PathObject> filteredAnnotations = annotations.stream().filter(a -> a.getPathClass() == pathClass).toList();
            if (filteredAnnotations.isEmpty()) continue;
            String name = distanceString + " to annotation " + String.valueOf(pathClass) + " " + unit;
            DistanceTools.centroidToBoundsDistance2D(detections, filteredAnnotations, pixelWidth, pixelHeight, name, signedDistances);
        }
        hierarchy.fireObjectMeasurementsChangedEvent(DistanceTools.class, detections);
    }

    public static void detectionCentroidDistances(ImageData<?> imageData, boolean splitClassNames) {
        ImageServer<?> server = imageData.getServer();
        PathObjectHierarchy hierarchy = imageData.getHierarchy();
        Collection<PathObject> detections = hierarchy.getCellObjects();
        if (detections.isEmpty()) {
            detections = hierarchy.getDetectionObjects();
        }
        if (hierarchy.getTMAGrid() != null) {
            logger.warn("Detection centroid distances command currently ignores TMA grid information!");
        }
        Set pathClasses = detections.stream().map(p -> p.getPathClass()).filter(p -> p != null && p.isValid() && !PathClassTools.isIgnoredClass(p)).collect(Collectors.toSet());
        PixelCalibration cal = server.getPixelCalibration();
        String xUnit = cal.getPixelWidthUnit();
        String yUnit = cal.getPixelHeightUnit();
        double pixelWidth = cal.getPixelWidth().doubleValue();
        double pixelHeight = cal.getPixelHeight().doubleValue();
        if (!xUnit.equals(yUnit)) {
            throw new IllegalArgumentException("Pixel width & height units do not match! Width " + xUnit + ", height " + yUnit);
        }
        String unit = xUnit;
        for (PathClass pathClass : pathClasses) {
            if (splitClassNames) {
                List<String> names = PathClassTools.splitNames(pathClass);
                for (String name : names) {
                    logger.debug("Computing distances for {}", (Object)pathClass);
                    List<PathObject> filteredDetections = detections.stream().filter(a -> PathClassTools.containsName(a.getPathClass(), name)).toList();
                    if (filteredDetections.isEmpty()) continue;
                    String measurementName = "Distance to detection with " + name + " " + unit;
                    DistanceTools.centroidToCentroidDistance2D(detections, filteredDetections, pixelWidth, pixelHeight, measurementName);
                }
                continue;
            }
            logger.debug("Computing distances for {}", (Object)pathClass);
            List<PathObject> filteredDetections = detections.stream().filter(a -> a.getPathClass() == pathClass).toList();
            if (filteredDetections.isEmpty()) continue;
            String name = "Distance to detection " + String.valueOf(pathClass) + " " + unit;
            DistanceTools.centroidToCentroidDistance2D(detections, filteredDetections, pixelWidth, pixelHeight, name);
        }
        hierarchy.fireObjectMeasurementsChangedEvent(DistanceTools.class, detections);
    }

    public static void centroidToBoundsDistance2D(Collection<PathObject> sourceObjects, Collection<PathObject> targetObjects, double pixelWidth, double pixelHeight, String measurementName) {
        DistanceTools.centroidToBoundsDistance2D(sourceObjects, targetObjects, pixelWidth, pixelHeight, measurementName, false);
    }

    public static void centroidToBoundsDistance2D(Collection<PathObject> sourceObjects, Collection<PathObject> targetObjects, double pixelWidth, double pixelHeight, String measurementName, boolean signedDistances) {
        DistanceTools.centroidToObjectsDistance2D(sourceObjects, targetObjects, pixelWidth, pixelHeight, measurementName, signedDistances, DistanceType.CENTROID_TO_BOUNDS);
    }

    private static void centroidToObjectsDistance2D(Collection<PathObject> sourceObjects, Collection<PathObject> targetObjects, double pixelWidth, double pixelHeight, String measurementName, boolean signedDistances, DistanceType distanceType) {
        boolean preferNucleusForCentroids = true;
        boolean preferNucleusForNonCentroidTargets = false;
        if (distanceType == DistanceType.CENTROID_TO_CENTROID && targetObjects.size() > 50 && !(targetObjects instanceof Set)) {
            targetObjects = new HashSet<PathObject>(targetObjects);
        }
        TreeSet<Integer> timePoints = new TreeSet<Integer>();
        TreeSet<Integer> zSlices = new TreeSet<Integer>();
        for (PathObject temp : sourceObjects) {
            timePoints.add(temp.getROI().getT());
            zSlices.add(temp.getROI().getZ());
        }
        AffineTransformation transform = pixelWidth == 1.0 && pixelHeight == 1.0 ? null : AffineTransformation.scaleInstance((double)pixelWidth, (double)pixelHeight);
        Iterator iterator = timePoints.iterator();
        while (iterator.hasNext()) {
            int t = (Integer)iterator.next();
            Iterator iterator2 = zSlices.iterator();
            while (iterator2.hasNext()) {
                IndexedPointInAreaLocator locator;
                int z = (Integer)iterator2.next();
                PrecisionModel precision = null;
                ArrayList<Geometry> areaGeometries = new ArrayList<Geometry>();
                ArrayList<Geometry> lineGeometries = new ArrayList<Geometry>();
                ArrayList<Geometry> pointGeometries = new ArrayList<Geometry>();
                for (PathObject annotation : targetObjects) {
                    Geometry geom;
                    ROI roi = annotation.getROI();
                    if (roi == null || roi.getZ() != z || roi.getT() != t) continue;
                    if (distanceType == DistanceType.CENTROID_TO_CENTROID) {
                        geom = PathObjectTools.getROI(annotation, preferNucleusForCentroids).getGeometry();
                        geom = geom.getCentroid();
                    } else {
                        geom = PathObjectTools.getROI(annotation, preferNucleusForNonCentroidTargets).getGeometry();
                    }
                    if (transform != null) {
                        geom = transform.transform(geom);
                        if (precision == null) {
                            precision = geom.getPrecisionModel();
                        }
                    }
                    if (geom instanceof Puntal) {
                        pointGeometries.add(geom);
                        continue;
                    }
                    if (geom instanceof Lineal) {
                        lineGeometries.add(geom);
                        continue;
                    }
                    if (geom instanceof Polygonal) {
                        areaGeometries.add(geom);
                        continue;
                    }
                    for (int i = 0; i < geom.getNumGeometries(); ++i) {
                        Geometry geom2 = geom.getGeometryN(i);
                        if (geom2 instanceof Puntal) {
                            pointGeometries.add(geom2);
                            continue;
                        }
                        if (geom2 instanceof Lineal) {
                            lineGeometries.add(geom2);
                            continue;
                        }
                        if (geom2 instanceof Polygonal) {
                            areaGeometries.add(geom2);
                            continue;
                        }
                        logger.warn("Unexpected nested Geometry collection, some Geometries may be ignored");
                    }
                }
                if (areaGeometries.isEmpty() && pointGeometries.isEmpty() && lineGeometries.isEmpty()) continue;
                PrecisionModel precisionModel = precision == null ? GeometryTools.getDefaultFactory().getPrecisionModel() : precision;
                ArrayList<Coordinate> pointCoords = new ArrayList<Coordinate>();
                Geometry temp = null;
                if (!areaGeometries.isEmpty()) {
                    temp = areaGeometries.size() == 1 ? (Geometry)areaGeometries.get(0) : GeometryCombiner.combine(areaGeometries);
                }
                Geometry shapeGeometry = temp;
                temp = null;
                if (!lineGeometries.isEmpty()) {
                    temp = lineGeometries.size() == 1 ? (Geometry)lineGeometries.get(0) : GeometryCombiner.combine(lineGeometries);
                }
                Geometry lineGeometry = temp;
                if (!pointGeometries.isEmpty()) {
                    for (Geometry geom : pointGeometries) {
                        for (Coordinate coord : geom.getCoordinates()) {
                            precisionModel.makePrecise(coord);
                            pointCoords.add(coord);
                        }
                    }
                }
                STRtree pointTree = pointCoords != null && pointCoords.size() > 1000 ? DistanceTools.createCoordinateCache(pointCoords) : null;
                CoordinateDistance coordinateDistance = new CoordinateDistance();
                int zi = z;
                int ti = t;
                IndexedPointInAreaLocator indexedPointInAreaLocator = locator = shapeGeometry == null ? null : new IndexedPointInAreaLocator(shapeGeometry);
                if (locator != null) {
                    locator.locate(new Coordinate(0.0, 0.0));
                }
                Collection<PathObject> finalTargets = targetObjects;
                sourceObjects.parallelStream().forEach(p -> {
                    double distance;
                    ROI roi = PathObjectTools.getROI(p, preferNucleusForCentroids);
                    if (roi.getZ() != zi || roi.getT() != ti) {
                        return;
                    }
                    if (distanceType == DistanceType.CENTROID_TO_CENTROID && finalTargets.contains(p)) {
                        distance = 0.0;
                    } else {
                        Coordinate coord = new Coordinate(roi.getCentroidX() * pixelWidth, roi.getCentroidY() * pixelHeight);
                        precisionModel.makePrecise(coord);
                        double pointDistance = pointCoords == null ? Double.POSITIVE_INFINITY : DistanceTools.computeCoordinateDistance(coord, pointCoords, pointTree, coordinateDistance);
                        double lineDistance = lineGeometry == null ? Double.POSITIVE_INFINITY : DistanceTools.computeDistance(coord, lineGeometry, null, false);
                        double shapeDistance = shapeGeometry == null ? Double.POSITIVE_INFINITY : DistanceTools.computeDistance(coord, shapeGeometry, (PointOnGeometryLocator)locator, signedDistances);
                        distance = Math.min(lineDistance, Math.min(pointDistance, shapeDistance));
                    }
                    try (MeasurementList ml = p.getMeasurementList();){
                        ml.put(measurementName, distance);
                    }
                });
            }
        }
    }

    public static void centroidToCentroidDistance2D(Collection<PathObject> sourceObjects, Collection<PathObject> targetObjects, double pixelWidth, double pixelHeight, String measurementName) {
        DistanceTools.centroidToObjectsDistance2D(sourceObjects, targetObjects, pixelWidth, pixelHeight, measurementName, false, DistanceType.CENTROID_TO_CENTROID);
    }

    public static double computeCoordinateDistance(Coordinate coord, Collection<Coordinate> targets) {
        double d = Double.POSITIVE_INFINITY;
        for (Coordinate target : targets) {
            d = Math.min(d, coord.distance(target));
        }
        return d;
    }

    public static double computeCoordinateDistance(Coordinate coord, STRtree tree) {
        return DistanceTools.computeCoordinateDistance(coord, tree, new CoordinateDistance());
    }

    public static STRtree createCoordinateCache(Collection<Coordinate> coords) {
        STRtree tree = new STRtree();
        for (Coordinate c : coords) {
            tree.insert(new Envelope(c), (Object)c);
        }
        return tree;
    }

    private static double computeCoordinateDistance(Coordinate coord, STRtree tree, ItemDistance distance) {
        if (tree.isEmpty()) {
            return Double.POSITIVE_INFINITY;
        }
        Envelope env = new Envelope(coord);
        Coordinate nearest = (Coordinate)tree.nearestNeighbour(env, (Object)coord, (ItemDistance)new CoordinateDistance());
        return nearest == null ? Double.POSITIVE_INFINITY : coord.distance(nearest);
    }

    private static double computeCoordinateDistance(Coordinate coord, Collection<Coordinate> targets, STRtree tree, ItemDistance distance) {
        if (tree != null) {
            return DistanceTools.computeCoordinateDistance(coord, tree, distance);
        }
        double d = Double.POSITIVE_INFINITY;
        for (Coordinate target : targets) {
            d = Math.min(d, coord.distance(target));
        }
        return d;
    }

    public static double computeDistance(Coordinate coord, Geometry geometry, PointOnGeometryLocator locator) {
        return DistanceTools.computeDistance(coord, geometry, locator, false);
    }

    public static double computeDistance(Coordinate coord, Geometry geometry, PointOnGeometryLocator locator, boolean signedDistance) {
        if (locator == null) {
            PointPairDistance dist = new PointPairDistance();
            DistanceToPoint.computeDistance((Geometry)geometry, (Coordinate)coord, (PointPairDistance)dist);
            double distance = dist.getDistance();
            if (signedDistance && distance != 0.0 && SimplePointInAreaLocator.isContained((Coordinate)coord, (Geometry)geometry)) {
                return -distance;
            }
            return distance;
        }
        int location = locator.locate(coord);
        double distance = 0.0;
        if (location == 2) {
            PointPairDistance dist = new PointPairDistance();
            DistanceToPoint.computeDistance((Geometry)geometry, (Coordinate)coord, (PointPairDistance)dist);
            distance = dist.getDistance();
        } else if (signedDistance && location == 0) {
            PointPairDistance dist = new PointPairDistance();
            DistanceToPoint.computeDistance((Geometry)geometry, (Coordinate)coord, (PointPairDistance)dist);
            distance = -dist.getDistance();
        }
        return distance;
    }

    private static enum DistanceType {
        CENTROID_TO_CENTROID,
        CENTROID_TO_BOUNDS;

    }

    private static class CoordinateDistance
    implements ItemDistance {
        private CoordinateDistance() {
        }

        public double distance(ItemBoundable item1, ItemBoundable item2) {
            Coordinate o1 = (Coordinate)item1.getItem();
            Coordinate o2 = (Coordinate)item2.getItem();
            return o1.distance(o2);
        }
    }
}

