/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.analysis;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.locationtech.jts.densify.Densifier;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.Polygonal;
import org.locationtech.jts.geom.PrecisionModel;
import org.locationtech.jts.geom.util.AffineTransformation;
import org.locationtech.jts.geom.util.GeometryCombiner;
import org.locationtech.jts.index.SpatialIndex;
import org.locationtech.jts.index.hprtree.HPRtree;
import org.locationtech.jts.index.quadtree.Quadtree;
import org.locationtech.jts.precision.GeometryPrecisionReducer;
import org.locationtech.jts.triangulate.DelaunayTriangulationBuilder;
import org.locationtech.jts.triangulate.IncrementalDelaunayTriangulator;
import org.locationtech.jts.triangulate.quadedge.QuadEdge;
import org.locationtech.jts.triangulate.quadedge.QuadEdgeLocator;
import org.locationtech.jts.triangulate.quadedge.QuadEdgeSubdivision;
import org.locationtech.jts.triangulate.quadedge.Vertex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.LogTools;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.PathObjects;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.ImageRegion;
import qupath.lib.roi.GeometryTools;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;

public class DelaunayTools {
    private static final Logger logger = LoggerFactory.getLogger(DelaunayTools.class);

    private static boolean calibrated(PixelCalibration cal) {
        return cal != null && (cal.getPixelHeight().doubleValue() != 1.0 || cal.getPixelWidth().doubleValue() != 1.0);
    }

    private static Function<PathObject, Collection<Coordinate>> createGeometryExtractor(PixelCalibration cal, boolean preferNucleus, double densifyFactor, double erosion) {
        PrecisionModel precision = DelaunayTools.calibrated(cal) ? null : GeometryTools.getDefaultFactory().getPrecisionModel();
        AffineTransformation transform = DelaunayTools.calibrated(cal) ? AffineTransformation.scaleInstance((double)cal.getPixelWidth().doubleValue(), (double)cal.getPixelHeight().doubleValue()) : null;
        return p -> {
            int n;
            Geometry geom2;
            double buffer;
            ROI roi = PathObjectTools.getROI(p, preferNucleus);
            if (roi == null || roi.isEmpty()) {
                return Collections.emptyList();
            }
            Geometry geom = roi.getGeometry();
            if (transform != null) {
                geom = transform.transform(geom);
            }
            if ((buffer = -Math.abs(erosion)) < 0.0 && geom instanceof Polygonal) {
                Geometry geomBefore = geom;
                if ((geom = GeometryTools.attemptOperation(geom, g -> g.buffer(buffer))).isEmpty()) {
                    geom = geomBefore;
                }
            }
            if (precision != null && !(geom2 = GeometryTools.attemptOperation(geom, g -> GeometryPrecisionReducer.reduce((Geometry)g, (PrecisionModel)precision))).isEmpty()) {
                geom = geom2;
            }
            if (densifyFactor > 0.0 && !(geom2 = GeometryTools.attemptOperation(geom, g -> Densifier.densify((Geometry)g, (double)densifyFactor))).isEmpty()) {
                geom = geom2;
            }
            Coordinate[] coords = geom.getCoordinates();
            LinkedHashSet<Coordinate> output = new LinkedHashSet<Coordinate>();
            PrecisionModel p2 = precision;
            Coordinate lastCoordinate = null;
            if (p2 == null) {
                p2 = GeometryTools.getDefaultFactory().getPrecisionModel();
            }
            if ((n = coords.length) == 0) {
                logger.warn("Empty Geometry found for {}", p);
                return Collections.emptyList();
            }
            double minDistance = densifyFactor * 0.5;
            Coordinate firstCoordinate = coords[0];
            while (n > 2 && firstCoordinate.distance(coords[n - 1]) < minDistance) {
                --n;
            }
            for (int i = 0; i < n; ++i) {
                Coordinate c = coords[i];
                p2.makePrecise(c);
                if (i != 0 && (!(c.distance(lastCoordinate) > minDistance) || i >= n - 1 && !(c.distance(coords[0]) > minDistance))) continue;
                output.add(c);
                lastCoordinate = c;
            }
            return output;
        };
    }

    private static Function<PathObject, Collection<Coordinate>> createCentroidExtractor(PixelCalibration cal, boolean preferNucleus) {
        PrecisionModel precision = DelaunayTools.calibrated(cal) ? null : GeometryTools.getDefaultFactory().getPrecisionModel();
        return p -> {
            ROI roi = PathObjectTools.getROI(p, preferNucleus);
            if (roi != null) {
                double x = roi.getCentroidX();
                double y = roi.getCentroidY();
                if (precision != null) {
                    x = precision.makePrecise(x);
                    y = precision.makePrecise(y);
                }
                if (Double.isFinite(x) && Double.isFinite(y)) {
                    return Collections.singletonList(new Coordinate(x, y));
                }
            }
            return Collections.emptyList();
        };
    }

    public static Builder newBuilder(Collection<PathObject> pathObjects) {
        return new Builder(pathObjects);
    }

    public static Subdivision createFromCentroids(Collection<PathObject> pathObjects, boolean preferNucleusROI) {
        logger.debug("Creating subdivision from ROI centroids for {} objects", (Object)pathObjects.size());
        HashMap<Coordinate, PathObject> coords = new HashMap<Coordinate, PathObject>();
        ImagePlane plane = null;
        PrecisionModel precisionModel = GeometryTools.getDefaultFactory().getPrecisionModel();
        for (PathObject pathObject : pathObjects) {
            ROI roi = PathObjectTools.getROI(pathObject, preferNucleusROI);
            if (plane == null) {
                plane = roi.getImagePlane();
            } else if (!plane.equals(roi.getImagePlane())) {
                logger.warn("Non-matching image planes: {} and {}! Object will be skipped...", (Object)plane, (Object)roi.getImagePlane());
                continue;
            }
            double x = precisionModel.makePrecise(roi.getCentroidX());
            double y = precisionModel.makePrecise(roi.getCentroidY());
            Coordinate coord = new Coordinate(x, y);
            coords.put(coord, pathObject);
        }
        return new Subdivision(DelaunayTools.createSubdivision(coords.keySet(), 0.01), pathObjects, coords, plane);
    }

    public static Subdivision createFromGeometryCoordinates(Collection<PathObject> pathObjects, boolean preferNucleusROI, double densifyFactor) {
        logger.debug("Creating subdivision from geometry coordinates for {} objects", (Object)pathObjects.size());
        HashMap<Coordinate, PathObject> coords = new HashMap<Coordinate, PathObject>();
        ImagePlane plane = null;
        for (PathObject pathObject : pathObjects) {
            Coordinate[] coordsTemp;
            ROI roi = PathObjectTools.getROI(pathObject, preferNucleusROI);
            if (plane == null) {
                plane = roi.getImagePlane();
            } else if (!plane.equals(roi.getImagePlane())) {
                logger.warn("Non-matching image planes: {} and {}! Object will be skipped...", (Object)plane, (Object)roi.getImagePlane());
                continue;
            }
            Geometry geom = roi.getGeometry();
            if (densifyFactor > 0.0) {
                geom = Densifier.densify((Geometry)geom, (double)densifyFactor);
            }
            for (Coordinate c : coordsTemp = geom.getCoordinates()) {
                PathObject previous = coords.put(c, pathObject);
                if (previous == null) continue;
                logger.warn("Previous coordinate: {}", (Object)previous);
            }
        }
        return new Subdivision(DelaunayTools.createSubdivision(coords.keySet(), 0.001), pathObjects, coords, plane);
    }

    private static QuadEdgeSubdivision createSubdivision(Collection<Coordinate> coords, double tolerance) {
        Envelope envelope = DelaunayTriangulationBuilder.envelope(coords);
        QuadEdgeSubdivision subdiv = new QuadEdgeSubdivision(envelope, tolerance);
        IncrementalDelaunayTriangulator triangulator = new IncrementalDelaunayTriangulator(subdiv);
        triangulator.forceConvex(false);
        subdiv.setLocator(DelaunayTools.getDefaultLocator(subdiv));
        HashSet<QuadEdge> edgeSet = new HashSet<QuadEdge>();
        for (Coordinate coord : DelaunayTools.prepareCoordinates(coords)) {
            QuadEdge edge = triangulator.insertSite(new Vertex(coord));
            if (edgeSet.add(edge)) continue;
            logger.debug("Found duplicate edge!");
        }
        return subdiv;
    }

    static QuadEdgeLocator getDefaultLocator(QuadEdgeSubdivision subdiv) {
        return new FirstVertexLocator(subdiv);
    }

    private static Collection<Coordinate> prepareCoordinates(Collection<Coordinate> coords) {
        return DelaunayTriangulationBuilder.unique((Coordinate[])((Coordinate[])coords.toArray(Coordinate[]::new)));
    }

    public static BiPredicate<PathObject, PathObject> sameClassificationPredicate() {
        return (p1, p2) -> p1.getPathClass() == p2.getPathClass();
    }

    public static BiPredicate<PathObject, PathObject> centroidDistancePredicate(double maxDistance, boolean preferNucleus) {
        return (p1, p2) -> {
            ROI r2;
            ROI r1 = PathObjectTools.getROI(p1, preferNucleus);
            return RoiTools.getCentroidDistance(r1, r2 = PathObjectTools.getROI(p2, preferNucleus)) <= maxDistance;
        };
    }

    public static BiPredicate<PathObject, PathObject> boundaryDistancePredicate(double maxDistance, boolean preferNucleus) {
        return (p1, p2) -> {
            ROI r1 = PathObjectTools.getROI(p1, preferNucleus);
            ROI r2 = PathObjectTools.getROI(p2, preferNucleus);
            return r1.getGeometry().isWithinDistance(r2.getGeometry(), maxDistance);
        };
    }

    public static List<PathObject> createAnnotationsFromSubdivision(Subdivision subdivision, ROI bounds) {
        Map<PathObject, Geometry> mapVoronoi = subdivision.getVoronoiFaces();
        HashMap<PathClass, List> map = new HashMap<PathClass, List>();
        for (Map.Entry<PathObject, Geometry> entry : mapVoronoi.entrySet()) {
            PathClass pathClass = entry.getKey().getPathClass();
            List list = map.computeIfAbsent(pathClass, p -> new ArrayList());
            list.add(entry.getValue());
        }
        Geometry clip = bounds == null ? null : bounds.getGeometry();
        ArrayList<PathObject> annotations = new ArrayList<PathObject>();
        ImagePlane plane = subdivision.getImagePlane();
        for (Map.Entry entry : map.entrySet()) {
            Geometry geometry = GeometryTools.union((Collection)entry.getValue());
            if (clip != null && !clip.covers(geometry)) {
                geometry = clip.intersection(geometry);
            }
            if (geometry.isEmpty()) continue;
            ROI roi = GeometryTools.geometryToROI(geometry, plane);
            PathObject annotation = PathObjects.createAnnotationObject(roi, (PathClass)entry.getKey());
            annotation.setLocked(true);
            annotations.add(annotation);
        }
        return annotations;
    }

    public static Collection<PathObject> classifyObjectsByCluster(Collection<Collection<? extends PathObject>> clusters, Function<Integer, PathClass> pathClassFun) {
        int c = 0;
        ArrayList<PathObject> list = new ArrayList<PathObject>();
        for (Collection<? extends PathObject> cluster : clusters) {
            PathClass pathClass = pathClassFun.apply(c);
            for (PathObject pathObject : cluster) {
                pathObject.setPathClass(pathClass);
                list.add(pathObject);
            }
            ++c;
        }
        return list;
    }

    public static Collection<PathObject> classifyObjectsByCluster(Collection<Collection<? extends PathObject>> clusters) {
        return DelaunayTools.classifyObjectsByCluster(clusters, c -> PathClass.getInstance("Cluster " + (c + 1)));
    }

    public static Collection<PathObject> nameObjectsByCluster(Collection<Collection<? extends PathObject>> clusters, Function<Integer, PathClass> pathClassFun) {
        int c = 0;
        ArrayList<PathObject> list = new ArrayList<PathObject>();
        for (Collection<? extends PathObject> cluster : clusters) {
            PathClass pathClass = pathClassFun.apply(c);
            String name = pathClass == null ? null : pathClass.getName();
            Integer color = pathClass == null ? null : pathClass.getColor();
            for (PathObject pathObject : cluster) {
                pathObject.setName(name);
                pathObject.setColor(color);
                list.add(pathObject);
            }
            ++c;
        }
        return list;
    }

    public static Collection<PathObject> nameObjectsByCluster(Collection<Collection<? extends PathObject>> clusters) {
        return DelaunayTools.nameObjectsByCluster(clusters, c -> PathClass.getInstance("Cluster " + (c + 1)));
    }

    public static class Builder {
        private ExtractorType extractorType = ExtractorType.CENTROIDS;
        private boolean preferNucleusROI = true;
        private PixelCalibration cal = PixelCalibration.getDefaultInstance();
        private double densifyFactor = Double.NaN;
        private double erosion = 1.0;
        private final ImagePlane plane;
        private final Collection<PathObject> pathObjects = new ArrayList<PathObject>();
        private Function<PathObject, Collection<Coordinate>> coordinateExtractor;

        private Builder(Collection<PathObject> pathObjects) {
            ImagePlane plane = null;
            for (PathObject pathObject : pathObjects) {
                ImagePlane currentPlane = pathObject.getROI().getImagePlane();
                if (plane == null) {
                    plane = currentPlane;
                } else if (!plane.equals(currentPlane)) {
                    logger.warn("Non-matching image planes: {} and {}! Object will be skipped...", (Object)plane, (Object)currentPlane);
                    continue;
                }
                this.pathObjects.add(pathObject);
            }
            this.plane = plane == null ? ImagePlane.getDefaultPlane() : plane;
        }

        public Builder calibration(PixelCalibration cal) {
            this.cal = cal;
            return this;
        }

        public Builder centroids() {
            this.extractorType = ExtractorType.CENTROIDS;
            return this;
        }

        public Builder roiBounds() {
            return this.roiBounds(this.densifyFactor, -2.0);
        }

        public Builder roiBounds(double densify, double erosion) {
            this.extractorType = ExtractorType.ROI;
            this.densifyFactor = densify;
            this.erosion = erosion;
            return this;
        }

        public Builder preferNucleus(boolean prefer) {
            this.preferNucleusROI = prefer;
            return this;
        }

        public Builder coordinateExtractor(Function<PathObject, Collection<Coordinate>> coordinateExtractor) {
            this.extractorType = ExtractorType.CUSTOM;
            this.coordinateExtractor = coordinateExtractor;
            return this;
        }

        public Subdivision build() {
            logger.debug("Creating subdivision for {} objects", (Object)this.pathObjects.size());
            HashMap<Coordinate, PathObject> coords = new HashMap<Coordinate, PathObject>();
            double densify = this.densifyFactor;
            if (!Double.isFinite(densify)) {
                densify = this.cal.getAveragedPixelSize().doubleValue() * 4.0;
            }
            Function<PathObject, Collection<Coordinate>> extractor = this.coordinateExtractor;
            switch (this.extractorType.ordinal()) {
                case 1: {
                    extractor = DelaunayTools.createCentroidExtractor(this.cal, this.preferNucleusROI);
                    break;
                }
                case 2: {
                    extractor = DelaunayTools.createGeometryExtractor(this.cal, this.preferNucleusROI, densify, this.erosion);
                    break;
                }
            }
            for (PathObject pathObject : this.pathObjects) {
                for (Coordinate c : extractor.apply(pathObject)) {
                    coords.put(c, pathObject);
                }
            }
            double tolerance = this.cal.getAveragedPixelSize().doubleValue() / 1000.0;
            return new Subdivision(DelaunayTools.createSubdivision(coords.keySet(), tolerance), this.pathObjects, coords, this.plane);
        }

        private static enum ExtractorType {
            CUSTOM,
            CENTROIDS,
            ROI;

        }
    }

    public static class Subdivision {
        private static final Logger logger = LoggerFactory.getLogger(Subdivision.class);
        private final Collection<PathObject> pathObjects;
        private final Map<Coordinate, PathObject> coordinateMap;
        private final QuadEdgeSubdivision subdivision;
        private final ImagePlane plane;
        private volatile transient Map<PathObject, Geometry> voronoiFaces;
        private volatile transient NeighborMap neighbors;

        private Subdivision(QuadEdgeSubdivision subdivision, Collection<PathObject> pathObjects, Map<Coordinate, PathObject> coordinateMap, ImagePlane plane) {
            this.subdivision = subdivision;
            this.pathObjects = pathObjects.stream().distinct().toList();
            this.plane = plane == null ? pathObjects.stream().filter(PathObject::hasROI).map(PathObject::getROI).map(ROI::getImagePlane).findFirst().orElse(ImagePlane.getDefaultPlane()) : plane;
            this.coordinateMap = Map.copyOf(coordinateMap);
        }

        public ImagePlane getImagePlane() {
            return this.plane;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Map<PathObject, Geometry> getVoronoiFaces() {
            if (this.voronoiFaces == null) {
                Subdivision subdivision = this;
                synchronized (subdivision) {
                    if (this.voronoiFaces == null) {
                        this.voronoiFaces = Collections.unmodifiableMap(this.calculateVoronoiFaces());
                    }
                }
            }
            return this.voronoiFaces;
        }

        public Map<PathObject, ROI> getVoronoiROIs(Geometry clip) {
            Map<PathObject, Geometry> faces = this.getVoronoiFaces();
            HashMap<PathObject, ROI> map = new HashMap<PathObject, ROI>();
            for (Map.Entry<PathObject, Geometry> entry : faces.entrySet()) {
                PathObject pathObject = entry.getKey();
                Geometry face = entry.getValue();
                if (clip != null && !clip.covers(face)) {
                    face = clip.intersection(face);
                }
                ROI roi = GeometryTools.geometryToROI(face, pathObject.getROI().getImagePlane());
                map.put(pathObject, roi);
            }
            return map;
        }

        @Deprecated
        public Collection<PathObject> getPathObjects() {
            LogTools.warnOnce(logger, "getPathObjects() is deprecated; use getObjects() instead");
            return this.pathObjects;
        }

        public Collection<PathObject> getObjects() {
            return this.pathObjects;
        }

        public Collection<PathObject> getObjectsForRegion(ImageRegion region) {
            if (region.getZ() != this.plane.getZ() || region.getT() != this.plane.getT()) {
                return Collections.emptyList();
            }
            Envelope env = new Envelope((double)region.getX(), (double)(region.getX() + region.getWidth()), (double)region.getY(), (double)(region.getY() + region.getHeight()));
            List edges = this.getEdgeIndex().query(env);
            ArrayList<PathObject> pathObjects = new ArrayList<PathObject>();
            for (Object item : edges) {
                QuadEdge edge = (QuadEdge)item;
                pathObjects.add(this.getPathObject(edge.orig()));
                pathObjects.add(this.getPathObject(edge.dest()));
            }
            return pathObjects.stream().distinct().toList();
        }

        public PathObject getNearestNeighbor(PathObject pathObject) {
            List<PathObject> temp = this.getNeighbors(pathObject);
            if (temp.isEmpty()) {
                return null;
            }
            return temp.get(0);
        }

        public PathObject getNearestNeighbor(PathObject pathObject, BiPredicate<PathObject, PathObject> predicate) {
            List<PathObject> temp = this.getFilteredNeighbors(pathObject, predicate);
            if (temp.isEmpty()) {
                return null;
            }
            return temp.get(0);
        }

        public List<PathObject> getFilteredNeighbors(PathObject pathObject, BiPredicate<PathObject, PathObject> predicate) {
            Map<PathObject, List<PathObject>> allNeighbors = this.getAllNeighbors();
            if (predicate != null) {
                return this.filterByPredicate(pathObject, allNeighbors.getOrDefault(pathObject, Collections.emptyList()), predicate);
            }
            return allNeighbors.getOrDefault(pathObject, Collections.emptyList());
        }

        public List<PathObject> getNeighbors(PathObject pathObject) {
            return this.getFilteredNeighbors(pathObject, null);
        }

        public Map<PathObject, List<PathObject>> getFilteredNeighbors(BiPredicate<PathObject, PathObject> predicate) {
            Map<PathObject, List<PathObject>> allNeighbors = this.getAllNeighbors();
            if (predicate != null) {
                LinkedHashMap<PathObject, List<PathObject>> map = new LinkedHashMap<PathObject, List<PathObject>>();
                for (Map.Entry<PathObject, List<PathObject>> entry : allNeighbors.entrySet()) {
                    PathObject pathObject = entry.getKey();
                    List<PathObject> list = entry.getValue();
                    map.put(pathObject, this.filterByPredicate(pathObject, list, predicate));
                }
                return Collections.unmodifiableMap(map);
            }
            return allNeighbors;
        }

        public Map<PathObject, List<PathObject>> getAllNeighbors() {
            return this.getNeighborMap().neighbors();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private NeighborMap getNeighborMap() {
            if (this.neighbors == null) {
                Subdivision subdivision = this;
                synchronized (subdivision) {
                    if (this.neighbors == null) {
                        this.neighbors = this.calculateAllNeighbors();
                    }
                }
            }
            return this.neighbors;
        }

        public boolean isEmpty() {
            return this.pathObjects.isEmpty();
        }

        public int size() {
            return this.pathObjects.size();
        }

        private synchronized NeighborMap calculateAllNeighbors() {
            logger.debug("Calculating all neighbors for {} objects", (Object)this.size());
            List<QuadEdge> edges = this.subdivision.getEdges().stream().sorted(Comparator.comparingDouble(QuadEdge::getLength)).toList();
            HashMap<PathObject, List> neighbors = new HashMap<PathObject, List>();
            HPRtree edgeIndex = new HPRtree();
            for (QuadEdge quadEdge : edges) {
                PathObject pathOrigin = this.getPathObject(quadEdge.orig());
                PathObject pathDest = this.getPathObject(quadEdge.dest());
                if (pathOrigin == null || pathDest == null || pathDest == pathOrigin || neighbors.getOrDefault(pathOrigin, Collections.emptyList()).contains(pathDest)) continue;
                neighbors.computeIfAbsent(pathOrigin, a -> new ArrayList()).add(pathDest);
                neighbors.computeIfAbsent(pathDest, a -> new ArrayList()).add(pathOrigin);
                Envelope env = Subdivision.createEnvelope(pathOrigin.getROI(), pathDest.getROI());
                edgeIndex.insert(env, (Object)quadEdge);
            }
            for (Map.Entry entry : neighbors.entrySet()) {
                entry.setValue(List.copyOf((Collection)entry.getValue()));
            }
            return new NeighborMap(Map.copyOf(neighbors), (SpatialIndex)edgeIndex);
        }

        private static Envelope createEnvelope(ROI roi1, ROI roi2) {
            double x1 = Math.min(roi1.getBoundsX(), roi2.getBoundsX());
            double x2 = Math.max(roi1.getBoundsX() + roi1.getBoundsWidth(), roi2.getBoundsX() + roi2.getBoundsWidth());
            double y1 = Math.min(roi1.getBoundsY(), roi2.getBoundsY());
            double y2 = Math.max(roi1.getBoundsY() + roi1.getBoundsHeight(), roi2.getBoundsY() + roi2.getBoundsHeight());
            return new Envelope(x1, x2, y1, y2);
        }

        private SpatialIndex getEdgeIndex() {
            return this.getNeighborMap().index;
        }

        private PathObject getPathObject(Vertex vertex) {
            return this.coordinateMap.get(vertex.getCoordinate());
        }

        private synchronized Map<PathObject, Geometry> calculateVoronoiFacesByLocations() {
            PathObject pathObject;
            logger.debug("Calculating Voronoi faces for {} objects by location", (Object)this.size());
            List polygons = this.subdivision.getVoronoiCellPolygons(new GeometryFactory());
            HashMap<PathObject, Geometry> map = new HashMap<PathObject, Geometry>();
            HashMap<PathObject, List> mapToMerge = new HashMap<PathObject, List>();
            HashSet<Object> invalidPolygons = new HashSet<Object>();
            HashSet<PathObject> invalidPolygonObjects = new HashSet<PathObject>();
            for (Object polygon : polygons) {
                Geometry existing;
                Coordinate coordinate = (Coordinate)polygon.getUserData();
                if (coordinate == null) {
                    logger.debug("Missing coordinate!");
                    continue;
                }
                pathObject = this.coordinateMap.getOrDefault(coordinate, null);
                if (pathObject == null) {
                    logger.warn("Missing object for coordinate {}", (Object)coordinate);
                    continue;
                }
                if (!polygon.isValid()) {
                    invalidPolygons.add(polygon);
                    invalidPolygonObjects.add(pathObject);
                }
                if ((existing = map.put(pathObject, (Geometry)polygon)) == null) continue;
                List list = mapToMerge.computeIfAbsent(pathObject, g -> {
                    ArrayList<Geometry> l = new ArrayList<Geometry>();
                    l.add(existing);
                    return l;
                });
                list.add(polygon);
            }
            GeometryPrecisionReducer precisionReducer = new GeometryPrecisionReducer(GeometryTools.getDefaultFactory().getPrecisionModel());
            for (Map.Entry entry : mapToMerge.entrySet()) {
                pathObject = (PathObject)entry.getKey();
                List list = (List)entry.getValue();
                Geometry geometry = null;
                try {
                    geometry = GeometryCombiner.combine((Collection)list);
                    geometry = geometry.buffer(0.0);
                    geometry = precisionReducer.reduce(geometry);
                }
                catch (Exception e) {
                    logger.debug("Error doing fast geometry combine for Voronoi faces: {}", (Object)e.getMessage(), (Object)e);
                    try {
                        geometry = GeometryTools.union(list);
                    }
                    catch (Exception e2) {
                        logger.debug("Error doing fallback geometry combine for Voronoi faces: {}", (Object)e2.getMessage(), (Object)e2);
                    }
                }
                map.put(pathObject, geometry);
            }
            if (!invalidPolygons.isEmpty()) {
                String message = "Number of invalid polygons found in Voronoi diagram: {}/{}";
                logger.warn(message, (Object)invalidPolygons.size(), (Object)polygons.size());
            }
            return map;
        }

        private synchronized Map<PathObject, Geometry> calculateVoronoiFaces() {
            if (this.pathObjects.size() < this.coordinateMap.size()) {
                return this.calculateVoronoiFacesByLocations();
            }
            logger.debug("Calculating Voronoi faces for {} objects", (Object)this.size());
            List polygons = this.subdivision.getVoronoiCellPolygons(GeometryTools.getDefaultFactory());
            HashMap<PathObject, Geometry> map = new HashMap<PathObject, Geometry>();
            HashMap<PathObject, List> mapToMerge = new HashMap<PathObject, List>();
            for (Polygon polygon : polygons) {
                if (polygon.isEmpty() || !(polygon instanceof Polygonal)) continue;
                Coordinate coord = (Coordinate)polygon.getUserData();
                PathObject pathObject = this.coordinateMap.get(coord);
                if (pathObject == null) {
                    logger.warn("No detection found for {}", (Object)coord);
                    continue;
                }
                Geometry existing = map.put(pathObject, (Geometry)polygon);
                if (existing == null) continue;
                List list = mapToMerge.computeIfAbsent(pathObject, g -> {
                    ArrayList<Geometry> l = new ArrayList<Geometry>();
                    l.add(existing);
                    return l;
                });
                list.add(polygon);
            }
            for (Map.Entry entry : mapToMerge.entrySet()) {
                PathObject pathObject = (PathObject)entry.getKey();
                List list = (List)entry.getValue();
                Geometry geometry = null;
                try {
                    geometry = GeometryCombiner.combine((Collection)list).buffer(0.0);
                }
                catch (Exception e) {
                    logger.debug("Error doing fast geometry combine for Voronoi faces: {}", (Object)e.getMessage(), (Object)e);
                    try {
                        geometry = GeometryTools.union(list);
                    }
                    catch (Exception e2) {
                        logger.debug("Error doing fallback geometry combine for Voronoi faces: {}", (Object)e2.getMessage(), (Object)e2);
                    }
                }
                map.put(pathObject, geometry);
            }
            return map;
        }

        private List<PathObject> filterByPredicate(PathObject pathObject, List<? extends PathObject> list, BiPredicate<PathObject, PathObject> predicate) {
            return list.stream().filter(p -> predicate.test(pathObject, (PathObject)p)).collect(Collectors.toList());
        }

        public List<Collection<PathObject>> getClusters(BiPredicate<PathObject, PathObject> predicate) {
            HashSet<PathObject> alreadyClustered = new HashSet<PathObject>();
            ArrayList<Collection<PathObject>> output = new ArrayList<Collection<PathObject>>();
            Map<PathObject, List<PathObject>> neighbors = this.getFilteredNeighbors(predicate);
            for (PathObject pathObject : this.getObjects()) {
                if (alreadyClustered.contains(pathObject)) continue;
                Collection<PathObject> cluster = this.buildCluster(pathObject, neighbors, alreadyClustered);
                output.add(cluster);
            }
            return output;
        }

        private Collection<PathObject> buildCluster(PathObject parent, Map<PathObject, List<PathObject>> neighbors, Collection<PathObject> alreadyClustered) {
            ArrayList<PathObject> cluster = new ArrayList<PathObject>();
            ArrayDeque<PathObject> deque = new ArrayDeque<PathObject>();
            deque.add(parent);
            while (!deque.isEmpty()) {
                PathObject pathObject = (PathObject)deque.pop();
                if (!alreadyClustered.add(pathObject)) continue;
                cluster.add(pathObject);
                for (PathObject neighbor : neighbors.get(pathObject)) {
                    if (alreadyClustered.contains(neighbor)) continue;
                    deque.add(neighbor);
                }
            }
            return cluster;
        }

        private record NeighborMap(Map<PathObject, List<PathObject>> neighbors, SpatialIndex index) {
        }
    }

    static class FirstVertexLocator
    implements QuadEdgeLocator {
        private final QuadEdgeSubdivision subdiv;
        private QuadEdge firstLiveEdge;

        FirstVertexLocator(QuadEdgeSubdivision subdiv) {
            this.subdiv = subdiv;
        }

        private QuadEdge firstEdge() {
            if (this.firstLiveEdge == null || !this.firstLiveEdge.isLive()) {
                this.firstLiveEdge = (QuadEdge)this.subdiv.getEdges().iterator().next();
            }
            return this.firstLiveEdge;
        }

        public QuadEdge locate(Vertex v) {
            return this.subdiv.locateFromEdge(v, this.firstEdge());
        }
    }

    private static class QuadTreeQuadEdgeLocator
    implements QuadEdgeLocator {
        private final Quadtree tree;
        private final QuadEdgeSubdivision subdiv;
        private final Envelope env;
        private QuadEdge lastEdge;
        private final Set<QuadEdge> existingEdges;
        private int calledFirst = 0;
        private int usedCache = 0;

        QuadTreeQuadEdgeLocator(QuadEdgeSubdivision subdiv) {
            this.subdiv = subdiv;
            this.tree = new Quadtree();
            this.env = new Envelope();
            this.existingEdges = new HashSet<QuadEdge>();
        }

        private QuadEdge firstEdge() {
            ++this.calledFirst;
            return (QuadEdge)this.subdiv.getEdges().iterator().next();
        }

        void debugLog() {
            logger.info("Called first edge: {} times", (Object)this.calledFirst);
            logger.info("Used spatial cache: {} times", (Object)this.usedCache);
        }

        public QuadEdge locate(Vertex v) {
            double pad = 2.0;
            this.env.init(v.getX() - pad, v.getX() + pad, v.getY() - pad, v.getY() + pad);
            QuadEdge closestEdge = null;
            double closestDistance = Double.POSITIVE_INFINITY;
            Coordinate coord = v.getCoordinate();
            if (this.lastEdge != null && this.lastEdge.isLive()) {
                closestDistance = Math.min(this.lastEdge.orig().getCoordinate().distance(coord), this.lastEdge.dest().getCoordinate().distance(coord));
                closestEdge = this.lastEdge;
            }
            if (closestDistance > pad) {
                List list = this.tree.query(this.env);
                for (QuadEdge e : list) {
                    if (e.isLive()) {
                        double dist = Math.min(e.orig().getCoordinate().distance(coord), e.dest().getCoordinate().distance(coord));
                        if (!(dist < closestDistance)) continue;
                        closestEdge = e;
                        closestDistance = dist;
                        continue;
                    }
                    this.existingEdges.remove(e);
                    this.tree.remove(this.env, (Object)e);
                }
                if (closestEdge != null) {
                    ++this.usedCache;
                    this.lastEdge = closestEdge;
                }
                if (this.lastEdge == null || !this.lastEdge.isLive()) {
                    this.lastEdge = this.firstEdge();
                }
            }
            this.lastEdge = this.subdiv.locateFromEdge(v, this.lastEdge);
            if (this.existingEdges.add(this.lastEdge)) {
                this.env.init(this.lastEdge.orig().getCoordinate());
                this.tree.insert(this.env, (Object)this.lastEdge);
                this.env.init(this.lastEdge.dest().getCoordinate());
                this.tree.insert(this.env, (Object)this.lastEdge);
            }
            return this.lastEdge;
        }
    }
}

