/*
 * Decompiled with CFR 0.152.
 */
package qupath.opencv.tools;

import ij.CompositeImage;
import ij.ImagePlus;
import ij.ImageStack;
import ij.process.ByteProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import ij.process.ShortProcessor;
import java.awt.image.BandedSampleModel;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.IndexColorModel;
import java.awt.image.WritableRaster;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.DoublePredicate;
import java.util.function.DoubleUnaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.DoubleStream;
import org.apache.commons.math3.stat.descriptive.rank.Percentile;
import org.apache.commons.math3.stat.ranking.NaNStrategy;
import org.bytedeco.javacpp.IntPointer;
import org.bytedeco.javacpp.Pointer;
import org.bytedeco.javacpp.PointerScope;
import org.bytedeco.javacpp.indexer.ByteIndexer;
import org.bytedeco.javacpp.indexer.DoubleIndexer;
import org.bytedeco.javacpp.indexer.FloatIndexer;
import org.bytedeco.javacpp.indexer.Index;
import org.bytedeco.javacpp.indexer.Indexer;
import org.bytedeco.javacpp.indexer.IntIndexer;
import org.bytedeco.javacpp.indexer.ShortIndexer;
import org.bytedeco.javacpp.indexer.UByteIndexer;
import org.bytedeco.javacpp.indexer.UShortIndexer;
import org.bytedeco.opencv.global.opencv_core;
import org.bytedeco.opencv.global.opencv_imgproc;
import org.bytedeco.opencv.opencv_core.Mat;
import org.bytedeco.opencv.opencv_core.MatExpr;
import org.bytedeco.opencv.opencv_core.MatVector;
import org.bytedeco.opencv.opencv_core.Point;
import org.bytedeco.opencv.opencv_core.Rect;
import org.bytedeco.opencv.opencv_core.Scalar;
import org.bytedeco.opencv.opencv_core.Size;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.images.ContourTracing;
import qupath.lib.analysis.images.SimpleImage;
import qupath.lib.analysis.images.SimpleImages;
import qupath.lib.color.ColorModelFactory;
import qupath.lib.common.ColorTools;
import qupath.lib.common.GeneralTools;
import qupath.lib.common.ThreadTools;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelType;
import qupath.lib.objects.PathObject;
import qupath.lib.regions.Padding;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.interfaces.ROI;
import qupath.opencv.tools.ProcessingCV;

public class OpenCVTools {
    private static final Logger logger = LoggerFactory.getLogger(OpenCVTools.class);
    private static Map<Integer, Mat> cachedSumDisks = Collections.synchronizedMap(new HashMap());
    private static Map<Integer, Mat> cachedMeanDisks = Collections.synchronizedMap(new HashMap());
    private static int DEFAULT_BORDER_TYPE = 2;

    public static Mat imageToMat(BufferedImage img) {
        switch (img.getType()) {
            case 4: 
            case 5: {
                return OpenCVTools.imageToMatBGR(img, false);
            }
            case 6: 
            case 7: {
                return OpenCVTools.imageToMatBGR(img, true);
            }
            case 2: 
            case 3: {
                return OpenCVTools.imageToMatRGB(img, true);
            }
            case 1: 
            case 8: 
            case 9: {
                return OpenCVTools.imageToMatRGB(img, false);
            }
        }
        int width = img.getWidth();
        int height = img.getHeight();
        WritableRaster raster = img.getRaster();
        DataBuffer buffer = raster.getDataBuffer();
        int nChannels = raster.getNumBands();
        Mat mat = new Mat(height, width, switch (buffer.getDataType()) {
            case 0 -> opencv_core.CV_8UC((int)nChannels);
            case 4 -> opencv_core.CV_32FC((int)nChannels);
            case 3 -> opencv_core.CV_32SC((int)nChannels);
            case 2 -> opencv_core.CV_16SC((int)nChannels);
            case 1 -> opencv_core.CV_16UC((int)nChannels);
            default -> opencv_core.CV_64FC((int)nChannels);
        }, Scalar.ZERO);
        OpenCVTools.putPixels(raster, mat);
        return mat;
    }

    private static void putPixels(WritableRaster raster, UByteIndexer indexer) {
        int[] pixels = null;
        int width = raster.getWidth();
        int height = raster.getHeight();
        for (int b = 0; b < raster.getNumBands(); ++b) {
            pixels = raster.getSamples(0, 0, width, height, b, pixels);
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    indexer.put((long)y, (long)x, (long)b, pixels[y * width + x]);
                }
            }
        }
    }

    public static int getOpenCVPixelType(PixelType pixelType) throws IllegalArgumentException {
        switch (pixelType) {
            case FLOAT32: {
                return 5;
            }
            case FLOAT64: {
                return 6;
            }
            case INT16: {
                return 3;
            }
            case INT32: {
                return 4;
            }
            case INT8: {
                return 1;
            }
            case UINT16: {
                return 2;
            }
            case UINT32: {
                logger.warn("OpenCV does not have a uint32 pixel type! Will returned signed type instead as the closest match.");
                return 4;
            }
            case UINT8: {
                return 0;
            }
        }
        throw new IllegalArgumentException("Unknown pixel type " + String.valueOf(pixelType));
    }

    public static void apply(Mat mat, DoubleUnaryOperator operator) {
        Indexer indexer = mat.createIndexer();
        long[] sizes = indexer.sizes();
        long total = 1L;
        for (long dim : sizes) {
            total *= dim;
        }
        Indexer indexer2 = indexer.reindex(Index.create((long)total));
        long[] inds = new long[1];
        long i = 0L;
        while (i < total) {
            inds[0] = i++;
            double val = indexer2.getDouble(inds);
            val = operator.applyAsDouble(val);
            indexer2.putDouble(inds, val);
        }
        indexer2.close();
        indexer.close();
    }

    public static Mat createMask(Mat mat, DoublePredicate predicate, double trueValue, double falseValue) {
        Mat matMask = mat.clone();
        OpenCVTools.apply(matMask, d -> predicate.test(d) ? trueValue : falseValue);
        return matMask;
    }

    public static Mat createBinaryMask(Mat mat, DoublePredicate predicate) {
        Mat matMask = OpenCVTools.createMask(mat, predicate, 255.0, 0.0);
        matMask.convertTo(matMask, 0);
        return matMask;
    }

    public static void replaceValues(Mat mat, double originalValue, double newValue) {
        Mat mask = opencv_core.equals((Mat)mat, (double)originalValue).asMat();
        OpenCVTools.fill(mat, mask, newValue);
        mask.close();
    }

    public static void replaceNaNs(Mat mat, double newValue) {
        int depth = mat.depth();
        if (depth == 5) {
            opencv_core.patchNaNs((Mat)mat, (double)newValue);
        } else if (depth == 6 || depth == 7) {
            Mat mask = opencv_core.notEquals((Mat)mat, (Mat)mat).asMat();
            OpenCVTools.fill(mat, mask, newValue);
            mask.close();
        }
    }

    public static void fill(Mat mat, Mat mask, double value) {
        Mat val = OpenCVTools.scalarMat(value, 6);
        if (mask == null) {
            mat.setTo(val);
        } else {
            mat.setTo(val, mask);
        }
        val.close();
    }

    public static void fill(Mat mat, double value) {
        OpenCVTools.fill(mat, null, value);
    }

    public static List<Mat> splitChannels(Mat mat) {
        ArrayList<Mat> list = new ArrayList<Mat>();
        int channels = mat.channels();
        for (int c = 0; c < channels; ++c) {
            Mat temp = new Mat();
            opencv_core.extractChannel((Mat)mat, (Mat)temp, (int)c);
            list.add(temp);
        }
        return list;
    }

    public static Mat mergeChannels(Collection<? extends Mat> channels, Mat dest) throws IllegalArgumentException {
        boolean mixChannels;
        if (dest == null) {
            dest = new Mat();
        }
        boolean firstMat = true;
        boolean allSingleChannel = true;
        int rows = -1;
        int cols = -1;
        int depth = 0;
        int nChannels = 0;
        for (Mat mat : channels) {
            if (firstMat) {
                rows = mat.rows();
                cols = mat.cols();
                depth = mat.depth();
                if (rows < 0 || cols < 0) {
                    throw new IllegalArgumentException("Rows and columns must be >= 0");
                }
                firstMat = false;
            }
            int nc = mat.channels();
            allSingleChannel = allSingleChannel && nc == 1;
            nChannels += nc;
            if (depth == mat.depth() && rows == mat.rows() && cols == mat.cols()) continue;
            throw new IllegalArgumentException("mergeChannels() requires all Mats to have the same dimensions and depth!");
        }
        boolean bl = mixChannels = !allSingleChannel;
        if (nChannels > 512) {
            throw new IllegalArgumentException("Can't merge more than 512 channels (you requested " + nChannels + ")");
        }
        try (PointerScope pointerScope = new PointerScope();){
            if (mixChannels) {
                dest.create(rows, cols, opencv_core.CV_MAKETYPE((int)depth, (int)nChannels));
                MatVector vecSource = new MatVector((Mat[])channels.toArray(Mat[]::new));
                MatVector vecDest = new MatVector(dest);
                int[] inds = new int[nChannels * 2];
                for (int i = 0; i < nChannels; ++i) {
                    inds[i * 2] = i;
                    inds[i * 2 + 1] = i;
                }
                opencv_core.mixChannels((MatVector)vecSource, (MatVector)vecDest, (IntPointer)new IntPointer(inds));
            } else {
                opencv_core.merge((MatVector)new MatVector((Mat[])channels.toArray(Mat[]::new)), (Mat)dest);
            }
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Merged channels: {}", Arrays.stream(dest.createIndexer().sizes()).mapToObj(l -> l).toList());
        }
        return dest;
    }

    public static boolean isFloat(Mat mat) {
        int depth = mat.depth();
        return depth == 7 || depth == 5 || depth == 6;
    }

    private static double round(double v) {
        if (Double.isFinite(v)) {
            return Math.round(v);
        }
        return v;
    }

    private static double ceil(double v) {
        if (Double.isFinite(v)) {
            return Math.ceil(v);
        }
        return v;
    }

    private static double floor(double v) {
        if (Double.isFinite(v)) {
            return Math.floor(v);
        }
        return v;
    }

    public static void floor(Mat mat) {
        if (!OpenCVTools.isFloat(mat)) {
            return;
        }
        OpenCVTools.apply(mat, OpenCVTools::floor);
    }

    public static void round(Mat mat) {
        if (!OpenCVTools.isFloat(mat)) {
            return;
        }
        OpenCVTools.apply(mat, OpenCVTools::round);
    }

    public static void ceil(Mat mat) {
        if (!OpenCVTools.isFloat(mat)) {
            return;
        }
        OpenCVTools.apply(mat, OpenCVTools::ceil);
    }

    public static Mat ensureContinuous(Mat mat, boolean inPlace) {
        if (!mat.isContinuous()) {
            Mat mat2 = mat.clone();
            if (!inPlace) {
                return mat2;
            }
            mat.put(mat2);
        }
        assert (mat.isContinuous());
        return mat;
    }

    public static Mat vConcat(Collection<? extends Mat> mats, Mat dest) {
        if (dest == null) {
            dest = new Mat();
        }
        opencv_core.vconcat((MatVector)new MatVector((Mat[])mats.toArray(Mat[]::new)), (Mat)dest);
        return dest;
    }

    public static Mat hConcat(Collection<? extends Mat> mats, Mat dest) {
        if (dest == null) {
            dest = new Mat();
        }
        opencv_core.hconcat((MatVector)new MatVector((Mat[])mats.toArray(Mat[]::new)), (Mat)dest);
        return dest;
    }

    public static void applyToChannels(Mat input, Consumer<Mat> fun) {
        if (input.channels() == 1) {
            fun.accept(input);
            return;
        }
        List<Mat> channels = OpenCVTools.splitChannels(input);
        for (Mat c : channels) {
            fun.accept(c);
        }
        OpenCVTools.mergeChannels(channels, input);
    }

    private static void putPixels(WritableRaster raster, UShortIndexer indexer) {
        int[] pixels = null;
        int width = raster.getWidth();
        int height = raster.getHeight();
        for (int b = 0; b < raster.getNumBands(); ++b) {
            pixels = raster.getSamples(0, 0, width, height, b, pixels);
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    indexer.put((long)y, (long)x, (long)b, pixels[y * width + x]);
                }
            }
        }
    }

    private static void putPixels(WritableRaster raster, ShortIndexer indexer) {
        int[] pixels = null;
        int width = raster.getWidth();
        int height = raster.getHeight();
        for (int b = 0; b < raster.getNumBands(); ++b) {
            pixels = raster.getSamples(0, 0, width, height, b, pixels);
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    indexer.put((long)y, (long)x, (long)b, (short)pixels[y * width + x]);
                }
            }
        }
    }

    private static void putPixels(WritableRaster raster, FloatIndexer indexer) {
        float[] pixels = null;
        int width = raster.getWidth();
        int height = raster.getHeight();
        for (int b = 0; b < raster.getNumBands(); ++b) {
            pixels = raster.getSamples(0, 0, width, height, b, pixels);
            for (int y = 0; y < height; ++y) {
                for (int x = 0; x < width; ++x) {
                    indexer.put((long)y, (long)x, (long)b, pixels[y * width + x]);
                }
            }
        }
    }

    private static void putPixels(WritableRaster raster, Mat mat) {
        Indexer indexer = mat.createIndexer();
        if (indexer instanceof UByteIndexer) {
            OpenCVTools.putPixels(raster, (UByteIndexer)indexer);
        } else if (indexer instanceof ShortIndexer) {
            OpenCVTools.putPixels(raster, (ShortIndexer)indexer);
        } else if (indexer instanceof UShortIndexer) {
            OpenCVTools.putPixels(raster, (UShortIndexer)indexer);
        } else if (indexer instanceof FloatIndexer) {
            OpenCVTools.putPixels(raster, (FloatIndexer)indexer);
        } else {
            double[] pixels = null;
            int width = raster.getWidth();
            int height = raster.getHeight();
            long[] indices = new long[3];
            for (int b = 0; b < raster.getNumBands(); ++b) {
                pixels = raster.getSamples(0, 0, width, height, b, pixels);
                indices[2] = b;
                for (int y = 0; y < height; ++y) {
                    indices[0] = y;
                    for (int x = 0; x < width; ++x) {
                        indices[1] = x;
                        indexer.putDouble(indices, pixels[y * width + x]);
                    }
                }
            }
        }
        indexer.release();
    }

    public static BufferedImage matToBufferedImage(Mat mat) {
        return OpenCVTools.matToBufferedImage(mat, null);
    }

    public static BufferedImage matToBufferedImage(Mat mat, ColorModel colorModel) {
        WritableRaster raster;
        int type;
        int bpp = 0;
        switch (mat.depth()) {
            case 0: {
                type = 0;
                bpp = 8;
                break;
            }
            case 1: {
                type = 2;
                bpp = 16;
                break;
            }
            case 2: {
                type = 1;
                bpp = 16;
                break;
            }
            case 3: {
                type = 2;
                bpp = 16;
                break;
            }
            case 4: {
                type = 3;
                bpp = 32;
                break;
            }
            case 5: {
                type = 4;
                bpp = 32;
                break;
            }
            default: {
                logger.warn("Unknown Mat depth {}, will default to CV64F ({})", (Object)mat.depth(), (Object)6);
            }
            case 6: {
                type = 5;
                bpp = 64;
            }
        }
        int width = mat.cols();
        int height = mat.rows();
        int channels = mat.channels();
        BufferedImage img = null;
        if (colorModel == null) {
            if (type == 0) {
                if (channels == 1) {
                    img = new BufferedImage(width, height, 10);
                } else if (channels == 3) {
                    img = new BufferedImage(width, height, 1);
                } else if (channels == 4) {
                    img = new BufferedImage(width, height, 2);
                }
            }
        } else if (colorModel instanceof IndexColorModel) {
            img = new BufferedImage(width, height, 13, (IndexColorModel)colorModel);
        }
        if (img != null) {
            raster = img.getRaster();
        } else if (colorModel != null) {
            raster = colorModel.createCompatibleWritableRaster(width, height);
            img = new BufferedImage(colorModel, raster, false, null);
        } else {
            BandedSampleModel sampleModel = new BandedSampleModel(type, width, height, channels);
            raster = WritableRaster.createWritableRaster(sampleModel, null);
            colorModel = ColorModelFactory.getDummyColorModel((int)(bpp * channels));
            img = new BufferedImage(colorModel, raster, false, null);
        }
        MatVector matvector = new MatVector();
        opencv_core.split((Mat)mat, (MatVector)matvector);
        int[] pixelsInt = null;
        float[] pixelsFloat = null;
        double[] pixelsDouble = null;
        for (int b = 0; b < channels; ++b) {
            Mat matChannel = matvector.get((long)b);
            Indexer indexer = matChannel.createIndexer();
            if (indexer instanceof UByteIndexer) {
                if (pixelsInt == null) {
                    pixelsInt = new int[width * height];
                }
                ((UByteIndexer)indexer).get(0L, pixelsInt);
            } else if (indexer instanceof UShortIndexer) {
                if (pixelsInt == null) {
                    pixelsInt = new int[width * height];
                }
                ((UShortIndexer)indexer).get(0L, pixelsInt);
            } else if (indexer instanceof FloatIndexer) {
                if (pixelsFloat == null) {
                    pixelsFloat = new float[width * height];
                }
                ((FloatIndexer)indexer).get(0L, pixelsFloat);
            } else if (indexer instanceof DoubleIndexer) {
                if (pixelsDouble == null) {
                    pixelsDouble = new double[width * height];
                }
                ((DoubleIndexer)indexer).get(0L, pixelsDouble);
            } else {
                if (pixelsDouble == null) {
                    pixelsDouble = new double[width * height];
                }
                pixelsDouble = new double[width * height];
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        pixelsDouble[y * width + x] = indexer.getDouble(new long[]{y, x, b});
                    }
                }
            }
            if (pixelsInt != null) {
                raster.setSamples(0, 0, width, height, b, pixelsInt);
                continue;
            }
            if (pixelsFloat != null) {
                raster.setSamples(0, 0, width, height, b, pixelsFloat);
                continue;
            }
            if (pixelsDouble == null) continue;
            raster.setSamples(0, 0, width, height, b, pixelsDouble);
        }
        return img;
    }

    public static Mat imageToMatRGB(BufferedImage img, boolean includeAlpha) {
        return OpenCVTools.imageToMatRGBorBGR(img, false, includeAlpha);
    }

    public static Mat imageToMatBGR(BufferedImage img, boolean includeAlpha) {
        return OpenCVTools.imageToMatRGBorBGR(img, true, includeAlpha);
    }

    private static Mat imageToMatRGBorBGR(BufferedImage img, boolean doBGR, boolean includeAlpha) {
        int width = img.getWidth();
        int height = img.getHeight();
        int[] data = img.getRGB(0, 0, width, height, null, 0, img.getWidth());
        Mat mat = includeAlpha ? new Mat(height, width, opencv_core.CV_8UC4) : new Mat(height, width, opencv_core.CV_8UC3);
        UByteIndexer indexer = (UByteIndexer)mat.createIndexer();
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                int val = data[y * width + x];
                int r = ColorTools.red((int)val);
                int g = ColorTools.green((int)val);
                int b = ColorTools.blue((int)val);
                if (doBGR) {
                    indexer.put((long)y, (long)x, 0L, b);
                    indexer.put((long)y, (long)x, 1L, g);
                    indexer.put((long)y, (long)x, 2L, r);
                } else {
                    indexer.put((long)y, (long)x, 0L, r);
                    indexer.put((long)y, (long)x, 1L, g);
                    indexer.put((long)y, (long)x, 2L, b);
                }
                if (!includeAlpha) continue;
                int a = ColorTools.alpha((int)val);
                indexer.put((long)y, (long)x, 3L, a);
            }
        }
        indexer.release();
        return mat;
    }

    @Deprecated
    public static void labelImage(Mat matBinary, Mat matLabels, int contourRetrievalMode) {
        MatVector contours = new MatVector();
        Mat hierarchy = new Mat();
        opencv_imgproc.findContours((Mat)matBinary, (MatVector)contours, (Mat)hierarchy, (int)contourRetrievalMode, (int)2);
        Point offset = new Point(0, 0);
        int c = 0;
        while ((long)c < contours.size()) {
            opencv_imgproc.drawContours((Mat)matLabels, (MatVector)contours, (int)c, (Scalar)Scalar.all((double)(c + 1)), (int)-1, (int)8, (Mat)hierarchy, (int)2, (Point)offset);
            ++c;
        }
        hierarchy.close();
        contours.close();
    }

    public static void addNoise(Mat mat, double mean, double stdDev) {
        if (!Double.isFinite(mean) || !Double.isFinite(stdDev)) {
            throw new IllegalArgumentException("Noise mean and standard deviation must be finite (specified " + mean + " and " + stdDev + ")");
        }
        if (stdDev < 0.0) {
            throw new IllegalArgumentException("Noise standard deviation must be >= 0, but specified value is " + stdDev);
        }
        Mat matMean = new Mat(1, 1, opencv_core.CV_32FC1, Scalar.all((double)mean));
        Mat matStdDev = new Mat(1, 1, opencv_core.CV_32FC1, Scalar.all((double)stdDev));
        int nChannels = mat.channels();
        if (nChannels == 1) {
            opencv_core.randn((Mat)mat, (Mat)matMean, (Mat)matStdDev);
        } else {
            OpenCVTools.applyToChannels(mat, m -> opencv_core.randn((Mat)m, (Mat)matMean, (Mat)matStdDev));
        }
        matMean.close();
        matStdDev.close();
    }

    public static double median(Mat mat) {
        return OpenCVTools.percentiles(mat, 50.0)[0];
    }

    static Percentile createPercentile() {
        return new Percentile().withEstimationType(Percentile.EstimationType.R_7).withNaNStrategy(NaNStrategy.REMOVED);
    }

    public static double[] percentiles(Mat mat, double ... percentiles) {
        boolean doParallel = OpenCVTools.totalPixels(mat) > 1000000L && ThreadTools.getParallelism() > 2;
        boolean doSort = doParallel && percentiles.length > 5;
        return OpenCVTools.percentilesStream(mat, doParallel, doSort, percentiles);
    }

    static double[] percentilesSorted(Mat mat, double ... percentiles) {
        int n;
        double[] result = new double[percentiles.length];
        if (result.length == 0) {
            return result;
        }
        Percentile percentile = OpenCVTools.createPercentile();
        double[] values = OpenCVTools.extractDoubles(mat);
        Arrays.sort(values);
        for (n = (int)OpenCVTools.totalPixels(mat); n >= 0 && Double.isNaN(values[n - 1]); --n) {
        }
        if (n < values.length) {
            values = Arrays.copyOf(values, n);
        }
        double minValue = Double.NaN;
        boolean minSet = false;
        percentile.setData(values);
        for (int i = 0; i < percentiles.length; ++i) {
            if (percentiles[i] == 0.0) {
                if (!minSet) {
                    minValue = Arrays.stream(values).filter(d -> !Double.isNaN(d)).min().orElse(Double.NaN);
                    minSet = true;
                }
                result[i] = minValue;
                continue;
            }
            result[i] = percentile.evaluate(percentiles[i]);
        }
        return result;
    }

    static double[] percentilesStream(Mat mat, boolean doParallel, boolean doSort, double ... percentiles) {
        double[] result = new double[percentiles.length];
        if (result.length == 0) {
            return result;
        }
        Percentile percentile = OpenCVTools.createPercentile();
        double[] values = OpenCVTools.extractDoubles(mat);
        DoubleStream stream = Arrays.stream(values);
        if (doParallel) {
            stream = stream.parallel();
        }
        if (doSort) {
            stream = stream.sorted();
        }
        values = stream.filter(d -> !Double.isNaN(d)).toArray();
        double minValue = Double.NaN;
        boolean minSet = false;
        percentile.setData(values);
        for (int i = 0; i < percentiles.length; ++i) {
            if (percentiles[i] == 0.0) {
                if (!minSet) {
                    minValue = Arrays.stream(values).filter(d -> !Double.isNaN(d)).min().orElse(Double.NaN);
                    minSet = true;
                }
                result[i] = minValue;
                continue;
            }
            result[i] = percentile.evaluate(percentiles[i]);
        }
        return result;
    }

    public static double mean(Mat mat) {
        return OpenCVTools.reduceMat(mat, 1);
    }

    public static double[] channelMean(Mat mat) {
        return OpenCVTools.reduceMat(mat, 1, true);
    }

    public static double stdDev(Mat mat) {
        Mat temp = mat.channels() == 1 ? mat : mat.reshape(1, mat.rows() * mat.cols());
        double[] output = OpenCVTools.channelStdDev(temp);
        assert (output.length == 1);
        return output[0];
    }

    public static double[] channelStdDev(Mat mat) {
        Mat mean = new Mat();
        Mat stdDev = new Mat();
        opencv_core.meanStdDev((Mat)mat, (Mat)mean, (Mat)stdDev);
        double[] output = OpenCVTools.extractDoubles(stdDev);
        mean.close();
        stdDev.close();
        return output;
    }

    public static double sum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 0);
    }

    public static double[] channelSum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 0, true);
    }

    public static double minimum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 3, false)[0];
    }

    public static double[] channelMinimum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 3, true);
    }

    public static double maximum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 2, false)[0];
    }

    public static double[] channelMaximum(Mat mat) {
        return OpenCVTools.reduceMat(mat, 2, true);
    }

    private static double[] reduceMat(Mat mat, int reduction, boolean byChannel) {
        if (byChannel && mat.channels() > 1) {
            return OpenCVTools.splitChannels(mat).stream().mapToDouble(m -> OpenCVTools.reduceMat(m, reduction)).toArray();
        }
        return new double[]{OpenCVTools.reduceMat(mat, reduction)};
    }

    private static double reduceMat(Mat mat, int reduction) {
        double[] values = OpenCVTools.extractDoubles(mat);
        switch (reduction) {
            case 1: {
                return Arrays.stream(values).filter(d -> !Double.isNaN(d)).average().orElse(Double.NaN);
            }
            case 2: {
                return Arrays.stream(values).filter(d -> !Double.isNaN(d)).max().orElse(Double.NaN);
            }
            case 3: {
                return Arrays.stream(values).filter(d -> !Double.isNaN(d)).min().orElse(Double.NaN);
            }
            case 0: {
                return Arrays.stream(values).filter(d -> !Double.isNaN(d)).sum();
            }
        }
        throw new IllegalArgumentException("Unknown reduction type " + reduction);
    }

    public static int typeToChannels(int type) {
        return opencv_core.CV_MAT_CN((int)type);
    }

    public static int typeToDepth(int type) {
        return opencv_core.CV_MAT_DEPTH((int)type);
    }

    public static Mat scalarMatWithType(double value, int type) {
        if (opencv_core.CV_MAT_CN((int)type) <= 4) {
            return new Mat(1, 1, type, Scalar.all((double)value));
        }
        Mat mat = new Mat(1, 1, type);
        OpenCVTools.fill(mat, value);
        return mat;
    }

    public static Mat scalarMat(double value, int depth) {
        return new Mat(1, 1, OpenCVTools.typeToDepth(depth), Scalar.all((double)value));
    }

    public static void putPixelsUnsigned(Mat mat, byte[] pixels) {
        Indexer indexer = mat.createIndexer();
        if (indexer instanceof ByteIndexer) {
            ((ByteIndexer)indexer).put(0L, pixels);
        } else if (indexer instanceof UByteIndexer) {
            int n = pixels.length;
            for (int i = 0; i < n; ++i) {
                ((UByteIndexer)indexer).put(0L, pixels[i] & 0xFF);
            }
        } else {
            throw new IllegalArgumentException("Expected a ByteIndexer, but instead got " + String.valueOf(indexer.getClass()));
        }
    }

    public static void putPixelsFloat(Mat mat, float[] pixels) {
        Indexer indexer = mat.createIndexer();
        if (!(indexer instanceof FloatIndexer)) {
            throw new IllegalArgumentException("Expected a FloatIndexer, but instead got " + String.valueOf(indexer.getClass()));
        }
        ((FloatIndexer)indexer).put(0L, pixels);
    }

    @Deprecated
    public static Mat getCircularStructuringElement(int radius) {
        Mat strel = new Mat(radius * 2 + 1, radius * 2 + 1, opencv_core.CV_8UC1, Scalar.ZERO);
        opencv_imgproc.circle((Mat)strel, (Point)new Point(radius, radius), (int)radius, (Scalar)Scalar.ONE, (int)-1, (int)8, (int)0);
        return strel;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Mat createDisk(int radius, boolean doMean) {
        if (radius <= 0) {
            throw new IllegalArgumentException("Radius must be > 0");
        }
        Map<Integer, Mat> cache = doMean ? cachedMeanDisks : cachedSumDisks;
        Mat kernel = cache.get(radius);
        if (kernel != null) {
            Mat mat = kernel;
            synchronized (mat) {
                if (!kernel.isNull()) {
                    return kernel.clone();
                }
            }
        }
        kernel = new Mat();
        Mat kernelCenter = new Mat(radius * 2 + 1, radius * 2 + 1, opencv_core.CV_8UC1, Scalar.WHITE);
        try (UByteIndexer idxKernel = (UByteIndexer)kernelCenter.createIndexer();){
            idxKernel.put((long)radius, (long)radius, 0);
        }
        opencv_imgproc.distanceTransform((Mat)kernelCenter, (Mat)kernel, (int)2, (int)0);
        opencv_imgproc.threshold((Mat)kernel, (Mat)kernel, (double)radius, (double)1.0, (int)1);
        if (doMean) {
            double sum = opencv_core.sumElems((Mat)kernel).get();
            opencv_core.dividePut((Mat)kernel, (double)sum);
        }
        kernel.retainReference();
        cache.put(radius, kernel);
        return kernel.clone();
    }

    public static void invertBinary(Mat matBinary, Mat matDest) {
        opencv_core.compare((Mat)matBinary, (Mat)new Mat(1, 1, opencv_core.CV_32FC1, Scalar.ZERO), (Mat)matDest, (int)0);
    }

    public static float[] extractPixels(Mat mat, float[] pixels) {
        if (pixels == null) {
            pixels = new float[(int)OpenCVTools.totalPixels(mat)];
        }
        Mat mat2 = null;
        if (mat.depth() != 5) {
            mat2 = new Mat();
            mat.convertTo(mat2, 5);
            OpenCVTools.ensureContinuous(mat2, true);
        } else {
            mat2 = OpenCVTools.ensureContinuous(mat, false);
        }
        FloatIndexer idx = (FloatIndexer)mat2.createIndexer();
        idx.get(0L, pixels);
        idx.release();
        if (mat2 != mat) {
            mat2.close();
        }
        return pixels;
    }

    static long totalPixels(Mat mat) {
        int nChannels = mat.channels();
        if (nChannels > 0) {
            return mat.total() * (long)nChannels;
        }
        return mat.total();
    }

    public static double[] extractPixels(Mat mat, double[] pixels) {
        if (pixels == null) {
            pixels = new double[(int)OpenCVTools.totalPixels(mat)];
        }
        Mat mat2 = null;
        if (mat.depth() != 6) {
            mat2 = new Mat();
            mat.convertTo(mat2, 6);
            OpenCVTools.ensureContinuous(mat2, true);
        } else {
            mat2 = OpenCVTools.ensureContinuous(mat, false);
        }
        DoubleIndexer idx = (DoubleIndexer)mat2.createIndexer();
        idx.get(0L, pixels);
        idx.release();
        if (mat2 != mat) {
            mat2.close();
        }
        return pixels;
    }

    public static double[] extractDoubles(Mat mat) {
        return OpenCVTools.extractPixels(mat, (double[])null);
    }

    public static float[] extractFloats(Mat mat) {
        return OpenCVTools.extractPixels(mat, null);
    }

    public static SimpleImage matToSimpleImage(Mat mat, int channel) {
        Mat temp = mat;
        if (mat.channels() > 1) {
            temp = new Mat();
            opencv_core.extractChannel((Mat)mat, (Mat)temp, (int)channel);
        }
        float[] pixels = OpenCVTools.extractPixels(temp, null);
        return SimpleImages.createFloatImage((float[])pixels, (int)mat.cols(), (int)mat.rows());
    }

    public static void fillSmallHoles(Mat matBinary, double maxArea) {
        Mat matHoles = new Mat();
        OpenCVTools.invertBinary(matBinary, matHoles);
        MatVector contours = new MatVector();
        Mat hierarchy = new Mat();
        opencv_imgproc.findContours((Mat)matHoles, (MatVector)contours, (Mat)hierarchy, (int)2, (int)2);
        Scalar color = Scalar.WHITE;
        int ind = 0;
        Point offset = new Point(0, 0);
        Indexer indexerHierarchy = hierarchy.createIndexer();
        int c = 0;
        while ((long)c < contours.size()) {
            Mat contour = contours.get((long)c);
            if (indexerHierarchy.getDouble(new long[]{0L, ind, 3L}) >= 0.0 || opencv_imgproc.contourArea((Mat)contour) > maxArea) {
                ++ind;
            } else {
                opencv_imgproc.drawContours((Mat)matBinary, (MatVector)contours, (int)c, (Scalar)color, (int)-1, (int)8, null, (int)Integer.MAX_VALUE, (Point)offset);
                ++ind;
            }
            ++c;
        }
    }

    public static void watershedIntensitySplit(Mat matBinary, Mat matWatershedIntensities, double threshold, int maximaRadius) {
        Mat matTemp = new Mat();
        Mat strel = OpenCVTools.getCircularStructuringElement(maximaRadius);
        opencv_imgproc.dilate((Mat)matWatershedIntensities, (Mat)matTemp, (Mat)strel);
        opencv_core.compare((Mat)matWatershedIntensities, (Mat)matTemp, (Mat)matTemp, (int)0);
        opencv_imgproc.dilate((Mat)matTemp, (Mat)matTemp, (Mat)OpenCVTools.getCircularStructuringElement(2));
        Mat matWatershedSeedsBinary = matTemp;
        opencv_core.min((Mat)matWatershedSeedsBinary, (Mat)matBinary, (Mat)matWatershedSeedsBinary);
        Mat matLabels = new Mat(matWatershedIntensities.size(), 5, Scalar.ZERO);
        OpenCVTools.labelImage(matWatershedSeedsBinary, matLabels, 2);
        ProcessingCV.doWatershed(matWatershedIntensities, matLabels, threshold, true);
        opencv_core.multiply((Mat)matBinary, (Mat)matLabels, (Mat)matBinary, (double)1.0, (int)matBinary.type());
    }

    public static ImageProcessor matToImageProcessor(Mat mat) {
        if (mat.channels() != 1) {
            throw new IllegalArgumentException("Only a single-channel Mat can be converted to an ImageProcessor! Specified Mat has " + mat.channels() + " channels");
        }
        int w = mat.cols();
        int h = mat.rows();
        if (mat.depth() == 5) {
            FloatIndexer indexer = (FloatIndexer)mat.createIndexer();
            float[] pixels = new float[w * h];
            indexer.get(0L, pixels);
            return new FloatProcessor(w, h, pixels);
        }
        if (mat.depth() == 0) {
            UByteIndexer indexer = (UByteIndexer)mat.createIndexer();
            int[] pixels = new int[w * h];
            indexer.get(0L, pixels);
            ByteProcessor bp = new ByteProcessor(w, h);
            for (int i = 0; i < pixels.length; ++i) {
                bp.set(i, pixels[i]);
            }
            return bp;
        }
        if (mat.depth() == 2) {
            UShortIndexer indexer = (UShortIndexer)mat.createIndexer();
            int[] pixels = new int[w * h];
            indexer.get(0L, pixels);
            short[] shortPixels = new short[pixels.length];
            for (int i = 0; i < pixels.length; ++i) {
                shortPixels[i] = (short)pixels[i];
            }
            return new ShortProcessor(w, h, shortPixels, null);
        }
        Mat mat2 = new Mat();
        mat.convertTo(mat2, 5);
        ImageProcessor ip = OpenCVTools.matToImageProcessor(mat2);
        mat2.close();
        return ip;
    }

    public static ImagePlus matToImagePlus(Mat mat, String title) {
        if (mat.channels() == 1) {
            return new ImagePlus(title, OpenCVTools.matToImageProcessor(mat));
        }
        return OpenCVTools.matToImagePlus(title, mat);
    }

    public static ImagePlus matToImagePlus(String title, Mat ... mats) {
        ImageStack stack = null;
        int nChannels = 1;
        for (Mat mat : mats) {
            if (stack == null) {
                stack = new ImageStack(mat.cols(), mat.rows());
            } else if (mat.channels() != nChannels) {
                throw new IllegalArgumentException("Number of channels must be the same for all Mats!");
            }
            if (mat.channels() == 1) {
                ImageProcessor ip = OpenCVTools.matToImageProcessor(mat);
                stack.addSlice(ip);
                continue;
            }
            nChannels = mat.channels();
            MatVector split = new MatVector();
            opencv_core.split((Mat)mat, (MatVector)split);
            int c = 0;
            while ((long)c < split.size()) {
                stack.addSlice(OpenCVTools.matToImageProcessor(split.get((long)c)));
                ++c;
            }
        }
        ImagePlus imp = new ImagePlus(title, stack);
        imp.setDimensions(nChannels, mats.length, 1);
        return nChannels == 1 ? imp : new CompositeImage(imp);
    }

    public static double[] getGaussianDeriv(double sigma, int order, int length) {
        int n = length / 2;
        double[] kernel = new double[length];
        double denom2 = 2.0 * sigma * sigma;
        double denom = sigma * Math.sqrt(Math.PI * 2);
        switch (order) {
            case 0: {
                for (int x = -n; x < length - n; ++x) {
                    double val = Math.exp(-((double)(x * x)) / denom2);
                    kernel[x + n] = (float)(val / denom);
                }
                return kernel;
            }
            case 1: {
                denom *= sigma * sigma;
                for (int x = -n; x < length - n; ++x) {
                    double val = (double)(-x) * Math.exp(-((double)(x * x)) / denom2);
                    kernel[x + n] = (float)(val / denom);
                }
                return kernel;
            }
            case 2: {
                denom *= sigma * sigma * sigma * sigma;
                for (int x = -n; x < length - n; ++x) {
                    double val = -(sigma * sigma - (double)(x * x)) * Math.exp(-((double)(x * x)) / denom2);
                    kernel[x + n] = (float)(val / denom);
                }
                return kernel;
            }
        }
        throw new IllegalArgumentException("Order must be <= 2");
    }

    public static Mat getGaussianDerivKernel(double sigma, int order, boolean doColumn) {
        int n = (int)(sigma * 4.0);
        int len = n * 2 + 1;
        double[] kernel = OpenCVTools.getGaussianDeriv(sigma, order, len);
        Mat mat = doColumn ? new Mat(1, kernel.length, 6) : new Mat(kernel.length, 1, 6);
        DoubleIndexer indexer = (DoubleIndexer)mat.createIndexer();
        indexer.put(0L, kernel);
        indexer.release();
        return mat;
    }

    static int ensureInRange(int ind, int max, int border) {
        if (ind < 0) {
            switch (border) {
                case 2: {
                    return OpenCVTools.ensureInRange(-(ind + 1), max, border);
                }
                case 4: {
                    return OpenCVTools.ensureInRange(-ind, max, border);
                }
            }
            return 0;
        }
        if (ind >= max) {
            switch (border) {
                case 2: {
                    return OpenCVTools.ensureInRange(2 * max - ind - 1, max, border);
                }
                case 4: {
                    return OpenCVTools.ensureInRange(2 * max - ind - 2, max, border);
                }
            }
            return max - 1;
        }
        return ind;
    }

    static void weightedSum(List<Mat> mats, double[] weights, Mat dest) {
        boolean isFirst = true;
        for (int i = 0; i < weights.length; ++i) {
            double w = weights[i];
            if (w == 0.0) continue;
            if (isFirst) {
                dest.put(opencv_core.multiply((Mat)mats.get(i), (double)w));
                isFirst = false;
                continue;
            }
            opencv_core.scaleAdd((Mat)mats.get(i), (double)w, (Mat)dest, (Mat)dest);
        }
        if (isFirst) {
            dest.create(mats.get(0).size(), mats.get(0).type());
            dest.put(Scalar.ZERO);
        }
    }

    public static Mat filterSingleZ(List<Mat> mats, double[] kernel, int ind3D, int border) {
        int n = mats.size();
        int halfSize = kernel.length / 2;
        int startInd = ind3D - halfSize;
        int endInd = startInd + kernel.length;
        double[] weights = new double[mats.size()];
        int k = 0;
        for (int i = startInd; i < endInd; ++i) {
            int ind;
            int n2 = ind = OpenCVTools.ensureInRange(i, n, border);
            weights[n2] = weights[n2] + kernel[k];
            ++k;
        }
        Mat result = new Mat();
        OpenCVTools.weightedSum(mats, weights, result);
        return result;
    }

    public static List<Mat> filterZ(List<Mat> mats, Mat kernelZ, int ind3D, int border) {
        boolean doWeightedSums = true;
        if (doWeightedSums) {
            int ks = (int)kernelZ.total();
            double[] kernelArray = new double[ks];
            DoubleIndexer idx = (DoubleIndexer)kernelZ.createIndexer();
            idx.get(0L, kernelArray);
            idx.release();
            if (ind3D >= 0) {
                Mat result = OpenCVTools.filterSingleZ(mats, kernelArray, ind3D, border);
                return Arrays.asList(result);
            }
            ArrayList<Mat> output = new ArrayList<Mat>();
            for (int i = 0; i < mats.size(); ++i) {
                Mat result = OpenCVTools.filterSingleZ(mats, kernelArray, i, border);
                output.add(result);
            }
            return output;
        }
        Mat[] columns = new Mat[mats.size()];
        int nRows = 0;
        for (int i = 0; i < mats.size(); ++i) {
            Mat mat = mats.get(i);
            nRows = mat.rows();
            columns[i] = mat.reshape(mat.channels(), mat.rows() * mat.cols());
        }
        Mat matConcatZ = new Mat();
        opencv_core.hconcat((MatVector)new MatVector(columns), (Mat)matConcatZ);
        if (kernelZ.rows() > 1) {
            kernelZ = kernelZ.t().asMat();
        }
        opencv_imgproc.filter2D((Mat)matConcatZ, (Mat)matConcatZ, (int)5, (Mat)kernelZ, null, (double)0.0, (int)border);
        int start = 0;
        int end = mats.size();
        if (ind3D >= 0) {
            start = ind3D;
            end = ind3D + 1;
        }
        ArrayList<Mat> output = new ArrayList<Mat>();
        for (int i = start; i < end; ++i) {
            output.add(matConcatZ.col(i).clone().reshape(matConcatZ.channels(), nRows));
        }
        return output;
    }

    public static List<Mat> extractZStack(ImageServer<BufferedImage> server, RegionRequest request, int zMin, int zMax) throws IOException {
        ArrayList<Mat> list = new ArrayList<Mat>();
        for (int z = zMin; z < zMax; ++z) {
            RegionRequest request2 = RegionRequest.createInstance((String)server.getPath(), (double)request.getDownsample(), (int)request.getX(), (int)request.getY(), (int)request.getWidth(), (int)request.getHeight(), (int)z, (int)request.getT());
            BufferedImage img = (BufferedImage)server.readRegion(request2);
            list.add(OpenCVTools.imageToMat(img));
        }
        return list;
    }

    public static List<Mat> extractZStack(ImageServer<BufferedImage> server, RegionRequest request) throws IOException {
        return OpenCVTools.extractZStack(server, request, 0, server.nZSlices());
    }

    public static Mat crop(Mat mat, int x, int y, int width, int height) {
        try (Rect rect = new Rect(x, y, width, height);){
            Mat temp = mat.apply(rect);
            Mat mat2 = temp.clone();
            return mat2;
        }
    }

    public static Mat crop(Mat mat, Padding padding) {
        return OpenCVTools.crop(mat, padding.getX1(), padding.getY1(), mat.cols() - padding.getXSum(), mat.rows() - padding.getYSum());
    }

    public static Mat applyTiled(Function<Mat, Mat> fun, Mat mat, int tileWidth, int tileHeight, Padding padding, int borderType) {
        int top = 0;
        int bottom = 0;
        int left = 0;
        int right = 0;
        boolean doPad = false;
        Mat matResult = new Mat();
        int width = mat.cols();
        int height = mat.rows();
        try (PointerScope scope = new PointerScope();){
            if (mat.cols() > tileWidth) {
                Padding paddingY = Padding.getPadding((int)0, (int)0, (int)padding.getY1(), (int)padding.getY2());
                ArrayList<Mat> horizontal = new ArrayList<Mat>();
                int xStart = 0;
                boolean lastTile = false;
                while (!lastTile) {
                    int xEnd = Math.min(xStart + tileWidth, width);
                    Mat matTemp = OpenCVTools.applyTiled(fun, mat.colRange(xStart, xEnd).clone(), tileWidth, tileHeight, paddingY, borderType);
                    int pad1 = xStart == 0 ? 0 : padding.getX1();
                    int pad2 = 0;
                    if (xEnd >= width) {
                        lastTile = true;
                        pad2 = xEnd - width;
                    } else {
                        pad2 = padding.getX2();
                    }
                    OpenCVTools.cropX(matTemp, pad1, pad2);
                    if (matTemp.cols() == 0) break;
                    xStart = xEnd - padding.getX1() - pad2;
                    horizontal.add(matTemp);
                }
                opencv_core.hconcat((MatVector)new MatVector((Mat[])horizontal.toArray(Mat[]::new)), (Mat)matResult);
                Mat xEnd = matResult;
                return xEnd;
            }
            if (mat.rows() > tileHeight) {
                Padding paddingX = Padding.getPadding((int)padding.getX1(), (int)padding.getX2(), (int)0, (int)0);
                ArrayList<Mat> vertical = new ArrayList<Mat>();
                int yStart = 0;
                boolean lastTile = false;
                while (!lastTile) {
                    int yEnd = Math.min(yStart + tileHeight, height);
                    Mat matTemp = OpenCVTools.applyTiled(fun, mat.rowRange(yStart, yEnd).clone(), tileWidth, tileHeight, paddingX, borderType);
                    int pad1 = yStart == 0 ? 0 : padding.getY1();
                    int pad2 = 0;
                    if (yEnd >= height) {
                        lastTile = true;
                        pad2 = yEnd - height;
                    } else {
                        pad2 = padding.getY2();
                    }
                    OpenCVTools.cropY(matTemp, pad1, pad2);
                    yStart = yEnd - padding.getY1() - pad2;
                    vertical.add(matTemp);
                }
                opencv_core.vconcat((MatVector)new MatVector((Mat[])vertical.toArray(Mat[]::new)), (Mat)matResult);
                Mat mat2 = matResult;
                return mat2;
            }
            if (mat.cols() < tileWidth || mat.rows() < tileHeight) {
                top = (tileHeight - mat.rows()) / 2;
                left = (tileWidth - mat.cols()) / 2;
                bottom = tileHeight - mat.rows() - top;
                right = tileWidth - mat.cols() - left;
                Mat matPadded = new Mat();
                opencv_core.copyMakeBorder((Mat)mat, (Mat)matPadded, (int)top, (int)bottom, (int)left, (int)right, (int)borderType);
                mat = matPadded;
                doPad = true;
            }
            matResult.put(fun.apply(mat));
            int nRows = mat.rows();
            int nCols = mat.cols();
            if (matResult.rows() != nRows || matResult.cols() != nCols) {
                logger.warn("Resizing tiled image from {}x{} to {}x{}", new Object[]{matResult.cols(), matResult.rows(), nRows, nCols});
                opencv_imgproc.resize((Mat)matResult, (Mat)matResult, (Size)mat.size());
            }
            if (doPad) {
                matResult.put(OpenCVTools.crop(matResult, left, top, matResult.cols() - right - left, matResult.rows() - top - bottom));
            }
        }
        return matResult;
    }

    private static void cropX(Mat mat, int x1, int x2) {
        if (x1 == 0 && x2 == 0) {
            return;
        }
        if (x1 < 0 || x2 < 0) {
            throw new IllegalArgumentException("Cannot crop x by a negative amount (" + x1 + ", " + x2 + ")");
        }
        int width = mat.cols();
        mat.put(mat.colRange(x1, width - x2));
    }

    private static void cropY(Mat mat, int y1, int y2) {
        if (y1 == 0 && y2 == 0) {
            return;
        }
        if (y1 < 0 || y2 < 0) {
            throw new IllegalArgumentException("Cannot crop y by a negative amount (" + y1 + ", " + y2 + ")");
        }
        int height = mat.rows();
        mat.put(mat.rowRange(y1, height - y2));
    }

    private static Mat radiusToStrel(int radius) {
        try (Size size = new Size(radius * 2 + 1, radius * 2 + 1);){
            Mat mat = opencv_imgproc.getStructuringElement((int)2, (Size)size);
            return mat;
        }
    }

    public static void sepFilter2D(Mat mat, Mat kx, Mat ky) {
        OpenCVTools.sepFilter2D(mat, kx, ky, DEFAULT_BORDER_TYPE);
    }

    public static void sepFilter2D(Mat mat, Mat kx, Mat ky, int borderType) {
        opencv_imgproc.sepFilter2D((Mat)mat, (Mat)mat, (int)-1, (Mat)kx, (Mat)ky, null, (double)0.0, (int)borderType);
    }

    public static void filter2D(Mat mat, Mat kernel) {
        OpenCVTools.filter2D(mat, kernel, DEFAULT_BORDER_TYPE);
    }

    public static void filter2D(Mat mat, Mat kernel, int borderType) {
        opencv_imgproc.filter2D((Mat)mat, (Mat)mat, (int)-1, (Mat)kernel, null, (double)0.0, (int)borderType);
    }

    public static void meanFilter(Mat mat, int radius, int borderType) {
        Mat kernel = OpenCVTools.createDisk(radius, true);
        OpenCVTools.filter2D(mat, kernel, borderType);
    }

    public static void meanFilter(Mat mat, int radius) {
        OpenCVTools.meanFilter(mat, radius, DEFAULT_BORDER_TYPE);
    }

    public static void sumFilter(Mat mat, int radius, int borderType) {
        Mat kernel = OpenCVTools.createDisk(radius, false);
        OpenCVTools.filter2D(mat, kernel, borderType);
    }

    public static void sumFilter(Mat mat, int radius) {
        OpenCVTools.sumFilter(mat, radius, DEFAULT_BORDER_TYPE);
    }

    public static void varianceFilter(Mat mat, int radius, int borderType) {
        Mat kernel = OpenCVTools.createDisk(radius, true);
        Mat matSquared = mat.mul(mat).asMat();
        opencv_imgproc.filter2D((Mat)matSquared, (Mat)matSquared, (int)-1, (Mat)kernel, null, (double)0.0, (int)borderType);
        opencv_imgproc.filter2D((Mat)mat, (Mat)mat, (int)-1, (Mat)kernel, null, (double)0.0, (int)borderType);
        mat.put(opencv_core.subtract((Mat)matSquared, (MatExpr)mat.mul(mat)));
        matSquared.close();
    }

    public static void varianceFilter(Mat mat, int radius) {
        OpenCVTools.varianceFilter(mat, radius, DEFAULT_BORDER_TYPE);
    }

    public static void stdDevFilter(Mat mat, int radius, int borderType) {
        OpenCVTools.varianceFilter(mat, radius, borderType);
        opencv_core.sqrt((Mat)mat, (Mat)mat);
    }

    public static void stdDevFilter(Mat mat, int radius) {
        OpenCVTools.stdDevFilter(mat, radius, DEFAULT_BORDER_TYPE);
    }

    public static void maximumFilter(Mat mat, int radius) {
        Mat strel = OpenCVTools.radiusToStrel(radius);
        opencv_imgproc.dilate((Mat)mat, (Mat)mat, (Mat)strel);
    }

    public static void minimumFilter(Mat mat, int radius) {
        Mat strel = OpenCVTools.radiusToStrel(radius);
        opencv_imgproc.erode((Mat)mat, (Mat)mat, (Mat)strel);
    }

    public static void closingFilter(Mat mat, int radius) {
        Mat strel = OpenCVTools.radiusToStrel(radius);
        opencv_imgproc.morphologyEx((Mat)mat, (Mat)mat, (int)3, (Mat)strel);
    }

    public static void openingFilter(Mat mat, int radius) {
        Mat strel = OpenCVTools.radiusToStrel(radius);
        opencv_imgproc.morphologyEx((Mat)mat, (Mat)mat, (int)2, (Mat)strel);
    }

    public static void gaussianFilter(Mat mat, double sigma, int borderType) {
        int s = (int)Math.ceil(sigma * 4.0) * 2 + 1;
        try (Size size = new Size(s, s);){
            opencv_imgproc.GaussianBlur((Mat)mat, (Mat)mat, (Size)size, (double)sigma, (double)sigma, (int)borderType);
        }
    }

    public static void gaussianFilter(Mat mat, double sigma) {
        OpenCVTools.gaussianFilter(mat, sigma, DEFAULT_BORDER_TYPE);
    }

    public static Mat label(Mat matBinary, int connectivity) {
        Mat matLabels = new Mat();
        OpenCVTools.label(matBinary, matLabels, connectivity);
        return matLabels;
    }

    public static int label(Mat matBinary, Mat matLabels, int connectivity) {
        int alg = connectivity == 8 ? 1 : 3;
        return opencv_imgproc.connectedComponentsWithAlgorithm((Mat)matBinary, (Mat)matLabels, (int)connectivity, (int)4, (int)alg) - 1;
    }

    public static Map<Number, ROI> createROIs(Mat matLabels, RegionRequest region, int minLabel, int maxLabel) {
        if (matLabels.channels() != 1) {
            throw new IllegalArgumentException("Input to createROIs must be a single-channel Mat - current input has " + matLabels.channels() + " channels");
        }
        SimpleImage image = OpenCVTools.matToSimpleImage(matLabels, 0);
        return ContourTracing.createROIs((SimpleImage)image, (RegionRequest)region, (int)minLabel, (int)maxLabel);
    }

    public static List<PathObject> createDetections(Mat matLabels, RegionRequest region, int minLabel, int maxLabel) {
        SimpleImage image = OpenCVTools.matToSimpleImage(matLabels, 0);
        return ContourTracing.createDetections((SimpleImage)image, (RegionRequest)region, (int)minLabel, (int)maxLabel);
    }

    public static List<PathObject> createAnnotations(Mat matLabels, RegionRequest region, int minLabel, int maxLabel) {
        SimpleImage image = OpenCVTools.matToSimpleImage(matLabels, 0);
        return ContourTracing.createAnnotations((SimpleImage)image, (RegionRequest)region, (int)minLabel, (int)maxLabel);
    }

    public static List<PathObject> createObjects(Mat matLabels, RegionRequest region, int minLabel, int maxLabel, BiFunction<ROI, Number, PathObject> creator) {
        SimpleImage image = OpenCVTools.matToSimpleImage(matLabels, 0);
        return ContourTracing.createObjects((SimpleImage)image, (RegionRequest)region, (int)minLabel, (int)maxLabel, creator);
    }

    public static List<PathObject> createCells(Mat matLabelsNuclei, Mat matLabelsCells, RegionRequest region, int minLabel, int maxLabel) {
        SimpleImage imageNuclei = OpenCVTools.matToSimpleImage(matLabelsNuclei, 0);
        SimpleImage imageCells = OpenCVTools.matToSimpleImage(matLabelsCells, 0);
        return ContourTracing.createCells((SimpleImage)imageNuclei, (SimpleImage)imageCells, (RegionRequest)region, (int)minLabel, (int)maxLabel);
    }

    private static MaxLabelStat[] updateMaxLabelStats(Mat mat, Mat matLabels, int nLabels, int connectivity) {
        int n = nLabels;
        MaxLabelStat[] stats = new MaxLabelStat[n + 1];
        for (int i = 0; i <= n; ++i) {
            stats[i] = new MaxLabelStat(i);
        }
        int h = mat.rows();
        int w = mat.cols();
        long[] inds = new long[3];
        try (Indexer idx = mat.createIndexer();
             IntIndexer idxLabels = (IntIndexer)matLabels.createIndexer();){
            for (int y = 1; y < h - 1; ++y) {
                for (int x = 1; x < w - 1; ++x) {
                    int lab = idxLabels.get((long)y, (long)x);
                    double val = OpenCVTools.getValue(idx, y, x, 0L, inds);
                    stats[lab].inside(val);
                    for (int y2 = y - 1; y2 <= y + 1; ++y2) {
                        for (int x2 = x - 1; x2 <= x + 1; ++x2) {
                            int lab2;
                            if (x == x2 && y == y2 || connectivity == 4 && x != x2 && y != y2 || (lab2 = idxLabels.get((long)y2, (long)x2)) == lab) continue;
                            stats[lab2].boundary(val);
                        }
                    }
                }
            }
        }
        return stats;
    }

    public static Mat findRegionalMaxima(Mat mat) {
        OpenCVTools.checkSingleChannel(mat);
        int connectivity = 8;
        Mat matMax = new Mat();
        Mat kernel = Mat.ones((int)3, (int)3, (int)opencv_core.CV_32FC1).asMat();
        opencv_imgproc.dilate((Mat)mat, (Mat)matMax, (Mat)kernel);
        kernel.close();
        matMax.put(opencv_core.equals((Mat)mat, (Mat)matMax));
        Mat matLabels = OpenCVTools.label(matMax, connectivity);
        int nLabels = (int)Math.ceil(OpenCVTools.maximum(matLabels));
        MaxLabelStat[] stats = OpenCVTools.updateMaxLabelStats(mat, matLabels, nLabels, connectivity);
        Set nonMaxima = Arrays.stream(stats).filter(s -> s.maxBoundary >= s.maxInside).map(s -> s.label).collect(Collectors.toSet());
        if (!nonMaxima.isEmpty()) {
            for (Integer nonMax : nonMaxima) {
                OpenCVTools.replaceValues(matLabels, nonMax.intValue(), 0.0);
            }
        }
        return matLabels;
    }

    public static String physicalMemory() {
        long physicalBytes = Pointer.availablePhysicalBytes();
        double physicalMB = (double)physicalBytes / 1048576.0;
        double physicalPercent = (double)physicalBytes / (double)Pointer.maxPhysicalBytes() * 100.0;
        return "Physical memory: " + GeneralTools.formatNumber((double)physicalMB, (int)1) + " MB (" + GeneralTools.formatNumber((double)physicalPercent, (int)1) + " %)";
    }

    public static String trackedMemory() {
        double trackedMB = (double)Pointer.totalBytes() / 1048576.0;
        long nPointers = Pointer.totalCount();
        return "Tracked memory: " + GeneralTools.formatNumber((double)trackedMB, (int)1) + " MB (" + nPointers + " pointers)";
    }

    public static String memoryReport(CharSequence delimiter) {
        return String.join(delimiter, OpenCVTools.physicalMemory(), OpenCVTools.trackedMemory());
    }

    public static Collection<IndexedPixel> findMaxima(Mat mat, Mat mask) {
        Mat matMax = OpenCVTools.findRegionalMaxima(mat);
        List<IndexedPixel> points = OpenCVTools.labelsToPoints(matMax);
        try (Indexer idx = mat.createIndexer();){
            for (IndexedPixel p2 : points) {
                p2.value = idx.getDouble(p2.inds);
            }
        }
        Collections.sort(points, Comparator.comparingDouble(p -> p.getValue()).reversed());
        if (mask == null) {
            return points;
        }
        idx = mask.createIndexer();
        try {
            List<IndexedPixel> list = points.stream().filter(p -> idx.getDouble(p.inds) != 0.0).toList();
            return list;
        }
        finally {
            if (idx != null) {
                idx.close();
            }
        }
    }

    private static void checkSingleChannel(Mat mat) {
        if (mat.channels() != 1) {
            throw new IllegalArgumentException("Method requires a single-channel mat, but input has " + mat.channels() + " channels");
        }
    }

    public static List<IndexedPixel> getMaskedPixels(Mat mat, Mat mask) {
        OpenCVTools.checkSingleChannel(mat);
        OpenCVTools.checkSingleChannel(mask);
        ArrayList<IndexedPixel> pixels = new ArrayList<IndexedPixel>();
        int height = mat.rows();
        int width = mat.cols();
        long[] inds = new long[2];
        try (Indexer idx = mat.createIndexer();
             Indexer idxMask = mask.createIndexer();){
            for (int y = 0; y < height; ++y) {
                inds[0] = y;
                for (int x = 0; x < width; ++x) {
                    inds[1] = x;
                    if (idxMask.getDouble(inds) == 0.0) continue;
                    pixels.add(new IndexedPixel(x, y, idx.getDouble(inds)));
                }
            }
        }
        return pixels;
    }

    public static float[] extractMaskedFloats(Mat input, Mat mask, int channel) {
        try (PointerScope scope = new PointerScope();){
            float[] fArray;
            Mat maskChannel;
            Mat inputChannel;
            if (input.channels() == 1) {
                inputChannel = input;
            } else {
                inputChannel = new Mat();
                opencv_core.extractChannel((Mat)input, (Mat)inputChannel, (int)channel);
            }
            if (mask.channels() == 1) {
                maskChannel = mask;
            } else {
                maskChannel = new Mat();
                opencv_core.extractChannel((Mat)mask, (Mat)maskChannel, (int)channel);
            }
            if (inputChannel.rows() != maskChannel.rows()) {
                throw new IllegalArgumentException("The input and the mask don't have the same number of rows");
            }
            if (inputChannel.cols() != maskChannel.cols()) {
                throw new IllegalArgumentException("The input and the mask don't have the same number of columns");
            }
            int height = inputChannel.rows();
            int width = inputChannel.cols();
            float[] output = new float[height * width];
            long[] indices = new long[2];
            int outputIndex = 0;
            try (Indexer idx = inputChannel.createIndexer();
                 Indexer idxMask = maskChannel.createIndexer();){
                for (int y = 0; y < height; ++y) {
                    indices[0] = y;
                    for (int x = 0; x < width; ++x) {
                        indices[1] = x;
                        if (idxMask.getDouble(indices) == 0.0) continue;
                        output[outputIndex] = (float)idx.getDouble(indices);
                        ++outputIndex;
                    }
                }
            }
            if (outputIndex < output.length) {
                fArray = Arrays.copyOf(output, outputIndex);
                return fArray;
            }
            fArray = output;
            return fArray;
        }
    }

    public static double[] extractMaskedDoubles(Mat input, Mat mask, int channel) {
        try (PointerScope scope = new PointerScope();){
            double[] dArray;
            Mat maskChannel;
            Mat inputChannel;
            if (input.channels() == 1) {
                inputChannel = input;
            } else {
                inputChannel = new Mat();
                opencv_core.extractChannel((Mat)input, (Mat)inputChannel, (int)channel);
            }
            if (mask.channels() == 1) {
                maskChannel = mask;
            } else {
                maskChannel = new Mat();
                opencv_core.extractChannel((Mat)mask, (Mat)maskChannel, (int)channel);
            }
            if (inputChannel.rows() != maskChannel.rows()) {
                throw new IllegalArgumentException("The input and the mask don't have the same number of rows");
            }
            if (inputChannel.cols() != maskChannel.cols()) {
                throw new IllegalArgumentException("The input and the mask don't have the same number of columns");
            }
            int height = inputChannel.rows();
            int width = inputChannel.cols();
            double[] output = new double[height * width];
            long[] indices = new long[2];
            int outputIndex = 0;
            try (Indexer idx = inputChannel.createIndexer();
                 Indexer idxMask = maskChannel.createIndexer();){
                for (int y = 0; y < height; ++y) {
                    indices[0] = y;
                    for (int x = 0; x < width; ++x) {
                        indices[1] = x;
                        if (idxMask.getDouble(indices) == 0.0) continue;
                        output[outputIndex] = idx.getDouble(indices);
                        ++outputIndex;
                    }
                }
            }
            if (outputIndex < output.length) {
                dArray = Arrays.copyOf(output, outputIndex);
                return dArray;
            }
            dArray = output;
            return dArray;
        }
    }

    public static Mat shrinkLabels(Mat mat) {
        if (mat.channels() != 1) {
            throw new IllegalArgumentException("shrinkLabels requires a single-channel mat, but input has " + mat.channels() + " channels");
        }
        List<IndexedPixel> points = OpenCVTools.labelsToPoints(mat);
        Mat mat2 = new Mat(mat.rows(), mat.cols(), mat.type(), Scalar.ZERO);
        try (IntIndexer idx2 = (IntIndexer)mat2.createIndexer();){
            for (IndexedPixel p : points) {
                idx2.putDouble(p.inds, p.getValue());
            }
        }
        return mat2;
    }

    private static List<IndexedPixel> labelsToPoints(Mat mat) {
        if (mat.channels() != 1) {
            throw new IllegalArgumentException("labelsToPoints requires a single-channel mat, but input has " + mat.channels() + " channels");
        }
        int w = mat.cols();
        int h = mat.rows();
        ArrayList<IndexedPixel> pixels = new ArrayList<IndexedPixel>();
        try (IntIndexer idx = (IntIndexer)mat.createIndexer();){
            for (int y = 0; y < h; ++y) {
                for (int x = 0; x < w; ++x) {
                    int lab = idx.get((long)y, (long)x);
                    if (lab == 0) continue;
                    pixels.add(new IndexedPixel(x, y, lab));
                }
            }
        }
        Map<Double, List<IndexedPixel>> groups = pixels.stream().collect(Collectors.groupingBy(p -> p.value));
        return groups.values().stream().map(l -> OpenCVTools.closestToCentroid(l)).toList();
    }

    private static IndexedPixel closestToCentroid(Collection<IndexedPixel> pixels) {
        if (pixels.size() <= 2) {
            return pixels.iterator().next();
        }
        double cx = pixels.stream().mapToDouble(p -> p.getX()).average().getAsDouble();
        double cy = pixels.stream().mapToDouble(p -> p.getY()).average().getAsDouble();
        IndexedPixel closest = null;
        double minDistanceSq = Double.POSITIVE_INFINITY;
        for (IndexedPixel p2 : pixels) {
            double dy;
            double dx = cx - (double)p2.getX();
            double dist = dx * dx + (dy = cy - (double)p2.getY()) * dy;
            if (!(dist < minDistanceSq)) continue;
            minDistanceSq = dist;
            closest = p2;
        }
        return closest;
    }

    private static double getValue(Indexer idx, long i, long j, long k, long[] inds) {
        inds[0] = i;
        inds[1] = j;
        inds[2] = k;
        return idx.getDouble(inds);
    }

    private static class MaxLabelStat {
        int label;
        double minInside = Double.POSITIVE_INFINITY;
        double maxInside = Double.NEGATIVE_INFINITY;
        double minBoundary = Double.POSITIVE_INFINITY;
        double maxBoundary = Double.NEGATIVE_INFINITY;

        MaxLabelStat(int label) {
            this.label = label;
        }

        public void inside(double value) {
            if (value < this.minInside) {
                this.minInside = value;
            }
            if (value > this.maxInside) {
                this.maxInside = value;
            }
        }

        public void boundary(double value) {
            if (value < this.minBoundary) {
                this.minBoundary = value;
            }
            if (value > this.maxBoundary) {
                this.maxBoundary = value;
            }
        }
    }

    public static class IndexedPixel {
        private long[] inds;
        private double value;

        IndexedPixel(long[] inds, double value) {
            this.inds = (long[])inds.clone();
            this.value = value;
        }

        IndexedPixel(int x, int y, double value) {
            this.inds = new long[]{y, x};
            this.value = value;
        }

        public int getX() {
            return (int)this.inds[1];
        }

        public int getY() {
            return (int)this.inds[0];
        }

        public int getC() {
            return this.inds.length < 3 ? 0 : (int)this.inds[2];
        }

        public long[] getInds() {
            return (long[])this.inds.clone();
        }

        public double getValue() {
            return this.value;
        }

        public double getValue(Indexer idx) {
            return idx.getDouble(this.inds);
        }

        public long distanceSq(IndexedPixel p2) {
            if (this.inds.length != p2.inds.length) {
                throw new IllegalArgumentException("Cannot compute distance between points with different numbers of indices!");
            }
            long sumSq = 0L;
            for (int i = 0; i < this.inds.length; ++i) {
                long d = this.inds[i] - p2.inds[i];
                sumSq += d * d;
            }
            return sumSq;
        }
    }
}

