/*
 * Decompiled with CFR 0.152.
 */
package qupath.opencv.ops;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.invoke.CallSite;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.math3.util.FastMath;
import org.bytedeco.javacpp.Pointer;
import org.bytedeco.javacpp.PointerScope;
import org.bytedeco.javacpp.indexer.DoubleIndexer;
import org.bytedeco.javacpp.indexer.FloatIndexer;
import org.bytedeco.javacpp.indexer.Indexer;
import org.bytedeco.opencv.global.opencv_core;
import org.bytedeco.opencv.global.opencv_imgproc;
import org.bytedeco.opencv.opencv_core.Mat;
import org.bytedeco.opencv.opencv_core.MatExpr;
import org.bytedeco.opencv.opencv_core.MatVector;
import org.bytedeco.opencv.opencv_core.Rect;
import org.bytedeco.opencv.opencv_core.Scalar;
import org.bytedeco.opencv.opencv_core.Size;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.color.ColorDeconvolutionStains;
import qupath.lib.common.ColorTools;
import qupath.lib.common.GeneralTools;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ColorTransforms;
import qupath.lib.images.servers.ImageChannel;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.images.servers.PixelType;
import qupath.lib.images.servers.ServerTools;
import qupath.lib.io.GsonTools;
import qupath.lib.io.UriResource;
import qupath.lib.regions.Padding;
import qupath.lib.regions.RegionRequest;
import qupath.opencv.dnn.AbstractDnnModel;
import qupath.opencv.dnn.DnnModel;
import qupath.opencv.dnn.DnnShape;
import qupath.opencv.ml.FeaturePreprocessor;
import qupath.opencv.ml.OpenCVClassifiers;
import qupath.opencv.ops.ImageDataOp;
import qupath.opencv.ops.ImageDataServer;
import qupath.opencv.ops.ImageOp;
import qupath.opencv.ops.ImageOpServer;
import qupath.opencv.tools.LocalNormalization;
import qupath.opencv.tools.MultiscaleFeatures;
import qupath.opencv.tools.OpenCVTools;

public class ImageOps {
    private static final Logger logger = LoggerFactory.getLogger(ImageOps.class);
    private static List<String> LEGACY_CATEGORIES = Arrays.asList("op.filters.", "op.ml.", "op.normalize.", "op.threshold.");
    private static GsonTools.SubTypeAdapterFactory<ImageOp> factoryOps = GsonTools.createSubTypeAdapterFactory(ImageOp.class, (String)"type");
    private static GsonTools.SubTypeAdapterFactory<ImageDataOp> factoryDataOps;

    private static <T> void registerTypes(GsonTools.SubTypeAdapterFactory<T> factory, Class<T> factoryType, Class<?> cls, String base) {
        OpType annotation = cls.getAnnotation(OpType.class);
        if (annotation != null) {
            String annotationValue = annotation.value();
            if (!annotationValue.isEmpty()) {
                base = (String)base + "." + annotation.value();
            }
            if (factoryType.isAssignableFrom(cls)) {
                logger.trace("Registering {} for class {}", base, cls);
                factory.registerSubtype(cls, (String)base);
                for (String cat : LEGACY_CATEGORIES) {
                    if (!((String)base).startsWith(cat)) continue;
                    String alias = ((String)base).replace(cat, "op.");
                    logger.trace("Registering alias {} for class {}", (Object)alias, cls);
                    factory.registerAlias(cls, alias);
                }
            } else {
                logger.trace("Cannot register {} for factory type {}", cls, factoryType);
            }
        } else if (!ImageOps.class.equals(cls)) {
            logger.trace("Skipping unannotated class {}", cls);
            return;
        }
        for (Class<?> c : cls.getDeclaredClasses()) {
            ImageOps.registerTypes(factory, factoryType, c, (String)base);
        }
    }

    public static void registerOp(Class<? extends ImageOp> cls, String label) {
        Objects.requireNonNull(cls);
        Objects.requireNonNull(label);
        logger.debug("Registering ImageOp {} with label {}", cls, (Object)label);
        if (!label.startsWith("op.")) {
            logger.warn("ImageOp label '{}' does not begin with 'op.'", (Object)label);
        }
        factoryOps.registerSubtype(cls, label);
    }

    public static void registerDataOp(Class<? extends ImageDataOp> cls, String label) {
        Objects.requireNonNull(cls);
        Objects.requireNonNull(label);
        logger.debug("Registering ImageOp {} with label {}", cls, (Object)label);
        if (!label.startsWith("data.op.")) {
            logger.warn("ImageDataOp label '{}' does not begin with 'data.op.'", (Object)label);
        }
        factoryDataOps.registerSubtype(cls, label);
    }

    public static ImageDataServer<BufferedImage> buildServer(ImageData<BufferedImage> imageData, ImageDataOp dataOp, PixelCalibration resolution) {
        return ImageOps.buildServer(imageData, dataOp, resolution, 512, 512);
    }

    public static ImageDataServer<BufferedImage> buildServer(ImageData<BufferedImage> imageData, ImageDataOp dataOp, PixelCalibration resolution, int tileWidth, int tileHeight) {
        double downsample;
        if (resolution.unitsMatch2D() && "px".equals(resolution.getPixelWidthUnit())) {
            downsample = resolution.getAveragedPixelSize().doubleValue();
        } else {
            PixelCalibration cal = imageData.getServer().getPixelCalibration();
            if (!resolution.getPixelWidthUnit().equals(cal.getPixelWidthUnit()) || !resolution.getPixelHeightUnit().equals(cal.getPixelHeightUnit())) {
                logger.warn("Resolution and pixel calibration units do not match! {} x {} vs {} x {}", new Object[]{resolution.getPixelWidthUnit(), resolution.getPixelHeightUnit(), cal.getPixelWidthUnit(), cal.getPixelHeightUnit()});
            }
            downsample = resolution.getAveragedPixelSize().doubleValue() / cal.getAveragedPixelSize().doubleValue();
        }
        return new ImageOpServer(imageData, downsample, tileWidth, tileHeight, dataOp);
    }

    public static ImageDataOp buildImageDataOp(ColorTransforms.ColorTransform ... inputChannels) {
        if (inputChannels == null || inputChannels.length == 0) {
            return new DefaultImageDataOp(null);
        }
        return new ChannelImageDataOp(null, inputChannels);
    }

    public static ImageDataOp buildImageDataOp(Collection<? extends ColorTransforms.ColorTransform> inputChannels) {
        return ImageOps.buildImageDataOp((ColorTransforms.ColorTransform[])inputChannels.toArray(ColorTransforms.ColorTransform[]::new));
    }

    public static Mat padAndApply(ImageOp op, Mat mat, int padType) {
        Padding padding = op.getPadding();
        if (padding.isEmpty()) {
            return op.apply(mat);
        }
        opencv_core.copyMakeBorder((Mat)mat, (Mat)mat, (int)padding.getY1(), (int)padding.getY2(), (int)padding.getX1(), (int)padding.getX2(), (int)padType);
        return op.apply(mat);
    }

    public static Mat padAndApply(ImageOp op, Mat mat) {
        return ImageOps.padAndApply(op, mat, 2);
    }

    private static Mat doPrediction(DnnModel model, Mat mat, String inputName, String ... outputNames) {
        Mat matResult = new Mat();
        try (PointerScope scope = new PointerScope();){
            Map<String, Mat> output = model.predict(Map.of(inputName, mat));
            if (!output.isEmpty()) {
                if (outputNames.length == 0 || outputNames.length == 1 && output.containsKey(outputNames[0])) {
                    matResult.put(output.values().iterator().next());
                } else {
                    Mat[] tempArray = new Mat[outputNames.length];
                    for (int i = 0; i < outputNames.length; ++i) {
                        String name = outputNames[i];
                        if (!output.containsKey(name)) {
                            throw new RuntimeException(String.format("Unable to find output '%s' in %s", name, model));
                        }
                        tempArray[i] = output.get(name);
                    }
                    opencv_core.merge((MatVector)new MatVector(tempArray), (Mat)matResult);
                }
            }
            scope.deallocate();
        }
        return matResult;
    }

    static void rescaleChannelsToProbabilities(Mat matRawInput, Mat matProbabilities, double maxValue, boolean doSoftmax) {
        if (matProbabilities == null) {
            matProbabilities = new Mat();
        }
        if (matRawInput != matProbabilities && matRawInput.rows() != matProbabilities.rows() && matRawInput.cols() != matProbabilities.cols()) {
            if (matProbabilities.empty()) {
                matProbabilities.create(matRawInput.rows(), matRawInput.cols(), matRawInput.type());
            } else {
                matProbabilities.create(matRawInput.rows(), matRawInput.cols(), matProbabilities.type());
            }
        }
        int warnNegativeValues = 0;
        Indexer idxInput = matRawInput.createIndexer();
        Indexer idxOutput = matProbabilities.createIndexer();
        long[] inds = new long[2];
        long rows = idxInput.size(0);
        long cols = idxOutput.size(1);
        double[] vals = new double[(int)cols];
        for (long r = 0L; r < rows; ++r) {
            inds[0] = r;
            double sum = 0.0;
            int k = 0;
            while ((long)k < cols) {
                inds[1] = k;
                double val = idxInput.getDouble(inds);
                if (doSoftmax) {
                    val = Math.exp(val);
                } else if (val < 0.0) {
                    val = 0.0;
                    ++warnNegativeValues;
                }
                vals[k] = val;
                sum += val;
                ++k;
            }
            k = 0;
            while ((long)k < cols) {
                inds[1] = k;
                idxOutput.putDouble(inds, vals[k] * (maxValue / sum));
                ++k;
            }
        }
        if (warnNegativeValues > 0) {
            long total = rows * cols;
            logger.warn(String.format("Negative raw 'probability' values detected (%d/%d, %.1f%%) -  - these will be clipped to 0.  Should softmax be being used...?", warnNegativeValues, total, (double)warnNegativeValues * (100.0 / (double)total)));
        }
    }

    static Mat stripPadding(Mat mat, Padding padding) {
        if (padding.isEmpty()) {
            return mat;
        }
        return mat.apply(new Rect(padding.getX1(), padding.getY1(), mat.cols() - padding.getXSum(), mat.rows() - padding.getYSum())).clone();
    }

    static Padding getDefaultGaussianPadding(double sigmaX, double sigmaY) {
        int padX = (int)Math.ceil(sigmaX * 3.0) + 1;
        int padY = (int)Math.ceil(sigmaY * 3.0) + 1;
        return Padding.getPadding((int)padX, (int)padX, (int)padY, (int)padY);
    }

    static void normalize(Mat mat, double[] subtract, double[] scale) {
        int n = mat.arrayChannels();
        assert (subtract.length == n);
        assert (scale.length == n);
        MatVector matvec = new MatVector();
        opencv_core.split((Mat)mat, (MatVector)matvec);
        for (int i = 0; i < n; ++i) {
            Mat temp = matvec.get((long)i);
            temp.put(opencv_core.multiply((MatExpr)opencv_core.subtract((Mat)temp, (Scalar)Scalar.all((double)subtract[i])), (double)scale[i]));
        }
        opencv_core.merge((MatVector)matvec, (Mat)mat);
    }

    static Collection<URI> getAllUris(UriResource ... items) throws IOException {
        LinkedHashSet<URI> list = new LinkedHashSet<URI>();
        for (UriResource item : items) {
            list.addAll(item.getURIs());
        }
        return list;
    }

    static boolean updateAllUris(Map<URI, URI> replacements, UriResource ... items) throws IOException {
        boolean changes = false;
        for (UriResource item : items) {
            changes |= item.updateURIs(replacements);
        }
        return changes;
    }

    static {
        GsonTools.getDefaultBuilder().registerTypeAdapterFactory(factoryOps);
        ImageOps.registerTypes(factoryOps, ImageOp.class, ImageOps.class, "op");
        factoryDataOps = GsonTools.createSubTypeAdapterFactory(ImageDataOp.class, (String)"type");
        GsonTools.getDefaultBuilder().registerTypeAdapterFactory(factoryDataOps);
        ImageOps.registerTypes(factoryDataOps, ImageDataOp.class, ImageOps.class, "data.op");
    }

    @Target(value={ElementType.TYPE})
    @Retention(value=RetentionPolicy.RUNTIME)
    private static @interface OpType {
        public String value();
    }

    @OpType(value="default")
    static class DefaultImageDataOp
    implements ImageDataOp {
        private ImageOp op;

        DefaultImageDataOp(ImageOp op) {
            this.op = op;
        }

        @Override
        public Mat apply(ImageData<BufferedImage> imageData, RegionRequest request) throws IOException {
            if (this.op == null) {
                BufferedImage img = (BufferedImage)imageData.getServer().readRegion(request);
                return OpenCVTools.imageToMat(img);
            }
            Padding padding = this.op.getPadding();
            BufferedImage img = ServerTools.getPaddedRequest((ImageServer)imageData.getServer(), (RegionRequest)request, (Padding)padding);
            Mat mat = OpenCVTools.imageToMat(img);
            mat.convertTo(mat, 5);
            try (PointerScope scope = new PointerScope();){
                mat.put(this.op.apply(mat));
                Mat mat2 = mat;
                return mat2;
            }
        }

        @Override
        public List<ImageChannel> getChannels(ImageData<BufferedImage> imageData) {
            if (this.op == null) {
                return imageData.getServerMetadata().getChannels();
            }
            return this.op.getChannels(imageData.getServerMetadata().getChannels());
        }

        @Override
        public boolean supportsImage(ImageData<BufferedImage> imageData) {
            return true;
        }

        @Override
        public ImageDataOp appendOps(ImageOp ... ops) {
            if (ops.length == 0) {
                return this;
            }
            if (this.op == null) {
                return new DefaultImageDataOp(Core.sequential(ops));
            }
            ArrayList<ImageOp> allOps = new ArrayList<ImageOp>();
            allOps.add(this.op);
            allOps.addAll(Arrays.asList(ops));
            ImageOp newOp = Core.sequential(allOps);
            return new DefaultImageDataOp(newOp);
        }

        @Override
        public PixelType getOutputType(PixelType inputType) {
            if (this.op == null) {
                return PixelType.FLOAT32;
            }
            return this.op.getOutputType(PixelType.FLOAT32);
        }

        public Collection<URI> getURIs() throws IOException {
            return this.op == null ? Collections.emptyList() : this.op.getURIs();
        }

        public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
            if (this.op == null) {
                return false;
            }
            return this.op.updateURIs(replacements);
        }
    }

    @OpType(value="channels")
    static class ChannelImageDataOp
    implements ImageDataOp {
        private ColorTransforms.ColorTransform[] colorTransforms;
        private ImageOp op;

        ChannelImageDataOp(ImageOp op, ColorTransforms.ColorTransform ... colorTransforms) {
            this.colorTransforms = (ColorTransforms.ColorTransform[])colorTransforms.clone();
            this.op = op;
        }

        @Override
        public boolean supportsImage(ImageData<BufferedImage> imageData) {
            for (ColorTransforms.ColorTransform t : this.colorTransforms) {
                if (t.supportsImage(imageData.getServer())) continue;
                return false;
            }
            return true;
        }

        @Override
        public Mat apply(ImageData<BufferedImage> imageData, RegionRequest request) throws IOException {
            BufferedImage img = this.op == null ? (BufferedImage)imageData.getServer().readRegion(request) : ServerTools.getPaddedRequest((ImageServer)imageData.getServer(), (RegionRequest)request, (Padding)this.op.getPadding());
            float[] pixels = null;
            ImageServer server = imageData.getServer();
            Mat mat = new Mat();
            try (PointerScope scope = new PointerScope();){
                ArrayList<Mat> channels = new ArrayList<Mat>();
                for (ColorTransforms.ColorTransform t : this.colorTransforms) {
                    Mat matTemp = new Mat(img.getHeight(), img.getWidth(), opencv_core.CV_32FC1);
                    pixels = t.extractChannel(server, img, pixels);
                    try (FloatIndexer idx = (FloatIndexer)matTemp.createIndexer();){
                        idx.put(0L, pixels);
                    }
                    channels.add(matTemp);
                }
                OpenCVTools.mergeChannels(channels, mat);
                if (this.op != null) {
                    mat.put(this.op.apply(mat));
                }
            }
            return mat;
        }

        @Override
        public List<ImageChannel> getChannels(ImageData<BufferedImage> imageData) {
            List<ImageChannel> channels = Arrays.stream(this.colorTransforms).map(c -> ImageChannel.getInstance((String)c.getName(), null)).toList();
            if (this.op == null) {
                return channels;
            }
            return this.op.getChannels(channels);
        }

        @Override
        public ImageDataOp appendOps(ImageOp ... ops) {
            if (ops.length == 0) {
                return this;
            }
            if (this.op == null) {
                return new ChannelImageDataOp(Core.sequential(ops), this.colorTransforms);
            }
            ArrayList<ImageOp> allOps = new ArrayList<ImageOp>();
            allOps.add(this.op);
            allOps.addAll(Arrays.asList(ops));
            ImageOp newOp = Core.sequential(allOps);
            return new ChannelImageDataOp(newOp, this.colorTransforms);
        }

        @Override
        public PixelType getOutputType(PixelType inputType) {
            if (this.op == null) {
                return inputType;
            }
            return this.op.getOutputType(inputType);
        }

        public Collection<URI> getURIs() throws IOException {
            return this.op == null ? Collections.emptyList() : this.op.getURIs();
        }

        public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
            if (this.op == null) {
                return false;
            }
            return this.op.updateURIs(replacements);
        }
    }

    public static abstract class PaddedOp
    implements ImageOp {
        private transient Padding padding;

        protected abstract Padding calculatePadding();

        protected abstract List<Mat> transformPadded(Mat var1);

        @Override
        public Mat apply(Mat input) {
            List<Mat> mats = this.transformPadded(input);
            Padding padding = this.getPadding();
            if (!padding.isEmpty()) {
                for (Mat mat : mats) {
                    Mat mat2 = ImageOps.stripPadding(mat, this.getPadding());
                    mat.put(mat2);
                    mat2.close();
                }
            }
            return mats.size() == 1 ? mats.get(0) : OpenCVTools.mergeChannels(mats, input);
        }

        @Override
        public Padding getPadding() {
            if (this.padding == null) {
                this.padding = this.calculatePadding();
            }
            return this.padding;
        }
    }

    @OpType(value="ml")
    public static class ML {
        public static ImageOp statModel(OpenCVClassifiers.OpenCVStatModel statModel, boolean requestProbabilities) {
            return new StatModelOp(statModel, requestProbabilities);
        }

        public static ImageOp dnn(DnnModel model, int inputWidth, int inputHeight, Padding padding, String ... outputNames) {
            return new DnnOp(model, inputWidth, inputHeight, padding, outputNames);
        }

        public static ImageOp preprocessor(FeaturePreprocessor preprocessor) {
            return new FeaturePreprocessorOp(preprocessor);
        }

        @OpType(value="opencv-statmodel")
        static class StatModelOp
        implements ImageOp {
            private OpenCVClassifiers.OpenCVStatModel model;
            private boolean requestProbabilities;

            StatModelOp(OpenCVClassifiers.OpenCVStatModel model, boolean requestProbabilities) {
                this.model = model;
                this.requestProbabilities = requestProbabilities;
            }

            @Override
            public Mat apply(Mat input) {
                try (PointerScope scope = new PointerScope();){
                    int w = input.cols();
                    int h = input.rows();
                    input.put(input.reshape(1, w * h));
                    Mat matResult = new Mat();
                    if (this.requestProbabilities) {
                        Mat temp = new Mat();
                        this.model.predict(input, temp, matResult);
                        temp.close();
                    } else {
                        this.model.predict(input, matResult, null);
                    }
                    input.put(matResult.reshape(matResult.cols(), h));
                }
                return input;
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return PixelType.FLOAT32;
            }
        }

        @OpType(value="opencv-dnn")
        static class DnnOp<T>
        extends PaddedOp {
            private static final Logger logger = LoggerFactory.getLogger(DnnOp.class);
            private DnnModel model;
            private int inputWidth;
            private int inputHeight;
            private String[] outputNames = new String[0];
            private Padding padding;
            private transient String inputName;
            private transient Map<Integer, List<ImageChannel>> outputChannels = Collections.synchronizedMap(new HashMap());

            DnnOp(DnnModel model, int inputWidth, int inputHeight, Padding padding, String ... outputNames) {
                this.model = model;
                this.inputWidth = inputWidth;
                this.inputHeight = inputHeight;
                this.padding = padding == null ? Padding.empty() : padding;
                this.outputNames = (String[])outputNames.clone();
            }

            @Override
            protected Padding calculatePadding() {
                return this.padding;
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                Mat result;
                String inputName = "input";
                if (this.inputWidth <= 0 && this.inputHeight <= 0 || input.cols() == this.inputWidth && input.rows() == this.inputHeight) {
                    result = ImageOps.doPrediction(this.model, input, inputName, this.outputNames);
                } else {
                    Padding padding = this.getPadding();
                    result = OpenCVTools.applyTiled(m -> ImageOps.doPrediction(this.model, m, inputName, this.outputNames), input, this.inputWidth, this.inputHeight, padding, 2);
                }
                return Collections.singletonList(result);
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return PixelType.FLOAT32;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                List outChannels = this.outputChannels.get(channels.size());
                if (outChannels == null) {
                    DnnOp dnnOp = this;
                    synchronized (dnnOp) {
                        outChannels = this.outputChannels.get(channels.size());
                        if (outChannels == null) {
                            AbstractDnnModel predictionModel;
                            Map<String, DnnShape> outputs;
                            ArrayList<CallSite> names = new ArrayList<CallSite>();
                            DnnModel dnnModel = this.model;
                            if (dnnModel instanceof AbstractDnnModel && (outputs = (predictionModel = (AbstractDnnModel)dnnModel).getPredictionFunction().getOutputs(DnnShape.of(1L, channels.size(), this.inputHeight, this.inputWidth))).size() > 1) {
                                Set<String> outputKeys = this.outputNames == null || this.outputNames.length == 0 ? outputs.keySet() : Arrays.asList(this.outputNames);
                                for (String key : outputKeys) {
                                    DnnShape shape = outputs.get(key);
                                    if (shape != null && !shape.isUnknown() && shape.numDimensions() > 2 && shape.get(1) != DnnShape.UNKNOWN_LENGTH) {
                                        int c = 0;
                                        while ((long)c < shape.get(1)) {
                                            names.add((CallSite)((Object)(key + ": " + c)));
                                            ++c;
                                        }
                                        continue;
                                    }
                                    logger.warn("Unknown output shape for {} - output channels are unknown", (Object)key);
                                }
                            }
                            Mat mat = new Mat(this.inputHeight, this.inputWidth, opencv_core.CV_32FC((int)channels.size()), Scalar.ZERO);
                            List<Mat> output = this.transformPadded(mat);
                            int nChannels = output.stream().mapToInt(Mat::channels).sum();
                            output.stream().forEach(Pointer::close);
                            outChannels = names.size() == nChannels ? ImageChannel.getChannelList((String[])((String[])names.toArray(String[]::new))) : ImageChannel.getDefaultChannelList((int)nChannels);
                            this.outputChannels.put(channels.size(), outChannels);
                            mat.close();
                        }
                        this.outputChannels.put(channels.size(), outChannels);
                    }
                }
                return outChannels;
            }

            @Override
            public Collection<URI> getURIs() throws IOException {
                if (this.model instanceof UriResource) {
                    return ((UriResource)this.model).getURIs();
                }
                return Collections.emptyList();
            }

            @Override
            public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
                if (this.model instanceof UriResource) {
                    return ((UriResource)this.model).updateURIs(replacements);
                }
                return false;
            }
        }

        @OpType(value="feature-preprocessor")
        static class FeaturePreprocessorOp
        implements ImageOp {
            private FeaturePreprocessor preprocessor;

            private FeaturePreprocessorOp(FeaturePreprocessor preprocessor) {
                this.preprocessor = preprocessor;
            }

            @Override
            public Mat apply(Mat input) {
                input.convertTo(input, 5);
                this.preprocessor.apply(input, true);
                return input;
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                if (!this.preprocessor.doesFeatureTransform()) {
                    return channels;
                }
                return IntStream.range(0, this.preprocessor.getOutputLength()).mapToObj(i -> ImageChannel.getInstance((String)("Feature " + i), (Integer)ColorTools.WHITE)).toList();
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return PixelType.FLOAT32;
            }
        }
    }

    @OpType(value="core")
    public static class Core {
        public static ImageOp ensureType(PixelType type) {
            return new ConvertTypeOp(type);
        }

        public static ImageOp multiply(double ... values) {
            return new MultiplyOp(values);
        }

        public static ImageOp divide(double ... values) {
            return new DivideOp(values);
        }

        public static ImageOp add(double ... values) {
            return new AddOp(values);
        }

        public static ImageOp subtract(double ... values) {
            return new SubtractOp(values);
        }

        public static ImageOp power(double value) {
            return new PowerOp(value);
        }

        public static ImageOp sqrt() {
            return new SqrtOp();
        }

        public static ImageOp sequential(Collection<? extends ImageOp> ops) {
            if (ops.size() == 1) {
                return ops.iterator().next();
            }
            return new SequentialMultiOp(ops);
        }

        public static ImageOp sequential(ImageOp ... ops) {
            return Core.sequential(Arrays.asList(ops));
        }

        public static ImageOp splitMerge(Collection<? extends ImageOp> ops) {
            return new SplitMergeOp((ImageOp[])ops.toArray(ImageOp[]::new));
        }

        public static ImageOp splitMerge(ImageOp ... ops) {
            return Core.splitMerge(Arrays.asList(ops));
        }

        public static ImageOp identity() {
            return new IdentityOp();
        }

        public static ImageOp exp() {
            return new ExponentialOp();
        }

        public static ImageOp log() {
            return new LogOp();
        }

        public static ImageOp round() {
            return new RoundOp();
        }

        public static ImageOp floor() {
            return new FloorOp();
        }

        public static ImageOp ceil() {
            return new CeilOp();
        }

        public static ImageOp replaceNaNs(double replaceValue) {
            return new ReplaceNaNsOp(replaceValue);
        }

        public static ImageOp replace(double originalValue, double newValue) {
            return new ReplaceValueOp(originalValue, newValue);
        }

        public static ImageOp splitAdd(ImageOp opLeft, ImageOp opRight) {
            return new SplitCombineOp(opLeft, opRight, SplitCombineType.ADD);
        }

        public static ImageOp splitSubtract(ImageOp opLeft, ImageOp opRight) {
            return new SplitCombineOp(opLeft, opRight, SplitCombineType.SUBTRACT);
        }

        public static ImageOp splitMultiply(ImageOp opLeft, ImageOp opRight) {
            return new SplitCombineOp(opLeft, opRight, SplitCombineType.MULTIPLY);
        }

        public static ImageOp splitDivide(ImageOp opTop, ImageOp opBottom) {
            return new SplitCombineOp(opTop, opBottom, SplitCombineType.DIVIDE);
        }

        public static ImageOp clip(double min, double max) {
            return new ClipOp(min, max);
        }

        @OpType(value="convert")
        static class ConvertTypeOp
        implements ImageOp {
            private PixelType pixelType;

            ConvertTypeOp(PixelType pixelType) {
                this.pixelType = pixelType;
            }

            @Override
            public Mat apply(Mat input) {
                input.convertTo(input, OpenCVTools.getOpenCVPixelType(this.pixelType));
                return input;
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return this.pixelType;
            }
        }

        @OpType(value="multiply")
        static class MultiplyOp
        implements ImageOp {
            private double[] values;

            MultiplyOp(double ... values) {
                this.values = (double[])values.clone();
            }

            @Override
            public Mat apply(Mat input) {
                if (this.values.length == 1) {
                    input.put(opencv_core.multiply((Mat)input, (double)this.values[0]));
                } else if (this.values.length == input.channels()) {
                    int i = 0;
                    List<Mat> channels = OpenCVTools.splitChannels(input);
                    for (Mat m : channels) {
                        m.put(opencv_core.multiply((Mat)m, (double)this.values[i]));
                        ++i;
                    }
                    OpenCVTools.mergeChannels(channels, input);
                } else {
                    throw new IllegalArgumentException("Multiply requires " + this.values.length + " channels, but Mat has " + input.channels());
                }
                return input;
            }
        }

        @OpType(value="divide")
        static class DivideOp
        implements ImageOp {
            private double[] values;

            DivideOp(double ... values) {
                this.values = (double[])values.clone();
            }

            @Override
            public Mat apply(Mat input) {
                if (this.values.length == 1) {
                    input.put(opencv_core.divide((Mat)input, (double)this.values[0]));
                } else if (this.values.length == input.channels()) {
                    int i = 0;
                    List<Mat> channels = OpenCVTools.splitChannels(input);
                    for (Mat m : channels) {
                        m.put(opencv_core.divide((Mat)m, (double)this.values[i]));
                        ++i;
                    }
                    OpenCVTools.mergeChannels(channels, input);
                } else {
                    throw new IllegalArgumentException("Divide requires " + this.values.length + " channels, but Mat has " + input.channels());
                }
                return input;
            }
        }

        @OpType(value="add")
        static class AddOp
        implements ImageOp {
            private double[] values;

            AddOp(double ... values) {
                this.values = (double[])values.clone();
            }

            @Override
            public Mat apply(Mat input) {
                if (this.values.length == 1) {
                    input.put(opencv_core.add((Mat)input, (Scalar)Scalar.all((double)this.values[0])));
                } else if (this.values.length == input.channels()) {
                    int i = 0;
                    List<Mat> channels = OpenCVTools.splitChannels(input);
                    for (Mat m : channels) {
                        m.put(opencv_core.add((Mat)m, (Scalar)Scalar.all((double)this.values[i])));
                        ++i;
                    }
                    OpenCVTools.mergeChannels(channels, input);
                } else {
                    throw new IllegalArgumentException("Add requires " + this.values.length + " channels, but Mat has " + input.channels());
                }
                return input;
            }
        }

        @OpType(value="subtract")
        static class SubtractOp
        implements ImageOp {
            private double[] values;

            SubtractOp(double ... values) {
                this.values = (double[])values.clone();
            }

            @Override
            public Mat apply(Mat input) {
                if (this.values.length == 1) {
                    input.put(opencv_core.subtract((Mat)input, (Scalar)Scalar.all((double)this.values[0])));
                } else if (this.values.length == input.channels()) {
                    int i = 0;
                    List<Mat> channels = OpenCVTools.splitChannels(input);
                    for (Mat m : channels) {
                        m.put(opencv_core.subtract((Mat)m, (Scalar)Scalar.all((double)this.values[i])));
                        ++i;
                    }
                    OpenCVTools.mergeChannels(channels, input);
                } else {
                    throw new IllegalArgumentException("Subtract requires " + this.values.length + " channels, but Mat has " + input.channels());
                }
                return input;
            }
        }

        @OpType(value="pow")
        static class PowerOp
        implements ImageOp {
            private double power;

            PowerOp(double power) {
                this.power = power;
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.apply(input, d -> FastMath.pow((double)d, (double)this.power));
                return input;
            }
        }

        @OpType(value="sqrt")
        static class SqrtOp
        implements ImageOp {
            SqrtOp() {
            }

            @Override
            public Mat apply(Mat input) {
                opencv_core.sqrt((Mat)input, (Mat)input);
                return input;
            }
        }

        @OpType(value="sequential")
        static class SequentialMultiOp
        extends PaddedOp {
            private static final Logger logger = LoggerFactory.getLogger(SequentialMultiOp.class);
            private List<ImageOp> ops;

            SequentialMultiOp(Collection<? extends ImageOp> ops) {
                this.ops = new ArrayList<ImageOp>(ops);
            }

            @Override
            protected Padding calculatePadding() {
                Padding padding = Padding.empty();
                for (ImageOp t : this.ops) {
                    padding = padding.add(t.getPadding());
                }
                return padding;
            }

            @Override
            public Mat apply(Mat input) {
                for (ImageOp t : this.ops) {
                    Mat output = t.apply(input);
                    if (output == input) continue;
                    input.put(output);
                    output.close();
                }
                return input;
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                logger.warn("transformPadded(Mat) should not be called directly for this class!");
                return Collections.singletonList(this.apply(input));
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                for (ImageOp t : this.ops) {
                    channels = t.getChannels(channels);
                }
                return channels;
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                for (ImageOp t : this.ops) {
                    inputType = t.getOutputType(inputType);
                }
                return inputType;
            }

            @Override
            public Collection<URI> getURIs() throws IOException {
                return ImageOps.getAllUris((UriResource[])this.ops.toArray(ImageOp[]::new));
            }

            @Override
            public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
                return ImageOps.updateAllUris(replacements, (UriResource[])this.ops.toArray(ImageOp[]::new));
            }
        }

        @OpType(value="split-merge")
        static class SplitMergeOp
        extends PaddedOp {
            private List<ImageOp> ops = new ArrayList<ImageOp>();

            SplitMergeOp(ImageOp ... ops) {
                for (ImageOp t : ops) {
                    this.ops.add(t);
                }
            }

            @Override
            public Mat apply(Mat input) {
                return OpenCVTools.mergeChannels(this.transformPadded(input), input);
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                return this.ops.stream().flatMap(t -> t.getChannels(channels).stream()).toList();
            }

            @Override
            protected Padding calculatePadding() {
                Padding padding = Padding.empty();
                for (ImageOp t : this.ops) {
                    padding = padding.max(t.getPadding());
                }
                return padding;
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                if (this.ops.isEmpty()) {
                    return Collections.singletonList(new Mat());
                }
                if (this.ops.size() == 1) {
                    return Collections.singletonList(this.ops.get(0).apply(input));
                }
                try (PointerScope scope = new PointerScope();){
                    ArrayList<Mat> mats = new ArrayList<Mat>();
                    Padding padding = this.getPadding();
                    for (ImageOp op : this.ops) {
                        Padding padExtra = padding.subtract(op.getPadding());
                        Mat temp = !padExtra.isEmpty() ? ImageOps.stripPadding(input, padExtra) : input.clone();
                        temp.put(op.apply(temp));
                        temp.retainReference();
                        mats.add(temp);
                    }
                    ArrayList<Mat> arrayList = mats;
                    return arrayList;
                }
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                for (ImageOp t : this.ops) {
                    inputType = t.getOutputType(inputType);
                }
                return inputType;
            }

            @Override
            public Collection<URI> getURIs() throws IOException {
                return ImageOps.getAllUris((UriResource[])this.ops.toArray(ImageOp[]::new));
            }

            @Override
            public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
                return ImageOps.updateAllUris(replacements, (UriResource[])this.ops.toArray(ImageOp[]::new));
            }
        }

        @OpType(value="identity")
        static class IdentityOp
        implements ImageOp {
            IdentityOp() {
            }

            @Override
            public Mat apply(Mat input) {
                return input;
            }
        }

        @OpType(value="exp")
        static class ExponentialOp
        implements ImageOp {
            ExponentialOp() {
            }

            @Override
            public Mat apply(Mat input) {
                opencv_core.exp((Mat)input, (Mat)input);
                return input;
            }
        }

        @OpType(value="log")
        static class LogOp
        implements ImageOp {
            LogOp() {
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.apply(input, d -> FastMath.log((double)d));
                return input;
            }
        }

        @OpType(value="round")
        static class RoundOp
        implements ImageOp {
            RoundOp() {
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.round(input);
                return input;
            }
        }

        @OpType(value="floor")
        static class FloorOp
        implements ImageOp {
            FloorOp() {
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.floor(input);
                return input;
            }
        }

        @OpType(value="ceil")
        static class CeilOp
        implements ImageOp {
            CeilOp() {
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.ceil(input);
                return input;
            }
        }

        @OpType(value="replace-nans")
        static class ReplaceNaNsOp
        implements ImageOp {
            private double value;

            ReplaceNaNsOp(double value) {
                this.value = value;
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.replaceNaNs(input, this.value);
                return input;
            }
        }

        @OpType(value="replace-values")
        static class ReplaceValueOp
        implements ImageOp {
            private double originalValue;
            private double newValue;

            ReplaceValueOp(double originalValue, double newValue) {
                this.originalValue = originalValue;
                this.newValue = newValue;
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.replaceValues(input, this.originalValue, this.newValue);
                return input;
            }
        }

        @OpType(value="split-combine")
        static class SplitCombineOp
        extends PaddedOp {
            private SplitCombineType combine;
            private ImageOp op1;
            private ImageOp op2;

            SplitCombineOp(ImageOp op1, ImageOp op2, SplitCombineType combine) {
                Objects.requireNonNull(combine);
                this.op1 = op1 == null ? new IdentityOp() : op1;
                this.op2 = op2 == null ? new IdentityOp() : op2;
                this.combine = combine;
            }

            @Override
            public Mat apply(Mat input) {
                List<Mat> result = this.transformPadded(input);
                return result.size() == 1 ? result.get(0) : OpenCVTools.mergeChannels(result, input);
            }

            private String getCombineStr() {
                switch (this.combine.ordinal()) {
                    case 0: {
                        return "+";
                    }
                    case 3: {
                        return "/";
                    }
                    case 2: {
                        return "*";
                    }
                    case 1: {
                        return "-";
                    }
                }
                throw new IllegalArgumentException("Unknown combine type " + String.valueOf((Object)this.combine));
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                List<ImageChannel> c1 = this.op1.getChannels(channels);
                List<ImageChannel> c2 = this.op2.getChannels(channels);
                if (c1.size() != c2.size()) {
                    throw new IllegalArgumentException("Channel counts do not match!");
                }
                String combo = " " + this.getCombineStr() + " ";
                ArrayList<ImageChannel> combinedChannels = new ArrayList<ImageChannel>();
                for (int i = 0; i < c1.size(); ++i) {
                    combinedChannels.add(ImageChannel.getInstance((String)(c1.get(i).getName() + combo + c2.get(i).getName()), (Integer)c1.get(i).getColor()));
                }
                return combinedChannels;
            }

            @Override
            protected Padding calculatePadding() {
                return this.op1.getPadding().max(this.op2.getPadding());
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                Padding padExtra2;
                Mat mat2 = this.op2.apply(input.clone());
                Mat mat1 = this.op1.apply(input);
                Padding padding = this.getPadding();
                Padding padExtra1 = padding.subtract(this.op1.getPadding());
                if (!padExtra1.isEmpty()) {
                    mat1.put(ImageOps.stripPadding(mat1, padExtra1));
                }
                if (!(padExtra2 = padding.subtract(this.op2.getPadding())).isEmpty()) {
                    mat2.put(ImageOps.stripPadding(mat2, padExtra2));
                }
                switch (this.combine.ordinal()) {
                    case 0: {
                        opencv_core.add((Mat)mat1, (Mat)mat2, (Mat)mat1);
                        break;
                    }
                    case 3: {
                        opencv_core.divide((Mat)mat1, (Mat)mat2, (Mat)mat1);
                        break;
                    }
                    case 2: {
                        mat1.put(mat1.mul(mat2));
                        break;
                    }
                    case 1: {
                        opencv_core.subtract((Mat)mat1, (Mat)mat2, (Mat)mat1);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unknown combine type " + String.valueOf((Object)this.combine));
                    }
                }
                mat2.close();
                return Collections.singletonList(mat1);
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return this.op1.getOutputType(inputType);
            }

            @Override
            public Collection<URI> getURIs() throws IOException {
                return ImageOps.getAllUris(this.op1, this.op2);
            }

            @Override
            public boolean updateURIs(Map<URI, URI> replacements) throws IOException {
                return ImageOps.updateAllUris(replacements, this.op1, this.op2);
            }
        }

        private static enum SplitCombineType {
            ADD,
            SUBTRACT,
            MULTIPLY,
            DIVIDE;

        }

        @OpType(value="clip")
        static class ClipOp
        implements ImageOp {
            private double min;
            private double max;

            private ClipOp(double min, double max) {
                this.min = min;
                this.max = max;
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.apply(input, v -> GeneralTools.clipValue((double)v, (double)this.min, (double)this.max));
                return input;
            }
        }
    }

    @OpType(value="threshold")
    public static class Threshold {
        public static ImageOp threshold(double ... thresholds) {
            return new FixedThresholdOp(thresholds);
        }

        public static ImageOp thresholdMeanStd(double ... k) {
            return new MeanStdDevOp(k);
        }

        public static ImageOp thresholdMedianAbsDev(double ... k) {
            return new MedianAbsDevThresholdOp(k);
        }

        @OpType(value="constant")
        static class FixedThresholdOp
        extends AbstractThresholdOp {
            private double[] thresholds;

            FixedThresholdOp(double ... thresholds) {
                this.thresholds = (double[])thresholds.clone();
            }

            @Override
            public double getThreshold(Mat mat, int channel) {
                return this.thresholds[Math.min(channel, this.thresholds.length - 1)];
            }
        }

        @OpType(value="mean-std")
        static class MeanStdDevOp
        extends AbstractThresholdOp {
            private double[] k;

            MeanStdDevOp(double ... k) {
                this.k = (double[])k.clone();
            }

            @Override
            public double getThreshold(Mat mat, int channel) {
                Mat mean = new Mat();
                Mat stddev = new Mat();
                opencv_core.meanStdDev((Mat)mat, (Mat)mean, (Mat)stddev);
                double m = mean.createIndexer().getDouble(new long[]{0L});
                double s = stddev.createIndexer().getDouble(new long[]{0L});
                double k = this.k[Math.min(channel, this.k.length - 1)];
                return m + s * k;
            }
        }

        @OpType(value="median-mad")
        static class MedianAbsDevThresholdOp
        extends AbstractThresholdOp {
            private double[] k;

            MedianAbsDevThresholdOp(double ... k) {
                this.k = (double[])k.clone();
            }

            @Override
            public double getThreshold(Mat mat, int channel) {
                double median = OpenCVTools.median(mat);
                Mat matAbs = opencv_core.abs((MatExpr)opencv_core.subtract((Mat)mat, (Scalar)Scalar.all((double)median))).asMat();
                double mad = OpenCVTools.median(matAbs) / 0.675;
                double k = this.k[Math.min(channel, this.k.length - 1)];
                return median + mad * k;
            }
        }

        static abstract class AbstractThresholdOp
        implements ImageOp {
            AbstractThresholdOp() {
            }

            @Override
            public Mat apply(Mat input) {
                MatVector matvec = new MatVector();
                opencv_core.split((Mat)input, (MatVector)matvec);
                MatVector matvec2 = new MatVector();
                int c = 0;
                while ((long)c < matvec.size()) {
                    Mat mat = matvec.get((long)c);
                    Mat mat2 = new Mat();
                    double threshold = this.getThreshold(mat, c);
                    opencv_imgproc.threshold((Mat)mat, (Mat)mat2, (double)threshold, (double)1.0, (int)0);
                    matvec2.push_back(mat2);
                    ++c;
                }
                opencv_core.merge((MatVector)matvec2, (Mat)input);
                return input;
            }

            public abstract double getThreshold(Mat var1, int var2);
        }
    }

    @OpType(value="channels")
    public static class Channels {
        public static ImageOp deconvolve(ColorDeconvolutionStains stains) {
            return new ColorDeconvolutionOp(stains);
        }

        public static ImageOp extract(int ... channels) {
            return new ExtractChannelsOp(channels);
        }

        public static ImageOp repeat(int numRepeats) {
            return new RepeatChannelsOp(numRepeats);
        }

        public static ImageOp sum() {
            return new SumChannelsOp();
        }

        public static ImageOp mean() {
            return new MeanChannelsOp();
        }

        public static ImageOp minimum() {
            return new MinChannelsOp();
        }

        public static ImageOp maximum() {
            return new MaxChannelsOp();
        }

        @OpType(value="color-deconvolution")
        static class ColorDeconvolutionOp
        implements ImageOp {
            private ColorDeconvolutionStains stains;
            private transient Mat matInv;

            ColorDeconvolutionOp(ColorDeconvolutionStains stains) {
                this.stains = stains;
            }

            @Override
            public Mat apply(Mat input) {
                assert (input.channels() == 3);
                int w = input.cols();
                int h = input.rows();
                input.convertTo(input, 5);
                Mat matCols = input.reshape(1, w * h);
                double[] max = new double[]{this.stains.getMaxRed(), this.stains.getMaxGreen(), this.stains.getMaxBlue()};
                for (int c = 0; c < 3; ++c) {
                    Mat col = matCols.col(c);
                    MatExpr expr = opencv_core.max((Mat)col, (double)1.0);
                    col.put(opencv_core.divide((MatExpr)expr, (double)max[c]));
                    opencv_core.log((Mat)col, (Mat)col);
                    expr = opencv_core.divide((Mat)col, (double)(-Math.log(10.0)));
                    col.put(expr);
                }
                matCols.put(opencv_core.max((Mat)matCols, (double)0.0));
                matCols.put(matCols.reshape(3, h));
                opencv_core.transform((Mat)matCols, (Mat)matCols, (Mat)this.getMatInv());
                input.put(matCols);
                return input;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private Mat getMatInv() {
                if (this.matInv == null || this.matInv.isNull()) {
                    ColorDeconvolutionOp colorDeconvolutionOp = this;
                    synchronized (colorDeconvolutionOp) {
                        if (this.matInv == null || this.matInv.isNull()) {
                            this.matInv = new Mat(3, 3, opencv_core.CV_64FC1, Scalar.ZERO);
                            double[][] inv = this.stains.getMatrixInverse();
                            try (DoubleIndexer idx = (DoubleIndexer)this.matInv.createIndexer();){
                                idx.put(0L, 0L, inv[0]);
                                idx.put(1L, 0L, inv[1]);
                                idx.put(2L, 0L, inv[2]);
                            }
                            this.matInv.put(this.matInv.t());
                            this.matInv.retainReference();
                        }
                    }
                }
                return this.matInv;
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                return Arrays.asList(ImageChannel.getInstance((String)this.stains.getStain(1).getName(), (Integer)this.stains.getStain(1).getColor()), ImageChannel.getInstance((String)this.stains.getStain(2).getName(), (Integer)this.stains.getStain(2).getColor()), ImageChannel.getInstance((String)this.stains.getStain(3).getName(), (Integer)this.stains.getStain(3).getColor()));
            }
        }

        @OpType(value="extract-channels")
        static class ExtractChannelsOp
        implements ImageOp {
            private int[] channels;

            ExtractChannelsOp(int ... channels) {
                if (channels.length == 0) {
                    throw new IllegalArgumentException("No channel indices provided to extract channels");
                }
                this.channels = (int[])channels.clone();
            }

            @Override
            public Mat apply(Mat input) {
                MatVector matvec = new MatVector();
                opencv_core.split((Mat)input, (MatVector)matvec);
                MatVector matvec2 = new MatVector();
                for (int c : this.channels) {
                    matvec2.push_back(matvec.get((long)c));
                }
                opencv_core.merge((MatVector)matvec2, (Mat)input);
                return input;
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                ArrayList<ImageChannel> newChannels = new ArrayList<ImageChannel>();
                for (int c : this.channels) {
                    newChannels.add(channels.get(c));
                }
                return newChannels;
            }

            public String toString() {
                if (this.channels == null || this.channels.length == 0) {
                    return "No channels";
                }
                if (this.channels.length == 1) {
                    return "Channel " + this.channels[0];
                }
                return "Channels [" + Arrays.stream(this.channels).mapToObj(c -> Integer.toString(c)).collect(Collectors.joining(",")) + "]";
            }
        }

        @OpType(value="repeat-channels")
        static class RepeatChannelsOp
        implements ImageOp {
            private int numRepeats;

            RepeatChannelsOp(int numRepeats) {
                this.numRepeats = numRepeats;
            }

            @Override
            public Mat apply(Mat input) {
                List<Mat> originalChannels = OpenCVTools.splitChannels(input);
                ArrayList<Mat> outputChannels = new ArrayList<Mat>();
                for (int i = 0; i < this.numRepeats; ++i) {
                    outputChannels.addAll(originalChannels);
                }
                return OpenCVTools.mergeChannels(outputChannels, input);
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                ArrayList<ImageChannel> newChannels = new ArrayList<ImageChannel>(channels);
                for (int i = 1; i < this.numRepeats; ++i) {
                    for (ImageChannel c : channels) {
                        newChannels.add(ImageChannel.getInstance((String)(c.getName() + "(" + i + ")"), (Integer)c.getColor()));
                    }
                }
                return newChannels;
            }

            public String toString() {
                return "Repeat channels " + this.numRepeats;
            }
        }

        @OpType(value="sum")
        static class SumChannelsOp
        extends ReduceChannelsOp {
            SumChannelsOp() {
            }

            @Override
            protected int getReduceOp() {
                return 0;
            }

            @Override
            protected String reduceName() {
                return "Sum";
            }
        }

        @OpType(value="mean")
        static class MeanChannelsOp
        extends ReduceChannelsOp {
            MeanChannelsOp() {
            }

            @Override
            protected int getReduceOp() {
                return 1;
            }

            @Override
            protected String reduceName() {
                return "Mean";
            }
        }

        @OpType(value="minimum")
        static class MinChannelsOp
        extends ReduceChannelsOp {
            MinChannelsOp() {
            }

            @Override
            protected int getReduceOp() {
                return 3;
            }

            @Override
            protected String reduceName() {
                return "Minimum";
            }
        }

        @OpType(value="maximum")
        static class MaxChannelsOp
        extends ReduceChannelsOp {
            MaxChannelsOp() {
            }

            @Override
            protected int getReduceOp() {
                return 2;
            }

            @Override
            protected String reduceName() {
                return "Maximum";
            }
        }

        static abstract class ReduceChannelsOp
        implements ImageOp {
            ReduceChannelsOp() {
            }

            @Override
            public Mat apply(Mat input) {
                if (input.channels() <= 1) {
                    return input;
                }
                Mat temp = input.reshape(1, input.rows() * input.cols());
                opencv_core.reduce((Mat)temp, (Mat)temp, (int)1, (int)this.getReduceOp());
                temp = temp.reshape(1, input.rows());
                return temp;
            }

            protected abstract int getReduceOp();

            protected abstract String reduceName();

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                List<String> allNames = channels.stream().map(c -> c.getName()).toList();
                String name = this.reduceName() + " [" + String.join((CharSequence)", ", allNames) + "]";
                return ImageChannel.getChannelList((String[])new String[]{name});
            }
        }
    }

    @OpType(value="filters")
    public static class Filters {
        public static ImageOp gaussianBlur(double sigmaX, double sigmaY) {
            return new GaussianFilterOp(sigmaX, sigmaY);
        }

        public static ImageOp gaussianBlur(double sigma) {
            return Filters.gaussianBlur(sigma, sigma);
        }

        public static ImageOp filter2D(Mat kernel) {
            return new FilterOp(kernel);
        }

        public static ImageOp mean(int radius) {
            return new MeanFilterOp(radius);
        }

        public static ImageOp sum(int radius) {
            return new SumFilterOp(radius);
        }

        public static ImageOp variance(int radius) {
            return new VarianceFilterOp(radius);
        }

        public static ImageOp stdDev(int radius) {
            return new StdDevFilterOp(radius);
        }

        public static ImageOp features(Collection<MultiscaleFeatures.MultiscaleFeature> features, double sigmaX, double sigmaY) {
            return new MultiscaleFeatureOp(features, sigmaX, sigmaY);
        }

        public static ImageOp maximum(int radius) {
            return new MaximumFilterOp(radius);
        }

        public static ImageOp minimum(int radius) {
            return new MinimumFilterOp(radius);
        }

        public static ImageOp opening(int radius) {
            return new MorphOpenFilterOp(radius);
        }

        public static ImageOp closing(int radius) {
            return new MorphCloseFilterOp(radius);
        }

        public static ImageOp median(int radius) {
            return new MedianFilterOp(radius);
        }

        static Mat createDefaultKernel(int radius) {
            Size size = new Size(radius * 2 + 1, radius * 2 + 1);
            if (radius == 1) {
                return opencv_imgproc.getStructuringElement((int)0, (Size)size);
            }
            return opencv_imgproc.getStructuringElement((int)2, (Size)size);
        }

        @OpType(value="gaussian")
        static class GaussianFilterOp
        extends PaddedOp {
            private double sigmaX;
            private double sigmaY;

            GaussianFilterOp(double sigmaX, double sigmaY) {
                this.sigmaX = sigmaX;
                this.sigmaY = sigmaY;
            }

            @Override
            public List<Mat> transformPadded(Mat input) {
                if (this.sigmaX == 0.0 && this.sigmaY == 0.0) {
                    return Collections.singletonList(input);
                }
                Padding padding = this.getPadding();
                Size size = new Size(padding.getX1() * 2 + 1, padding.getY1() * 2 + 1);
                OpenCVTools.applyToChannels(input, mat -> opencv_imgproc.GaussianBlur((Mat)mat, (Mat)mat, (Size)size, (double)this.sigmaX, (double)this.sigmaY, (int)2));
                return Collections.singletonList(input);
            }

            @Override
            protected Padding calculatePadding() {
                return ImageOps.getDefaultGaussianPadding(this.sigmaX, this.sigmaY);
            }
        }

        @OpType(value="filter2d")
        static class FilterOp
        extends PaddedOp {
            private Mat kernel;

            FilterOp(Mat kernel) {
                this.kernel = kernel;
            }

            @Override
            protected Padding calculatePadding() {
                int x = this.kernel.cols() / 2;
                int y = this.kernel.rows() / 2;
                return Padding.getPadding((int)x, (int)y);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                List<Mat> split = OpenCVTools.splitChannels(input);
                for (Mat mat : split) {
                    opencv_imgproc.filter2D((Mat)mat, (Mat)mat, (int)-1, (Mat)this.kernel);
                }
                return split;
            }
        }

        @OpType(value="mean")
        static class MeanFilterOp
        extends PaddedOp {
            private int radius;

            MeanFilterOp(int radius) {
                this.radius = radius;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                OpenCVTools.meanFilter(input, this.radius);
                return Collections.singletonList(input);
            }
        }

        @OpType(value="sum")
        static class SumFilterOp
        extends PaddedOp {
            private int radius;

            SumFilterOp(int radius) {
                this.radius = radius;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                OpenCVTools.sumFilter(input, this.radius);
                return Collections.singletonList(input);
            }
        }

        @OpType(value="variance")
        static class VarianceFilterOp
        extends PaddedOp {
            private int radius;

            VarianceFilterOp(int radius) {
                this.radius = radius;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                OpenCVTools.varianceFilter(input, this.radius);
                return Collections.singletonList(input);
            }
        }

        @OpType(value="stddev")
        static class StdDevFilterOp
        extends PaddedOp {
            private int radius;

            StdDevFilterOp(int radius) {
                this.radius = radius;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                OpenCVTools.stdDevFilter(input, this.radius);
                return Collections.singletonList(input);
            }
        }

        @OpType(value="multiscale")
        static class MultiscaleFeatureOp
        extends PaddedOp {
            private List<MultiscaleFeatures.MultiscaleFeature> features;
            private double sigmaX;
            private double sigmaY;
            private transient MultiscaleFeatures.MultiscaleResultsBuilder builder;

            MultiscaleFeatureOp(Collection<MultiscaleFeatures.MultiscaleFeature> features, double sigmaX, double sigmaY) {
                this.features = new ArrayList<MultiscaleFeatures.MultiscaleFeature>(new LinkedHashSet<MultiscaleFeatures.MultiscaleFeature>(features));
                this.sigmaX = sigmaX;
                this.sigmaY = sigmaY;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.padValue());
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                MultiscaleFeatures.MultiscaleResultsBuilder builder = this.getBuilder();
                try (PointerScope scope = new PointerScope();){
                    ArrayList<Mat> output = new ArrayList<Mat>();
                    List<Mat> channels = OpenCVTools.splitChannels(input);
                    for (Mat mat : channels) {
                        MultiscaleFeatures.MultiscaleResultsBuilder.FeatureMap results = builder.build(mat);
                        for (MultiscaleFeatures.MultiscaleFeature f : this.features) {
                            Mat temp = (Mat)results.get((Object)f);
                            temp.retainReference();
                            output.add(temp);
                        }
                    }
                    ArrayList<Mat> arrayList = output;
                    return arrayList;
                }
            }

            @Override
            public List<ImageChannel> getChannels(List<ImageChannel> channels) {
                ArrayList<ImageChannel> list = new ArrayList<ImageChannel>();
                for (ImageChannel c : channels) {
                    Integer color = c.getColor();
                    String name = c.getName();
                    for (MultiscaleFeatures.MultiscaleFeature f : this.features) {
                        list.add(ImageChannel.getInstance((String)String.format("%s (%s, sigma=%.1f,%.1f)", name, f.toString(), this.sigmaX, this.sigmaY), (Integer)color));
                    }
                }
                return list;
            }

            private int padValue() {
                return (int)(Math.ceil(Math.max(this.sigmaX, this.sigmaY) * 4.0) * 2.0 + 1.0);
            }

            private MultiscaleFeatures.MultiscaleResultsBuilder getBuilder() {
                if (this.builder == null) {
                    MultiscaleFeatures.MultiscaleResultsBuilder b = new MultiscaleFeatures.MultiscaleResultsBuilder(this.features);
                    b.sigmaX(this.sigmaX);
                    b.sigmaY(this.sigmaY);
                    this.builder = b;
                }
                return this.builder;
            }
        }

        @OpType(value="maximum")
        static class MaximumFilterOp
        extends MorphOp {
            MaximumFilterOp(int radius) {
                super(radius);
            }

            @Override
            protected int getOp() {
                return 1;
            }
        }

        @OpType(value="minimum")
        static class MinimumFilterOp
        extends MorphOp {
            MinimumFilterOp(int radius) {
                super(radius);
            }

            @Override
            protected int getOp() {
                return 0;
            }
        }

        @OpType(value="morph-open")
        static class MorphOpenFilterOp
        extends MorphOp {
            MorphOpenFilterOp(int radius) {
                super(radius);
            }

            @Override
            protected int getOp() {
                return 2;
            }
        }

        @OpType(value="morph-close")
        static class MorphCloseFilterOp
        extends MorphOp {
            MorphCloseFilterOp(int radius) {
                super(radius);
            }

            @Override
            protected int getOp() {
                return 3;
            }
        }

        @OpType(value="median")
        static class MedianFilterOp
        extends PaddedOp {
            private int radius;

            MedianFilterOp(int radius) {
                this.radius = radius;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                int c;
                if (this.radius > 2 && input.depth() != 0) {
                    logger.warn("MedianOp requires uint8 image for radius > 2");
                }
                if ((c = input.channels()) == 1 || c == 3 || c == 4) {
                    opencv_imgproc.medianBlur((Mat)input, (Mat)input, (int)(this.radius * 2 + 1));
                } else {
                    OpenCVTools.applyToChannels(input, m -> opencv_imgproc.medianBlur((Mat)m, (Mat)m, (int)(this.radius * 2 + 1)));
                }
                return Collections.singletonList(input);
            }
        }

        @OpType(value="fast-minima")
        static class FastMinimaOp
        extends PaddedOp {
            private int radius;
            private transient Mat kernel;

            FastMinimaOp(int radius) {
                this.radius = radius;
            }

            @Override
            public List<Mat> transformPadded(Mat input) {
                Mat temp = new Mat();
                opencv_imgproc.morphologyEx((Mat)input, (Mat)temp, (int)0, (Mat)this.getKernel());
                input.put(opencv_core.equals((Mat)input, (Mat)temp));
                temp.close();
                return Collections.singletonList(input);
            }

            private Mat getKernel() {
                if (this.kernel == null || this.kernel.isNull()) {
                    this.kernel = Filters.createDefaultKernel(this.radius);
                }
                return this.kernel;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }
        }

        @OpType(value="fast-maxima")
        static class FastMaximaOp
        extends PaddedOp {
            private int radius;
            private transient Mat kernel;

            FastMaximaOp(int radius) {
                this.radius = radius;
            }

            @Override
            public List<Mat> transformPadded(Mat input) {
                Mat temp = new Mat();
                opencv_imgproc.morphologyEx((Mat)input, (Mat)temp, (int)1, (Mat)this.getKernel());
                input.put(opencv_core.equals((Mat)input, (Mat)temp));
                temp.close();
                return Collections.singletonList(input);
            }

            private Mat getKernel() {
                if (this.kernel == null || this.kernel.isNull()) {
                    this.kernel = Filters.createDefaultKernel(this.radius);
                }
                return this.kernel;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }
        }

        static abstract class MorphOp
        extends PaddedOp {
            private int radius;
            private transient Mat kernel;

            MorphOp(int radius) {
                this.radius = radius;
            }

            @Override
            public List<Mat> transformPadded(Mat input) {
                opencv_imgproc.morphologyEx((Mat)input, (Mat)input, (int)this.getOp(), (Mat)this.getKernel());
                return Collections.singletonList(input);
            }

            protected abstract int getOp();

            private Mat getKernel() {
                if (this.kernel == null || this.kernel.isNull()) {
                    this.kernel = Filters.createDefaultKernel(this.radius);
                }
                return this.kernel;
            }

            @Override
            protected Padding calculatePadding() {
                return Padding.symmetric((int)this.radius);
            }
        }
    }

    @OpType(value="normalize")
    public static class Normalize {
        public static ImageOp minMax(double outputMin, double outputMax) {
            return new NormalizeMinMaxOp(outputMin, outputMax);
        }

        public static ImageOp minMax() {
            return Normalize.minMax(0.0, 1.0);
        }

        public static ImageOp percentile(double percentileMin, double percentileMax) {
            return Normalize.percentile(percentileMin, percentileMax, true, 0.0);
        }

        public static ImageOp percentile(double percentileMin, double percentileMax, boolean perChannel, double eps) {
            return new NormalizePercentileOp(percentileMin, percentileMax, perChannel, eps);
        }

        public static ImageOp channelSum(double maxValue) {
            return new NormalizeChannelsOp(maxValue, false);
        }

        public static ImageOp channelSoftmax(double maxValue) {
            return new NormalizeChannelsOp(maxValue, true);
        }

        public static ImageOp sigmoid() {
            return new SigmoidOp();
        }

        public static ImageOp zeroMeanUnitVariance(boolean perChannel) {
            return Normalize.zeroMeanUnitVariance(perChannel, 0.0);
        }

        public static ImageOp zeroMeanUnitVariance(boolean perChannel, double eps) {
            return new ZeroMeanUnitVarianceOp(perChannel, eps);
        }

        public static ImageOp localNormalization(double sigmaMean, double sigmaVariance) {
            return new LocalNormalizationOp(sigmaMean, sigmaVariance);
        }

        public static ImageOp localNormalizationMinMax(int radius, double sigma) {
            return new LocalMinMaxNormalizationOp(radius, sigma);
        }

        private static double sigmoid(double input) {
            return 1.0 / (1.0 + Math.exp(-input));
        }

        @OpType(value="min-max")
        static class NormalizeMinMaxOp
        implements ImageOp {
            private double outputMin = 0.0;
            private double outputMax = 1.0;

            NormalizeMinMaxOp(double outputMin, double outputMax) {
                this.outputMin = outputMin;
                this.outputMax = outputMax;
            }

            @Override
            public Mat apply(Mat input) {
                MatVector matvec = new MatVector();
                opencv_core.split((Mat)input, (MatVector)matvec);
                int i = 0;
                while ((long)i < matvec.size()) {
                    Mat mat = matvec.get((long)i);
                    opencv_core.normalize((Mat)mat, (Mat)mat, (double)this.outputMin, (double)this.outputMax, (int)32, (int)-1, null);
                    ++i;
                }
                opencv_core.merge((MatVector)matvec, (Mat)input);
                return input;
            }
        }

        @OpType(value="percentile")
        static class NormalizePercentileOp
        implements ImageOp {
            private double[] percentiles;
            private boolean perChannel = true;
            private double eps = 0.0;

            NormalizePercentileOp(double percentileMin, double percentileMax, boolean perChannel, double eps) {
                this.percentiles = new double[]{percentileMin, percentileMax};
                this.perChannel = perChannel;
                this.eps = eps;
                if (percentileMin == percentileMax) {
                    throw new IllegalArgumentException("Percentile min and max values cannot be identical!");
                }
            }

            @Override
            public Mat apply(Mat input) {
                if (this.perChannel) {
                    MatVector matvec = new MatVector();
                    opencv_core.split((Mat)input, (MatVector)matvec);
                    int i = 0;
                    while ((long)i < matvec.size()) {
                        Mat mat = matvec.get((long)i);
                        this.applyJoint(mat);
                        ++i;
                    }
                    opencv_core.merge((MatVector)matvec, (Mat)input);
                } else {
                    this.applyJoint(input);
                }
                return input;
            }

            private Mat applyJoint(Mat mat) {
                double scale;
                double[] range;
                Mat matTemp = mat;
                Padding padding = this.getPadding();
                if (!padding.isEmpty()) {
                    matTemp = OpenCVTools.crop(mat, padding);
                }
                if ((range = OpenCVTools.percentiles(matTemp, this.percentiles))[1] == range[0] && this.eps == 0.0) {
                    logger.warn("Normalization percentiles give the same value ({}), scale will be Infinity", (Object)range[0]);
                    scale = Double.POSITIVE_INFINITY;
                } else {
                    scale = 1.0 / (range[1] - range[0] + this.eps);
                }
                double offset = -range[0];
                mat.convertTo(mat, mat.type(), scale, offset * scale);
                return mat;
            }
        }

        @OpType(value="channels")
        static class NormalizeChannelsOp
        implements ImageOp {
            private double maxValue;
            private boolean doSoftmax;

            NormalizeChannelsOp(double maxValue, boolean doSoftmax) {
                this.maxValue = maxValue;
                this.doSoftmax = doSoftmax;
            }

            @Override
            public Mat apply(Mat input) {
                int nChannels = input.channels();
                int nRows = input.rows();
                input.put(input.reshape(1, input.rows() * input.cols()));
                ImageOps.rescaleChannelsToProbabilities(input, input, this.maxValue, this.doSoftmax);
                input.put(input.reshape(nChannels, nRows));
                return input;
            }
        }

        @OpType(value="sigmoid")
        static class SigmoidOp
        implements ImageOp {
            SigmoidOp() {
            }

            @Override
            public Mat apply(Mat input) {
                OpenCVTools.apply(input, Normalize::sigmoid);
                return input;
            }
        }

        @OpType(value="zero-mean-unit-variance")
        static class ZeroMeanUnitVarianceOp
        implements ImageOp {
            private boolean perChannel;
            private double eps = 0.0;

            ZeroMeanUnitVarianceOp(boolean perChannel, double eps) {
                this.perChannel = perChannel;
                this.eps = eps;
            }

            @Override
            public Mat apply(Mat input) {
                if (this.perChannel && input.channels() > 1) {
                    OpenCVTools.applyToChannels(input, m -> this.apply((Mat)m));
                    return input;
                }
                double mean = OpenCVTools.mean(input);
                double stdDev = OpenCVTools.stdDev(input);
                if (stdDev == 0.0 && this.eps == 0.0) {
                    OpenCVTools.apply(input, d -> 0.0);
                } else {
                    OpenCVTools.apply(input, d -> (d - mean) / (stdDev + this.eps));
                }
                return input;
            }
        }

        @OpType(value="local")
        static class LocalNormalizationOp
        extends PaddedOp {
            private double sigmaMean;
            private double sigmaStdDev;

            LocalNormalizationOp(double sigmaMean, double sigmaStdDev) {
                this.sigmaMean = sigmaMean;
                this.sigmaStdDev = sigmaStdDev;
            }

            @Override
            protected Padding calculatePadding() {
                double sigma = Math.max(this.sigmaMean, this.sigmaStdDev);
                return ImageOps.getDefaultGaussianPadding(sigma, sigma);
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                int depth = input.depth();
                List<Mat> channels = OpenCVTools.splitChannels(input);
                for (Mat m : channels) {
                    LocalNormalization.gaussianNormalize2D(m, this.sigmaMean, this.sigmaStdDev, 2);
                }
                OpenCVTools.mergeChannels(channels, input);
                if (depth != 6) {
                    depth = 5;
                }
                input.convertTo(input, depth);
                return Collections.singletonList(input);
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return inputType == PixelType.FLOAT64 ? inputType : PixelType.FLOAT32;
            }
        }

        @OpType(value="local-min-max")
        static class LocalMinMaxNormalizationOp
        extends PaddedOp {
            private int radius;
            private double sigmaSmooth;

            LocalMinMaxNormalizationOp(int radius, double sigmaSmooth) {
                if (radius < 1) {
                    throw new IllegalArgumentException("Radius must be greater than 0");
                }
                this.radius = radius;
                this.sigmaSmooth = Math.max(0.0, sigmaSmooth);
            }

            @Override
            protected Padding calculatePadding() {
                return ImageOps.getDefaultGaussianPadding(this.sigmaSmooth, this.sigmaSmooth).add(Padding.symmetric((int)((int)Math.ceil(this.radius))));
            }

            @Override
            protected List<Mat> transformPadded(Mat input) {
                int depth = input.depth();
                if (depth != 6 && depth != 5) {
                    input.convertTo(input, 5);
                }
                List<Mat> channels = OpenCVTools.splitChannels(input);
                Size size = new Size(this.radius * 2 + 1, this.radius * 2 + 1);
                Mat strel = opencv_imgproc.getStructuringElement((int)2, (Size)size);
                Mat tempMin = new Mat();
                Mat tempMax = new Mat();
                Size sizeGaussian = null;
                if (this.sigmaSmooth > 0.0) {
                    int s = (int)Math.ceil(this.sigmaSmooth * 4.0) * 2 + 1;
                    sizeGaussian = new Size(s, s);
                }
                for (Mat m : channels) {
                    opencv_imgproc.erode((Mat)m, (Mat)tempMin, (Mat)strel);
                    if (this.sigmaSmooth > 0.0) {
                        opencv_imgproc.GaussianBlur((Mat)tempMin, (Mat)tempMin, (Size)sizeGaussian, (double)this.sigmaSmooth);
                    }
                    opencv_imgproc.dilate((Mat)m, (Mat)tempMax, (Mat)strel);
                    if (this.sigmaSmooth > 0.0) {
                        opencv_imgproc.GaussianBlur((Mat)tempMax, (Mat)tempMax, (Size)sizeGaussian, (double)this.sigmaSmooth);
                    }
                    opencv_core.subtract((Mat)m, (Mat)tempMin, (Mat)m);
                    opencv_core.subtract((Mat)tempMax, (Mat)tempMin, (Mat)tempMax);
                    m.put(opencv_core.divide((Mat)m, (MatExpr)opencv_core.max((Mat)tempMax, (double)1.0E-6)));
                }
                tempMin.close();
                tempMax.close();
                strel.close();
                if (sizeGaussian != null) {
                    sizeGaussian.close();
                }
                OpenCVTools.mergeChannels(channels, input);
                input.convertTo(input, depth);
                return Collections.singletonList(input);
            }

            @Override
            public PixelType getOutputType(PixelType inputType) {
                return inputType == PixelType.FLOAT64 ? inputType : PixelType.FLOAT32;
            }
        }
    }
}

