/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.experimental.pixels;

import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.experimental.pixels.ImageSupplier;
import qupath.lib.experimental.pixels.MaskSupplier;
import qupath.lib.experimental.pixels.OutputHandler;
import qupath.lib.experimental.pixels.Parameters;
import qupath.lib.experimental.pixels.Processor;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.utils.ObjectMerger;
import qupath.lib.objects.utils.ObjectProcessor;
import qupath.lib.objects.utils.Tiler;
import qupath.lib.plugins.PathTask;
import qupath.lib.plugins.TaskRunner;
import qupath.lib.plugins.TaskRunnerUtils;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.Padding;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.interfaces.ROI;

public class PixelProcessor<S, T, U> {
    private static final Logger logger = LoggerFactory.getLogger(PixelProcessor.class);
    private final ImageSupplier<S> imageSupplier;
    private final MaskSupplier<S, T> maskSupplier;
    private final OutputHandler<S, T, U> outputHandler;
    private final Processor<S, T, U> processor;
    private final Padding padding;
    private final DownsampleCalculator downsampleCalculator;
    private final Tiler tiler;
    private final ObjectProcessor objectProcessor;

    private PixelProcessor(ImageSupplier<S> imageSupplier, MaskSupplier<S, T> maskSupplier, OutputHandler<S, T, U> outputHandler, Processor<S, T, U> processor, Tiler tiler, ObjectProcessor objectProcessor, Padding padding, DownsampleCalculator downsampleCalculator) {
        Objects.requireNonNull(imageSupplier, "Image supplier cannot be null");
        Objects.requireNonNull(processor, "Processor cannot be null");
        if (downsampleCalculator == null) {
            throw new IllegalArgumentException("Downsample must be specified");
        }
        this.imageSupplier = imageSupplier;
        this.maskSupplier = maskSupplier;
        this.outputHandler = outputHandler;
        this.processor = processor;
        this.tiler = tiler;
        this.objectProcessor = objectProcessor;
        this.padding = padding;
        this.downsampleCalculator = downsampleCalculator;
    }

    public void processObjects(ImageData<BufferedImage> imageData, Collection<? extends PathObject> pathObjects) {
        this.processObjects(TaskRunnerUtils.getDefaultInstance().createTaskRunner(), imageData, pathObjects);
    }

    public void processObjects(TaskRunner runner, ImageData<BufferedImage> imageData, Collection<? extends PathObject> pathObjects) {
        if (this.tiler != null) {
            this.processTiled(runner, this.tiler, imageData, pathObjects);
        } else {
            this.processUntiled(runner, imageData, pathObjects);
        }
    }

    private void processUntiled(TaskRunner runner, ImageData<BufferedImage> imageData, Collection<? extends PathObject> pathObjects) {
        List<ProcessorTask> tasks = pathObjects.stream().distinct().map(pathObject -> new ProcessorTask(imageData, (PathObject)pathObject, this.processor, null)).toList();
        runner.runTasks(tasks);
    }

    private void processTiled(TaskRunner runner, Tiler tiler, ImageData<BufferedImage> imageData, Collection<? extends PathObject> pathObjects) {
        if (tiler == null) {
            throw new IllegalStateException("Tiler must be specified for tiled processing");
        }
        ArrayList<ProcessorTask> tasks = new ArrayList<ProcessorTask>();
        LinkedHashMap tempObjects = new LinkedHashMap();
        for (PathObject pathObject : pathObjects) {
            if (tempObjects.containsKey(pathObject)) continue;
            ArrayList proxyList = new ArrayList();
            if (pathObject.isRootObject()) {
                ImageServer server = imageData.getServer();
                for (int t = 0; t < server.nTimepoints(); ++t) {
                    for (int z = 0; z < server.nZSlices(); ++z) {
                        ROI roi = ROIs.createRectangleROI((double)0.0, (double)0.0, (double)server.getWidth(), (double)server.getHeight(), (ImagePlane)ImagePlane.getPlane((int)t, (int)z));
                        proxyList.addAll(tiler.createAnnotations(roi));
                    }
                }
            } else {
                proxyList.addAll(tiler.createAnnotations(pathObject.getROI()));
            }
            tempObjects.put(pathObject, proxyList);
            for (PathObject proxy2 : proxyList) {
                tasks.add(new ProcessorTask(imageData, pathObject, this.processor, proxy2));
            }
        }
        String message = tasks.size() == 1 ? "Processing 1 tile" : "Processing " + tasks.size() + " tiles";
        runner.runTasks(message, tasks);
        if (runner.isCancelled() || Thread.interrupted()) {
            logger.warn("Tiled processing cancelled before merging");
            return;
        }
        ArrayList<Runnable> arrayList = new ArrayList<Runnable>();
        for (Map.Entry entry : tempObjects.entrySet()) {
            PathObject pathObject = (PathObject)entry.getKey();
            List proxyList = ((List)entry.getValue()).stream().flatMap(proxy -> proxy.getChildObjects().stream()).toList();
            if (this.objectProcessor != null) {
                arrayList.add(() -> PixelProcessor.postprocessObjects(this.objectProcessor, pathObject, proxyList));
                continue;
            }
            pathObject.clearChildObjects();
            pathObject.addChildObjects(proxyList);
            pathObject.setLocked(true);
        }
        if (!arrayList.isEmpty()) {
            runner.runTasks("Post-processing", arrayList);
        }
    }

    private static void postprocessObjects(ObjectProcessor objectProcessor, PathObject parent, List<PathObject> childObjects) {
        List toAdd = objectProcessor.process(childObjects);
        parent.clearChildObjects();
        parent.addChildObjects((Collection)toAdd);
        parent.setLocked(true);
    }

    public static <S, T, U> Builder<S, T, U> builder() {
        return new Builder();
    }

    private static class DownsampleCalculator {
        private final boolean isRequestedPixelSize;
        private final double resolution;

        private DownsampleCalculator(double resolution, boolean isRequestedPixelSize) {
            this.isRequestedPixelSize = isRequestedPixelSize;
            this.resolution = resolution;
        }

        private static DownsampleCalculator createForDownsample(double downsample) {
            return new DownsampleCalculator(downsample, false);
        }

        private static DownsampleCalculator createForRequestedPixelSize(double pixelSize) {
            return new DownsampleCalculator(pixelSize, true);
        }

        public double getDownsample(PixelCalibration cal) {
            if (this.isRequestedPixelSize) {
                return this.resolution / cal.getAveragedPixelSize().doubleValue();
            }
            return this.resolution;
        }
    }

    private class ProcessorTask
    implements PathTask {
        private static final Logger logger = LoggerFactory.getLogger(PixelProcessor.class);
        private final ImageData<BufferedImage> imageData;
        private final PathObject pathObject;
        private final PathObject parentProxy;
        private final Processor<S, T, U> processor;

        private ProcessorTask(ImageData<BufferedImage> imageData, PathObject pathObject, Processor<S, T, U> processor, PathObject parentProxy) {
            this.imageData = imageData;
            this.pathObject = pathObject;
            this.processor = processor;
            this.parentProxy = parentProxy;
        }

        public void run() {
            try {
                if (Thread.currentThread().isInterrupted()) {
                    logger.trace("Thread interrupted - skipping task for {}", (Object)this.pathObject);
                    return;
                }
                RegionRequest request = this.parentProxy != null ? this.createRequest(this.imageData.getServer(), this.parentProxy) : this.createRequest(this.imageData.getServer(), this.pathObject);
                Parameters.Builder builder = Parameters.builder();
                Parameters params = builder.imageData(this.imageData).imageFunction(PixelProcessor.this.imageSupplier).maskFunction(PixelProcessor.this.maskSupplier).region(request).parent(this.pathObject).parentProxy(this.parentProxy).build();
                Object output = this.processor.process(params);
                if (PixelProcessor.this.outputHandler != null) {
                    PixelProcessor.this.outputHandler.handleOutput(params, output);
                }
            }
            catch (Exception e) {
                logger.error("Error processing object", (Throwable)e);
            }
        }

        protected RegionRequest createRequest(ImageServer<?> server, PathObject pathObject) {
            double downsample = PixelProcessor.this.downsampleCalculator.getDownsample(server.getPixelCalibration());
            return RegionRequest.createInstance((String)server.getPath(), (double)downsample, (ROI)pathObject.getROI()).pad2D(PixelProcessor.this.padding).intersect2D(0, 0, server.getWidth(), server.getHeight());
        }

        public void taskComplete(boolean wasCancelled) {
            super.taskComplete(wasCancelled);
        }

        public String getLastResultsDescription() {
            return "Completed " + String.valueOf(this.pathObject);
        }
    }

    public static class Builder<S, T, U> {
        private ImageSupplier<S> imageSupplier;
        private MaskSupplier<S, T> maskSupplier;
        private OutputHandler<S, T, U> outputHandler;
        private Processor<S, T, U> processor;
        private Tiler tiler;
        private ObjectProcessor objectProcessor;
        private Padding padding = Padding.empty();
        private DownsampleCalculator downsampleCalculator = DownsampleCalculator.createForDownsample(1.0);

        public Builder<S, T, U> imageSupplier(ImageSupplier<S> imageSupplier) {
            this.imageSupplier = imageSupplier;
            return this;
        }

        public Builder<S, T, U> maskSupplier(MaskSupplier<S, T> maskSupplier) {
            this.maskSupplier = maskSupplier;
            return this;
        }

        public Builder<S, T, U> outputHandler(OutputHandler<S, T, U> outputHandler) {
            this.outputHandler = outputHandler;
            return this;
        }

        public Builder<S, T, U> processor(Processor<S, T, U> processor) {
            this.processor = processor;
            return this;
        }

        public Builder<S, T, U> padding(Padding padding) {
            this.padding = padding;
            return this;
        }

        public Builder<S, T, U> padding(int size) {
            this.padding = size <= 0 ? Padding.empty() : Padding.symmetric((int)size);
            return this;
        }

        public Builder<S, T, U> downsample(double downsample) {
            if (downsample <= 0.0) {
                throw new IllegalArgumentException("Downsample must be > 0!");
            }
            this.downsampleCalculator = DownsampleCalculator.createForDownsample(downsample);
            return this;
        }

        public Builder<S, T, U> pixelSize(double pixelSize) {
            if (pixelSize <= 0.0) {
                throw new IllegalArgumentException("Requested pixel size must be > 0!");
            }
            this.downsampleCalculator = DownsampleCalculator.createForRequestedPixelSize(pixelSize);
            return this;
        }

        public Builder<S, T, U> tiler(Tiler tiler) {
            this.tiler = tiler;
            return this;
        }

        public Builder<S, T, U> tile(int tileWidth, int tileHeight) {
            return this.tiler(Tiler.builder((int)tileWidth, (int)tileHeight).alignCenter().filterByCentroid(false).cropTiles(false).build());
        }

        @Deprecated
        public Builder<S, T, U> merger(ObjectMerger merger) {
            return this.postProcess((ObjectProcessor)merger);
        }

        public Builder<S, T, U> postProcess(ObjectProcessor objectProcessor) {
            this.objectProcessor = objectProcessor;
            return this;
        }

        public Builder<S, T, U> mergeSharedBoundaries(double threshold) {
            return this.postProcess((ObjectProcessor)ObjectMerger.createSharedTileBoundaryMerger((double)threshold));
        }

        public PixelProcessor<S, T, U> build() {
            return new PixelProcessor<S, T, U>(this.imageSupplier, this.maskSupplier, this.outputHandler, this.processor, this.tiler, this.objectProcessor, this.padding, this.downsampleCalculator);
        }
    }
}

