/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.analysis.heatmaps;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import org.bytedeco.javacpp.PointerScope;
import org.bytedeco.opencv.global.opencv_core;
import org.bytedeco.opencv.opencv_core.Mat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.heatmaps.ColorModels;
import qupath.lib.analysis.heatmaps.DensityMapDataOp;
import qupath.lib.awt.common.BufferedImageTools;
import qupath.lib.classifiers.pixel.PixelClassifier;
import qupath.lib.classifiers.pixel.PixelClassifierMetadata;
import qupath.lib.geom.Point;
import qupath.lib.geom.Point2;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.io.GsonTools;
import qupath.lib.objects.PathAnnotationObject;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectPredicates;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.PathObjects;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;
import qupath.opencv.ml.pixel.PixelClassifierTools;
import qupath.opencv.ml.pixel.PixelClassifiers;
import qupath.opencv.tools.OpenCVTools;

public class DensityMaps {
    private static final Logger logger = LoggerFactory.getLogger(DensityMaps.class);
    public static final String CHANNEL_ALL_OBJECTS = "Counts";
    private static int preferredTileSize = 2048;
    public static final String PROJECT_LOCATION = "classifiers/density_maps";

    public static DensityMapBuilder builder(PathObjectPredicates.PathObjectPredicate mainObjectFilter) {
        return new DensityMapBuilder(mainObjectFilter);
    }

    public static DensityMapBuilder builder(DensityMapBuilder builder) {
        return new DensityMapBuilder(builder.params);
    }

    private static PixelClassifier createClassifier(ImageData<BufferedImage> imageData, DensityMapParameters params) {
        PixelCalibration pixelSize = params.pixelSize;
        if (pixelSize == null) {
            if (imageData == null) {
                throw new IllegalArgumentException("You need to specify a pixel size or provide an ImageData to generate a density map!");
            }
            PixelCalibration cal = imageData.getServer().getPixelCalibration();
            double radius = params.radius;
            ImageServer server = imageData.getServer();
            double maxDownsample = Math.round(Math.max((double)server.getWidth() / (double)params.maxWidth, (double)server.getHeight() / (double)params.maxHeight));
            if (maxDownsample < 1.0) {
                maxDownsample = 1.0;
            }
            PixelCalibration minPixelSize = cal.createScaledInstance(maxDownsample, maxDownsample);
            if (radius > 0.0) {
                double radiusPixels = radius / cal.getAveragedPixelSize().doubleValue();
                double radiusDownsample = Math.round(radiusPixels / 10.0);
                pixelSize = cal.createScaledInstance(radiusDownsample, radiusDownsample);
            }
            if (pixelSize == null || pixelSize.getAveragedPixelSize().doubleValue() < minPixelSize.getAveragedPixelSize().doubleValue()) {
                pixelSize = minPixelSize;
            }
        }
        int radiusInt = (int)Math.round(params.radius / pixelSize.getAveragedPixelSize().doubleValue());
        logger.debug("Creating classifier with pixel size {}, radius = {}", (Object)pixelSize, (Object)radiusInt);
        DensityMapDataOp dataOp = new DensityMapDataOp(radiusInt, params.secondaryObjectFilters, params.mainObjectFilter, params.densityType);
        PixelClassifierMetadata metadata = new PixelClassifierMetadata.Builder().inputShape(preferredTileSize, preferredTileSize).inputResolution(pixelSize).setChannelType(ImageServerMetadata.ChannelType.DENSITY).outputChannels(dataOp.getChannels()).build();
        return PixelClassifiers.createClassifier(dataOp, metadata);
    }

    public static DensityMapBuilder loadDensityMap(Path path) throws IOException {
        logger.debug("Loading density map from {}", (Object)path);
        try (BufferedReader reader = Files.newBufferedReader(path);){
            DensityMapBuilder densityMapBuilder = (DensityMapBuilder)GsonTools.getInstance().fromJson((Reader)reader, DensityMapBuilder.class);
            return densityMapBuilder;
        }
    }

    public static boolean threshold(PathObjectHierarchy hierarchy, ImageServer<BufferedImage> densityServer, int channel, double threshold, PixelClassifierTools.CreateObjectOptions ... options) throws IOException {
        String pathClassName = densityServer.getChannel(channel).getName();
        return DensityMaps.threshold(hierarchy, densityServer, Map.of(channel, threshold), pathClassName, options);
    }

    public static boolean threshold(PathObjectHierarchy hierarchy, ImageServer<BufferedImage> densityServer, Map<Integer, ? extends Number> thresholds, String pathClassName, PixelClassifierTools.CreateObjectOptions ... options) throws IOException {
        logger.debug("Thresholding {} with thresholds {}, options {}", new Object[]{densityServer, thresholds, Arrays.asList(options)});
        PathClass lessThan = PathClass.StandardPathClasses.IGNORE;
        PathClass greaterThan = PathClass.fromString((String)pathClassName);
        List<PixelClassifierTools.CreateObjectOptions> optionsList = Arrays.asList(options);
        boolean changes = false;
        if (optionsList.contains((Object)PixelClassifierTools.CreateObjectOptions.DELETE_EXISTING)) {
            Collection<Object> toRemove;
            if (hierarchy.getSelectionModel().noSelection()) {
                toRemove = hierarchy.getAnnotationObjects().stream().filter(p -> p.getPathClass() == greaterThan).toList();
            } else {
                toRemove = new HashSet();
                LinkedHashSet selectedObjects = new LinkedHashSet(hierarchy.getSelectionModel().getSelectedObjects());
                for (PathObject selected : selectedObjects) {
                    PathObjectTools.getDescendantObjects((PathObject)selected, toRemove, PathAnnotationObject.class);
                }
                toRemove.removeAll(selectedObjects);
            }
            if (!toRemove.isEmpty()) {
                hierarchy.removeObjects(toRemove, true);
                changes = true;
            }
            options = (PixelClassifierTools.CreateObjectOptions[])optionsList.stream().filter(o -> o != PixelClassifierTools.CreateObjectOptions.DELETE_EXISTING).toArray(PixelClassifierTools.CreateObjectOptions[]::new);
        }
        ImageServer<BufferedImage> thresholdedServer = PixelClassifierTools.createThresholdServer(densityServer, thresholds, lessThan, greaterThan);
        return PixelClassifierTools.createAnnotationsFromPixelClassifier(hierarchy, thresholdedServer, Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY, options) | changes;
    }

    public static void findHotspots(PathObjectHierarchy hierarchy, ImageServer<BufferedImage> densityServer, int channel, int nHotspots, double radius, double minCount, PathClass hotspotClass, boolean deleteExisting, boolean peaksOnly) throws IOException {
        if (nHotspots <= 0) {
            logger.warn("Number of hotspots requested is {}!", (Object)nHotspots);
            return;
        }
        logger.debug("Finding {} hotspots in {} for channel {}, radius {}", new Object[]{nHotspots, densityServer, channel, radius});
        Collection<Object> parents = new ArrayList(hierarchy.getSelectionModel().getSelectedObjects());
        if (parents.isEmpty()) {
            parents = Collections.singleton(hierarchy.getRootObject());
        }
        double downsample = densityServer.getDownsampleForResolution(0);
        HashSet<PathObject> toDelete = new HashSet<PathObject>();
        if (deleteExisting) {
            toDelete.addAll(hierarchy.getAnnotationObjects().stream().filter(p -> p.getPathClass() == hotspotClass && p.isAnnotation() && p.getName() != null && p.getName().startsWith("Hotspot")).toList());
        }
        double radiusPixels = radius / densityServer.getPixelCalibration().getAveragedPixelSize().doubleValue();
        try (PointerScope scope = new PointerScope();){
            for (PathObject pathObject : parents) {
                ROI roiEroded;
                ROI roi = pathObject.getROI();
                if (roi == null) {
                    if (densityServer.nTimepoints() > 1 || densityServer.nZSlices() > 1) {
                        logger.warn("Hotspot detection without a parent object not supported for images with multiple z-slices/timepoints.");
                        logger.warn("I will apply detection to the first plane only. If you need hotspots elsewhere, create an annotation first and use it to define the ROI.");
                    }
                    roi = ROIs.createRectangleROI((double)0.0, (double)0.0, (double)densityServer.getWidth(), (double)densityServer.getHeight(), (ImagePlane)ImagePlane.getDefaultPlane());
                }
                if ((roiEroded = RoiTools.buffer((ROI)roi, (double)(-radiusPixels))).isEmpty() || roiEroded.getArea() == 0.0) {
                    logger.warn("ROI is too small! Cannot detected hotspots with radius {} in {}", (Object)radius, (Object)pathObject);
                    continue;
                }
                ImagePlane plane = roi.getImagePlane();
                RegionRequest request = RegionRequest.createInstance((String)densityServer.getPath(), (double)downsample, (int)0, (int)0, (int)densityServer.getWidth(), (int)densityServer.getHeight(), (int)plane.getZ(), (int)plane.getT());
                BufferedImage img = (BufferedImage)densityServer.readRegion(request);
                BufferedImage imgMask = BufferedImageTools.createROIMask((int)img.getWidth(), (int)img.getHeight(), (ROI)roiEroded, (RegionRequest)request);
                Mat mat = OpenCVTools.imageToMat(img);
                Mat matMask = OpenCVTools.imageToMat(imgMask);
                List<Mat> channels = OpenCVTools.splitChannels(mat);
                Mat density = channels.get(channel);
                if (minCount > 0.0) {
                    Mat thresholdMask = opencv_core.greaterThan((Mat)channels.get(channels.size() - 1), (double)minCount).asMat();
                    opencv_core.bitwise_and((Mat)matMask, (Mat)thresholdMask, (Mat)matMask);
                    thresholdMask.close();
                }
                if (peaksOnly) {
                    Mat matMaxima = OpenCVTools.findRegionalMaxima(density);
                    Mat matPeaks = OpenCVTools.shrinkLabels(matMaxima);
                    matPeaks.put(opencv_core.greaterThan((Mat)matPeaks, (double)0.0));
                    opencv_core.bitwise_and((Mat)matMask, (Mat)matPeaks, (Mat)matMask);
                    matPeaks.close();
                    matMaxima.close();
                }
                ArrayList<OpenCVTools.IndexedPixel> maxima = new ArrayList<OpenCVTools.IndexedPixel>(OpenCVTools.getMaskedPixels(density, matMask));
                Collections.sort(maxima, Comparator.comparingDouble(p -> p.getValue()).reversed());
                List<Point2> points = maxima.stream().map(p -> new Point2((double)p.getX() * downsample, (double)p.getY() * downsample)).toList();
                ArrayList<Point2> hotspotCentroids = new ArrayList<Point2>();
                double distSqThreshold = radiusPixels * radiusPixels * 4.0;
                for (Point2 p2 : points) {
                    boolean skip = false;
                    for (Point2 p22 : hotspotCentroids) {
                        if (!(p2.distanceSq((Point)p22) < distSqThreshold)) continue;
                        skip = true;
                        break;
                    }
                    if (skip) continue;
                    hotspotCentroids.add(p2);
                    if (hotspotCentroids.size() != nHotspots) continue;
                    break;
                }
                ArrayList<PathObject> hotspots = new ArrayList<PathObject>();
                int i = 0;
                for (Point2 p3 : hotspotCentroids) {
                    ROI ellipse = ROIs.createEllipseROI((double)(p3.getX() - radiusPixels), (double)(p3.getY() - radiusPixels), (double)(radiusPixels * 2.0), (double)(radiusPixels * 2.0), (ImagePlane)roi.getImagePlane());
                    PathObject hotspot = PathObjects.createAnnotationObject((ROI)ellipse, (PathClass)hotspotClass);
                    hotspot.setName("Hotspot " + ++i);
                    hotspots.add(hotspot);
                }
                if (hotspots.isEmpty()) {
                    logger.warn("No hotspots found in {}", (Object)pathObject);
                } else if (hotspots.size() < nHotspots) {
                    logger.warn("Only {}/{} hotspots could be found in {}", new Object[]{hotspots.size(), nHotspots, pathObject});
                }
                pathObject.addChildObjects(hotspots);
            }
            hierarchy.fireHierarchyChangedEvent(DensityMaps.class);
            if (!toDelete.isEmpty()) {
                hierarchy.removeObjects(toDelete, true);
            }
        }
    }

    public static class DensityMapBuilder {
        private DensityMapParameters params = null;
        private ColorModels.ColorModelBuilder colorModelBuilder = null;

        private DensityMapBuilder(DensityMapParameters params) {
            Objects.requireNonNull(params);
            this.params = new DensityMapParameters(params);
        }

        private DensityMapBuilder(PathObjectPredicates.PathObjectPredicate allObjects) {
            Objects.requireNonNull(allObjects);
            this.params = new DensityMapParameters();
            this.params.mainObjectFilter = allObjects;
        }

        public DensityMapBuilder pixelSize(PixelCalibration requestedPixelSize) {
            this.params.pixelSize = requestedPixelSize;
            return this;
        }

        public DensityMapBuilder type(DensityMapType type) {
            this.params.densityType = type;
            return this;
        }

        public DensityMapBuilder addDensities(String name, PathObjectPredicates.PathObjectPredicate filter) {
            this.params.secondaryObjectFilters.put(name, filter);
            return this;
        }

        public DensityMapBuilder colorModel(ColorModels.ColorModelBuilder colorModelBuilder) {
            this.colorModelBuilder = colorModelBuilder;
            return this;
        }

        public DensityMapBuilder radius(double radius) {
            this.params.radius = radius;
            return this;
        }

        public DensityMapParameters buildParameters() {
            return new DensityMapParameters(this.params);
        }

        public PixelClassifier buildClassifier(ImageData<BufferedImage> imageData) {
            logger.debug("Building density map classifier for {}", imageData);
            return DensityMaps.createClassifier(imageData, this.params);
        }

        public ImageServer<BufferedImage> buildServer(ImageData<BufferedImage> imageData) {
            logger.debug("Building density map server for {}", imageData);
            PixelClassifier classifier = DensityMaps.createClassifier(imageData, this.params);
            String id = UUID.randomUUID().toString();
            StringBuilder sb = new StringBuilder();
            sb.append("Density map (radius=");
            sb.append(this.params.radius);
            sb.append(")-");
            sb.append(imageData.getServerPath());
            sb.append("-");
            sb.append(id);
            ColorModel colorModel = this.colorModelBuilder == null ? null : this.colorModelBuilder.build();
            return PixelClassifierTools.createPixelClassificationServer(imageData, classifier, sb.toString(), colorModel, true);
        }
    }

    public static class DensityMapParameters {
        private PixelCalibration pixelSize = null;
        private int maxWidth;
        private int maxHeight = this.maxWidth = 1536;
        private double radius = 0.0;
        private DensityMapType densityType = DensityMapType.SUM;
        private PathObjectPredicates.PathObjectPredicate mainObjectFilter;
        private Map<String, PathObjectPredicates.PathObjectPredicate> secondaryObjectFilters = new LinkedHashMap<String, PathObjectPredicates.PathObjectPredicate>();

        private DensityMapParameters() {
        }

        private DensityMapParameters(DensityMapParameters params) {
            this.pixelSize = params.pixelSize;
            this.radius = params.radius;
            this.densityType = params.densityType;
            this.maxWidth = params.maxWidth;
            this.maxHeight = params.maxHeight;
            this.mainObjectFilter = params.mainObjectFilter;
            this.secondaryObjectFilters = new LinkedHashMap<String, PathObjectPredicates.PathObjectPredicate>(params.secondaryObjectFilters);
        }

        public double getRadius() {
            return this.radius;
        }

        public int getMaxWidth() {
            return this.maxWidth;
        }

        public int getMaxHeight() {
            return this.maxHeight;
        }

        public PixelCalibration getPixelSize() {
            return this.pixelSize;
        }

        public DensityMapType getDensityType() {
            return this.densityType;
        }

        public PathObjectPredicates.PathObjectPredicate getMainObjectFilter() {
            return this.mainObjectFilter;
        }

        public Map<String, PathObjectPredicates.PathObjectPredicate> getSecondaryObjectFilters() {
            return Collections.unmodifiableMap(this.secondaryObjectFilters);
        }
    }

    public static enum DensityMapType {
        SUM,
        GAUSSIAN,
        PERCENT;


        public String toString() {
            switch (this.ordinal()) {
                case 0: {
                    return "By area (raw counts)";
                }
                case 2: {
                    return "Objects %";
                }
                case 1: {
                    return "Gaussian-weighted";
                }
            }
            throw new IllegalArgumentException("Unknown enum " + String.valueOf((Object)this));
        }
    }
}

