/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.analysis.features;

import ij.ImagePlus;
import ij.gui.Roi;
import ij.process.ByteProcessor;
import ij.process.ColorProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.math3.stat.descriptive.DescriptiveStatistics;
import org.apache.commons.math3.stat.descriptive.StatisticalSummary;
import org.locationtech.jts.algorithm.Area;
import org.locationtech.jts.algorithm.Length;
import org.locationtech.jts.algorithm.MinimumBoundingCircle;
import org.locationtech.jts.algorithm.MinimumDiameter;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Lineal;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.Polygonal;
import org.locationtech.jts.geom.util.AffineTransformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.imagej.tools.IJTools;
import qupath.imagej.tools.PixelImageIJ;
import qupath.lib.analysis.images.SimpleImage;
import qupath.lib.common.GeneralTools;
import qupath.lib.images.PathImage;
import qupath.lib.images.servers.ImageChannel;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.objects.PathCellObject;
import qupath.lib.objects.PathObject;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.EllipseROI;
import qupath.lib.roi.interfaces.ROI;

public class ObjectMeasurements {
    private static final Logger logger = LoggerFactory.getLogger(ObjectMeasurements.class);
    private static final Collection<ShapeFeatures> ALL_SHAPE_FEATURES = Collections.unmodifiableSet(new HashSet<ShapeFeatures>(Arrays.asList(ShapeFeatures.values())));

    public static void addShapeMeasurements(PathObject pathObject, PixelCalibration cal, ShapeFeatures ... features) {
        ObjectMeasurements.addShapeMeasurements(Collections.singleton(pathObject), cal, features);
    }

    public static void addShapeMeasurements(Collection<? extends PathObject> pathObjects, PixelCalibration cal, ShapeFeatures ... features) {
        PixelCalibration calibration = cal == null || !cal.unitsMatch2D() ? PixelCalibration.getDefaultInstance() : cal;
        Collection<ShapeFeatures> featureCollection = features.length == 0 ? ALL_SHAPE_FEATURES : Arrays.asList(features);
        pathObjects.parallelStream().filter(p -> p.hasROI()).forEach(pathObject -> {
            if (pathObject instanceof PathCellObject) {
                PathCellObject cell = (PathCellObject)pathObject;
                ObjectMeasurements.addCellShapeMeasurements(cell, calibration, featureCollection);
            } else {
                try (MeasurementList ml = pathObject.getMeasurementList();){
                    ObjectMeasurements.addShapeMeasurements(ml, pathObject.getROI(), calibration, "", featureCollection);
                }
            }
        });
    }

    private static Geometry getScaledGeometry(ROI roi, PixelCalibration cal, ShapeFeatures ... features) {
        if (roi == null) {
            return null;
        }
        Geometry geom = roi.getGeometry();
        double pixelWidth = cal.getPixelWidth().doubleValue();
        double pixelHeight = cal.getPixelHeight().doubleValue();
        if (pixelWidth == 1.0 && pixelHeight == 1.0) {
            return geom;
        }
        return AffineTransformation.scaleInstance((double)pixelWidth, (double)pixelHeight).transform(geom);
    }

    private static void addCellShapeMeasurements(PathCellObject cell, PixelCalibration cal, Collection<ShapeFeatures> features) {
        ROI roiNucleus = cell.getNucleusROI();
        ROI roiCell = cell.getROI();
        try (MeasurementList ml = cell.getMeasurementList();){
            if (roiNucleus != null) {
                ObjectMeasurements.addShapeMeasurements(ml, roiNucleus, cal, "Nucleus: ", features);
            }
            if (roiCell != null) {
                ObjectMeasurements.addShapeMeasurements(ml, roiCell, cal, "Cell: ", features);
            }
            if (roiNucleus != null && roiCell != null && features.contains((Object)ShapeFeatures.NUCLEUS_CELL_RATIO)) {
                double pixelWidth = cal.getPixelWidth().doubleValue();
                double pixelHeight = cal.getPixelHeight().doubleValue();
                double nucleusCellAreaRatio = GeneralTools.clipValue((double)(roiNucleus.getScaledArea(pixelWidth, pixelHeight) / roiCell.getScaledArea(pixelWidth, pixelHeight)), (double)0.0, (double)1.0);
                ml.put("Nucleus/Cell area ratio", nucleusCellAreaRatio);
            }
        }
    }

    private static void addShapeMeasurements(MeasurementList ml, ROI roi, PixelCalibration cal, String baseName, Collection<ShapeFeatures> features) {
        if (roi == null) {
            return;
        }
        if (roi instanceof EllipseROI) {
            ObjectMeasurements.addShapeMeasurements(ml, (EllipseROI)roi, cal, baseName, features);
        } else {
            Geometry geom = ObjectMeasurements.getScaledGeometry(roi, cal, new ShapeFeatures[0]);
            String units = cal.getPixelWidthUnit();
            ObjectMeasurements.addShapeMeasurements(ml, geom, baseName, units, features);
        }
    }

    private static void addShapeMeasurements(MeasurementList ml, EllipseROI ellipse, PixelCalibration cal, String baseName, Collection<ShapeFeatures> features) {
        String units = cal.getPixelWidthUnit();
        String units2 = units + "^2";
        if (!((String)baseName).isEmpty() && !((String)baseName).endsWith(" ")) {
            baseName = (String)baseName + " ";
        }
        double pixelWidth = cal.getPixelWidth().doubleValue();
        double pixelHeight = cal.getPixelHeight().doubleValue();
        if (features.contains((Object)ShapeFeatures.AREA)) {
            ml.put((String)baseName + "Area " + units2, ellipse.getScaledArea(pixelWidth, pixelHeight));
        }
        if (features.contains((Object)ShapeFeatures.LENGTH)) {
            ml.put((String)baseName + "Length " + units, ellipse.getLength());
        }
        if (features.contains((Object)ShapeFeatures.CIRCULARITY)) {
            ml.put((String)baseName + "Circularity", 1.0);
        }
        if (features.contains((Object)ShapeFeatures.SOLIDITY)) {
            ml.put((String)baseName + "Solidity", 1.0);
        }
        if (features.contains((Object)ShapeFeatures.MAX_DIAMETER)) {
            double maxDiameter = Math.max(ellipse.getBoundsWidth() * pixelWidth, ellipse.getBoundsHeight() * pixelHeight);
            ml.put((String)baseName + "Max diameter " + units, maxDiameter);
        }
        if (features.contains((Object)ShapeFeatures.MIN_DIAMETER)) {
            double minDiameter = Math.min(ellipse.getBoundsWidth() * pixelWidth, ellipse.getBoundsHeight() * pixelHeight);
            ml.put((String)baseName + "Min diameter " + units, minDiameter);
        }
    }

    private static void addShapeMeasurements(MeasurementList ml, Geometry geom, String baseName, String units, Collection<ShapeFeatures> features) {
        boolean isArea = geom instanceof Polygonal;
        boolean isLine = geom instanceof Lineal;
        String units2 = units + "^2";
        if (!((String)baseName).isEmpty() && !((String)baseName).endsWith(" ")) {
            baseName = (String)baseName + " ";
        }
        double area = geom.getArea();
        double length = geom.getLength();
        if (isArea && features.contains((Object)ShapeFeatures.AREA)) {
            ml.put((String)baseName + "Area " + units2, area);
        }
        if ((isArea || isLine) && features.contains((Object)ShapeFeatures.LENGTH)) {
            ml.put((String)baseName + "Length " + units, length);
        }
        if (isArea && features.contains((Object)ShapeFeatures.CIRCULARITY)) {
            if (geom instanceof Polygon) {
                double ringLength;
                double ringArea;
                Polygon polygon = (Polygon)geom;
                if (polygon.getNumInteriorRing() == 0) {
                    ringArea = area;
                    ringLength = length;
                } else {
                    CoordinateSequence ring = ((Polygon)geom).getExteriorRing().getCoordinateSequence();
                    ringArea = Area.ofRing((CoordinateSequence)ring);
                    ringLength = Length.ofLine((CoordinateSequence)ring);
                }
                double circularity = Math.PI * 4 * ringArea / (ringLength * ringLength);
                ml.put((String)baseName + "Circularity", circularity);
            } else {
                logger.debug("Cannot compute circularity for {}", geom.getClass());
            }
        }
        if (isArea && features.contains((Object)ShapeFeatures.SOLIDITY)) {
            double solidity = area / geom.convexHull().getArea();
            ml.put((String)baseName + "Solidity", solidity);
        }
        if (features.contains((Object)ShapeFeatures.MAX_DIAMETER)) {
            double minCircleRadius = new MinimumBoundingCircle(geom).getRadius();
            ml.put((String)baseName + "Max diameter " + units, minCircleRadius * 2.0);
        }
        if (features.contains((Object)ShapeFeatures.MIN_DIAMETER)) {
            double minDiameter = new MinimumDiameter(geom).getLength();
            ml.put((String)baseName + "Min diameter " + units, minDiameter);
        }
    }

    public static void addIntensityMeasurements(ImageServer<BufferedImage> server, PathObject pathObject, double downsample, Collection<Measurements> measurements, Collection<Compartments> compartments) throws IOException {
        ROI roi = pathObject.getROI();
        int pad = (int)Math.ceil(downsample * 2.0);
        RegionRequest request = RegionRequest.createInstance((String)server.getPath(), (double)downsample, (ROI)roi).pad2D(pad, pad).intersect2D(0, 0, server.getWidth(), server.getHeight());
        PathImage<ImagePlus> pathImage = IJTools.convertToImagePlus(server, request);
        ImagePlus imp = (ImagePlus)pathImage.getImage();
        LinkedHashMap<String, ImageProcessor> channels = new LinkedHashMap<String, ImageProcessor>();
        List serverChannels = server.getMetadata().getChannels();
        if (server.isRGB() && imp.getStackSize() == 1 && imp.getProcessor() instanceof ColorProcessor) {
            ColorProcessor cp = (ColorProcessor)imp.getProcessor();
            for (int i = 0; i < serverChannels.size(); ++i) {
                channels.put(((ImageChannel)serverChannels.get(i)).getName(), (ImageProcessor)cp.getChannel(i + 1, null));
            }
        } else {
            assert (imp.getStackSize() == serverChannels.size());
            for (int i = 0; i < imp.getStackSize(); ++i) {
                channels.put(((ImageChannel)serverChannels.get(i)).getName(), imp.getStack().getProcessor(i + 1));
            }
        }
        ByteProcessor bpCell = new ByteProcessor(imp.getWidth(), imp.getHeight());
        bpCell.setValue(1.0);
        Roi roiIJ = IJTools.convertToIJRoi(roi, pathImage);
        bpCell.fill(roiIJ);
        if (pathObject instanceof PathCellObject) {
            PathCellObject cell = (PathCellObject)pathObject;
            ByteProcessor bpNucleus = new ByteProcessor(imp.getWidth(), imp.getHeight());
            if (cell.getNucleusROI() != null) {
                bpNucleus.setValue(1.0);
                Roi roiNucleusIJ = IJTools.convertToIJRoi(cell.getNucleusROI(), pathImage);
                bpNucleus.fill(roiNucleusIJ);
            }
            ObjectMeasurements.measureCells((ImageProcessor)bpNucleus, (ImageProcessor)bpCell, Map.of(1.0, cell), channels, compartments, measurements);
        } else {
            PixelImageIJ imgLabels = new PixelImageIJ((ImageProcessor)bpCell);
            for (Map.Entry entry : channels.entrySet()) {
                PixelImageIJ img = new PixelImageIJ((ImageProcessor)entry.getValue());
                ObjectMeasurements.measureObjects((SimpleImage)img, (SimpleImage)imgLabels, new PathObject[]{pathObject}, (String)entry.getKey(), measurements);
            }
        }
    }

    private static void measureCells(ImageProcessor ipNuclei, ImageProcessor ipCells, Map<? extends Number, ? extends PathObject> pathObjects, Map<String, ImageProcessor> channels, Collection<Compartments> compartments, Collection<Measurements> measurements) {
        block15: {
            String channelName;
            PixelImageIJ img;
            PixelImageIJ imgMembrane;
            PixelImageIJ imgCytoplasm;
            PixelImageIJ imgCells;
            PixelImageIJ imgNuclei;
            PathObject[] array;
            block14: {
                array = ObjectMeasurements.mapToArray(pathObjects);
                int width = ipNuclei.getWidth();
                int height = ipNuclei.getHeight();
                FloatProcessor ipMembrane = new FloatProcessor(width, height);
                ImageProcessor ipCytoplasm = ipCells.duplicate();
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        float cell = ipCells.getf(x, y);
                        float nuc = ipNuclei.getf(x, y);
                        if (nuc != 0.0f) {
                            ipCytoplasm.setf(x, y, 0.0f);
                        }
                        if (cell == 0.0f || !(y >= 1 && ipCells.getf(x, y - 1) != cell || y < height - 1 && ipCells.getf(x, y + 1) != cell || x >= 1 && ipCells.getf(x - 1, y) != cell) && (x >= width - 1 || ipCells.getf(x + 1, y) == cell)) continue;
                        ipMembrane.setf(x, y, cell);
                    }
                }
                imgNuclei = new PixelImageIJ(ipNuclei);
                imgCells = new PixelImageIJ(ipCells);
                imgCytoplasm = new PixelImageIJ(ipCytoplasm);
                imgMembrane = new PixelImageIJ((ImageProcessor)ipMembrane);
                boolean useLegacyNames = Boolean.parseBoolean(System.getProperty("OBJECT_MEASUREMENTS_USE_LEGACY_NAMES", "false").strip());
                if (!useLegacyNames) break block14;
                for (Map.Entry<String, ImageProcessor> entry : channels.entrySet()) {
                    PixelImageIJ img2 = new PixelImageIJ(entry.getValue());
                    if (compartments.contains((Object)Compartments.NUCLEUS)) {
                        ObjectMeasurements.measureObjects((SimpleImage)img2, (SimpleImage)imgNuclei, array, entry.getKey().trim() + ": Nucleus", measurements);
                    }
                    if (compartments.contains((Object)Compartments.CYTOPLASM)) {
                        ObjectMeasurements.measureObjects((SimpleImage)img2, (SimpleImage)imgCytoplasm, array, entry.getKey().trim() + ": Cytoplasm", measurements);
                    }
                    if (compartments.contains((Object)Compartments.MEMBRANE)) {
                        ObjectMeasurements.measureObjects((SimpleImage)img2, (SimpleImage)imgMembrane, array, entry.getKey().trim() + ": Membrane", measurements);
                    }
                    if (!compartments.contains((Object)Compartments.CELL)) continue;
                    ObjectMeasurements.measureObjects((SimpleImage)img2, (SimpleImage)imgCells, array, entry.getKey().trim() + ": Cell", measurements);
                }
                break block15;
            }
            if (compartments.contains((Object)Compartments.NUCLEUS)) {
                for (Map.Entry<String, ImageProcessor> entry : channels.entrySet()) {
                    img = new PixelImageIJ(entry.getValue());
                    channelName = entry.getKey().trim();
                    ObjectMeasurements.measureObjects((SimpleImage)img, (SimpleImage)imgNuclei, array, "Nucleus: " + channelName, measurements);
                }
            }
            if (compartments.contains((Object)Compartments.CYTOPLASM)) {
                for (Map.Entry<String, ImageProcessor> entry : channels.entrySet()) {
                    img = new PixelImageIJ(entry.getValue());
                    channelName = entry.getKey().trim();
                    ObjectMeasurements.measureObjects((SimpleImage)img, (SimpleImage)imgCytoplasm, array, "Cytoplasm: " + channelName, measurements);
                }
            }
            if (compartments.contains((Object)Compartments.MEMBRANE)) {
                for (Map.Entry<String, ImageProcessor> entry : channels.entrySet()) {
                    img = new PixelImageIJ(entry.getValue());
                    channelName = entry.getKey().trim();
                    ObjectMeasurements.measureObjects((SimpleImage)img, (SimpleImage)imgMembrane, array, "Membrane: " + channelName, measurements);
                }
            }
            if (!compartments.contains((Object)Compartments.CELL)) break block15;
            for (Map.Entry<String, ImageProcessor> entry : channels.entrySet()) {
                img = new PixelImageIJ(entry.getValue());
                channelName = entry.getKey().trim();
                ObjectMeasurements.measureObjects((SimpleImage)img, (SimpleImage)imgCells, array, "Cell: " + channelName, measurements);
            }
        }
    }

    private static PathObject[] mapToArray(Map<? extends Number, ? extends PathObject> pathObjects) {
        Number[] labels = new Number[pathObjects.size()];
        int n = 0;
        long maxLabel = 0L;
        int invalidLabels = 0;
        for (Number number : pathObjects.keySet()) {
            long lab = number.longValue();
            if (lab < 0L || (double)lab != number.doubleValue() || lab >= Integer.MAX_VALUE) {
                ++invalidLabels;
                continue;
            }
            labels[n] = number;
            maxLabel = Math.max(lab, maxLabel);
            ++n;
        }
        if (invalidLabels > 0) {
            logger.warn("Only {}/{} labels are integer values >= 0 and < Integer.MAX_VALUE, the rest will be discarded!", (Object)n, (Object)pathObjects.size());
        }
        PathObject[] array = new PathObject[n];
        for (Number label : labels) {
            array[label.intValue() - 1] = pathObjects.get(label);
        }
        return array;
    }

    private static void measureObjects(SimpleImage img, SimpleImage imgLabels, PathObject[] pathObjects, String baseName, Collection<Measurements> measurements) {
        int n = pathObjects.length;
        DescriptiveStatistics[] allStats = new DescriptiveStatistics[n];
        for (int i = 0; i < n; ++i) {
            allStats[i] = new DescriptiveStatistics(-1);
        }
        int width = img.getWidth();
        int height = img.getHeight();
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                int label = (int)imgLabels.getValue(x, y);
                if (label <= 0 || label > n) continue;
                float val = img.getValue(x, y);
                allStats[label - 1].addValue((double)val);
            }
        }
        if (!(measurements instanceof Set)) {
            measurements = new LinkedHashSet<Measurements>(measurements);
        }
        for (int i = 0; i < n; ++i) {
            PathObject pathObject = pathObjects[i];
            if (pathObject == null) continue;
            DescriptiveStatistics stats = allStats[i];
            try (MeasurementList ml = pathObject.getMeasurementList();){
                for (Measurements m : measurements) {
                    ml.put(baseName + ": " + m.getMeasurementName(), m.getMeasurement((StatisticalSummary)stats));
                }
                continue;
            }
        }
    }

    public static enum ShapeFeatures {
        AREA,
        LENGTH,
        CIRCULARITY,
        SOLIDITY,
        MAX_DIAMETER,
        MIN_DIAMETER,
        NUCLEUS_CELL_RATIO;


        public String toString() {
            switch (this.ordinal()) {
                case 0: {
                    return "Area";
                }
                case 2: {
                    return "Circularity";
                }
                case 1: {
                    return "Length";
                }
                case 4: {
                    return "Maximum diameter";
                }
                case 5: {
                    return "Minimum diameter";
                }
                case 3: {
                    return "Solidity";
                }
                case 6: {
                    return "Nucleus/Cell area ratio";
                }
            }
            throw new IllegalArgumentException("Unknown feature " + String.valueOf((Object)this));
        }
    }

    public static enum Compartments {
        NUCLEUS,
        CYTOPLASM,
        CELL,
        MEMBRANE;

    }

    public static enum Measurements {
        MEAN,
        MEDIAN,
        MIN,
        MAX,
        STD_DEV,
        VARIANCE;


        private String getMeasurementName() {
            switch (this.ordinal()) {
                case 3: {
                    return "Max";
                }
                case 0: {
                    return "Mean";
                }
                case 1: {
                    return "Median";
                }
                case 2: {
                    return "Min";
                }
                case 4: {
                    return "Std.Dev.";
                }
                case 5: {
                    return "Variance";
                }
            }
            throw new IllegalArgumentException("Unknown measurement " + String.valueOf((Object)this));
        }

        private double getMeasurement(StatisticalSummary stats) {
            switch (this.ordinal()) {
                case 3: {
                    return stats.getMax();
                }
                case 0: {
                    return stats.getMean();
                }
                case 1: {
                    if (stats instanceof DescriptiveStatistics) {
                        return ((DescriptiveStatistics)stats).getPercentile(50.0);
                    }
                    return Double.NaN;
                }
                case 2: {
                    return stats.getMin();
                }
                case 4: {
                    return stats.getStandardDeviation();
                }
                case 5: {
                    return stats.getVariance();
                }
            }
            throw new IllegalArgumentException("Unknown measurement " + String.valueOf((Object)this));
        }
    }
}

