/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.algorithms;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.features.LocalBinaryPatterns;
import qupath.lib.analysis.images.SimpleImage;
import qupath.lib.analysis.images.SimpleImages;
import qupath.lib.analysis.images.SimpleModifiableImage;
import qupath.lib.analysis.stats.RunningStatistics;
import qupath.lib.analysis.stats.StatisticsHelper;
import qupath.lib.color.ColorDeconvolutionStains;
import qupath.lib.color.ColorTransformer;
import qupath.lib.common.GeneralTools;
import qupath.lib.geom.ImmutableDimension;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.objects.PathDetectionObject;
import qupath.lib.objects.PathObject;
import qupath.lib.plugins.AbstractInteractivePlugin;
import qupath.lib.plugins.parameters.Parameter;
import qupath.lib.plugins.parameters.ParameterList;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.interfaces.ROI;

public class LocalBinaryPatternsPlugin
extends AbstractInteractivePlugin<BufferedImage> {
    private ParameterList params = new ParameterList().addDoubleParameter("magnification", "Magnification", 5.0).addChoiceParameter("stainChoice", "Stains", (Object)"Optical density", Arrays.asList("Optical density", "H-DAB", "H&E", "H-DAB (8-bit)", "H&E (8-bit)", "RGB", "Grayscale"));

    public LocalBinaryPatternsPlugin() {
        this.params.addDoubleParameter("tileSizeMicrons", "Tile diameter", 25.0, GeneralTools.micrometerSymbol(), "Size of image tile within which to calculate local binary patterns");
        this.params.addDoubleParameter("tileSizePx", "Tile diameter", 200.0, "px (full resolution image)", "Size of image tile within which to calculate local binary patterns");
        this.params.addBooleanParameter("includeStats", "Include basic statistics", false).addBooleanParameter("doCircular", "Use circular tiles", false);
    }

    static ImmutableDimension getPreferredTileSizePixels(ImageServer<BufferedImage> server, ParameterList params) {
        int tileHeight;
        int tileWidth;
        PixelCalibration cal = server.getPixelCalibration();
        if (cal.hasPixelSizeMicrons()) {
            double tileSize = params.getDoubleParameterValue("tileSizeMicrons");
            tileWidth = (int)(tileSize / cal.getPixelWidthMicrons() + 0.5);
            tileHeight = (int)(tileSize / cal.getPixelHeightMicrons() + 0.5);
        } else {
            tileHeight = tileWidth = (int)(params.getDoubleParameterValue("tileSizePx") + 0.5);
        }
        return ImmutableDimension.getInstance((int)tileWidth, (int)tileHeight);
    }

    static String getDiameterString(ImageServer<BufferedImage> server, ParameterList params) {
        if (server.getPixelCalibration().hasPixelSizeMicrons()) {
            return String.format("%.1f %s", params.getDoubleParameterValue("tileSizeMicrons"), GeneralTools.micrometerSymbol());
        }
        return String.format("%d px", (int)(params.getDoubleParameterValue("tileSizePx") + 0.5));
    }

    protected void addRunnableTasks(ImageData<BufferedImage> imageData, PathObject parentObject, List<Runnable> tasks) {
        ParameterList params = this.getParameterList(imageData);
        ImageServer server = imageData.getServer();
        tasks.add(new LBFRunnable((ImageServer<BufferedImage>)server, parentObject, params, imageData.getColorDeconvolutionStains()));
    }

    static boolean processObject(PathObject pathObject, ParameterList params, ImageServer<BufferedImage> server, ColorDeconvolutionStains stains) throws InterruptedException, IOException {
        String stainsName = (String)params.getChoiceParameterValue("stainChoice");
        double mag = params.getDoubleParameterValue("magnification");
        boolean includeStats = params.getBooleanParameterValue("includeStats");
        boolean doCircular = params.getBooleanParameterValue("doCircular");
        double downsample = server.getMetadata().getMagnification() / mag;
        ROI pathROI = pathObject.getROI();
        if (pathROI == null) {
            return false;
        }
        ImmutableDimension size = LocalBinaryPatternsPlugin.getPreferredTileSizePixels(server, params);
        if ((double)size.getWidth() / downsample < 1.0 || (double)size.getHeight() / downsample < 1.0) {
            return false;
        }
        RegionRequest region = RegionRequest.createInstance((String)server.getPath(), (double)downsample, (int)((int)(pathROI.getCentroidX() + 0.5) - size.width / 2), (int)((int)(pathROI.getCentroidY() + 0.5) - size.height / 2), (int)size.width, (int)size.height, (int)pathROI.getT(), (int)pathROI.getZ());
        BufferedImage img = (BufferedImage)server.readRegion(region);
        int w = img.getWidth();
        int h = img.getHeight();
        int[] buf = img.getRGB(0, 0, w, h, null, 0, w);
        float[] pixels = new float[buf.length];
        SimpleModifiableImage pxImg = SimpleImages.createFloatImage((float[])pixels, (int)w, (int)h);
        MeasurementList measurementList = pathObject.getMeasurementList();
        String postfix = " (" + LocalBinaryPatternsPlugin.getDiameterString(server, params) + ")";
        if (stainsName.equals("H-DAB")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Hematoxylin" + postfix, ColorTransformer.ColorTransformMethod.Hematoxylin_H_DAB, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "DAB" + postfix, ColorTransformer.ColorTransformMethod.DAB_H_DAB, stains, includeStats, doCircular);
        } else if (stainsName.equals("H&E")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Hematoxylin" + postfix, ColorTransformer.ColorTransformMethod.Hematoxylin_H_E, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Eosin" + postfix, ColorTransformer.ColorTransformMethod.Eosin_H_E, stains, includeStats, doCircular);
        } else if (stainsName.equals("H-DAB (8-bit)")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Hematoxylin 8-bit" + postfix, ColorTransformer.ColorTransformMethod.Hematoxylin_H_DAB_8_bit, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "DAB 8-bit" + postfix, ColorTransformer.ColorTransformMethod.DAB_H_DAB_8_bit, stains, includeStats, doCircular);
        } else if (stainsName.equals("H&E (8-bit)")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Hematoxylin 8-bit" + postfix, ColorTransformer.ColorTransformMethod.Hematoxylin_H_E_8_bit, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Eosin 8-bit" + postfix, ColorTransformer.ColorTransformMethod.Eosin_H_E_8_bit, stains, includeStats, doCircular);
        } else if (stainsName.equals("Optical density")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "OD sum" + postfix, ColorTransformer.ColorTransformMethod.Optical_density_sum, stains, includeStats, doCircular);
        } else if (stainsName.equals("RGB")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Red" + postfix, ColorTransformer.ColorTransformMethod.Red, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Green" + postfix, ColorTransformer.ColorTransformMethod.Green, stains, includeStats, doCircular);
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Blue" + postfix, ColorTransformer.ColorTransformMethod.Blue, stains, includeStats, doCircular);
        } else if (stainsName.equals("Grayscale")) {
            LocalBinaryPatternsPlugin.processTransformedImage(pxImg, buf, pixels, measurementList, "Grayscale" + postfix, ColorTransformer.ColorTransformMethod.RGB_mean, stains, includeStats, doCircular);
        }
        measurementList.close();
        return true;
    }

    static void processTransformedImage(SimpleModifiableImage pxImg, int[] buf, float[] pixels, MeasurementList measurementList, String name, ColorTransformer.ColorTransformMethod method, ColorDeconvolutionStains stains, boolean includeStats, boolean doCircular) {
        ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)method, (float[])pixels, (ColorDeconvolutionStains)stains);
        if (doCircular) {
            double w = pxImg.getWidth();
            double h = pxImg.getHeight();
            double cx = (w - 1.0) / 2.0;
            double cy = (h - 1.0) / 2.0;
            double radius = Math.max(w, h) * 0.5;
            double distThreshold = radius * radius;
            int y = 0;
            while ((double)y < h) {
                int x = 0;
                while ((double)x < w) {
                    if ((cx - (double)x) * (cx - (double)x) + (cy - (double)y) * (cy - (double)y) > distThreshold) {
                        pxImg.setValue(x, y, Float.NaN);
                    }
                    ++x;
                }
                ++y;
            }
        }
        if (includeStats) {
            LocalBinaryPatternsPlugin.addBasicStatistics((SimpleImage)pxImg, measurementList, name);
        }
        LocalBinaryPatternsPlugin.addLocalBinaryFeatures(LocalBinaryPatterns.computeLocalBinaryPatterns16((SimpleImage)pxImg, 2.0), measurementList, name);
    }

    static void addBasicStatistics(SimpleImage img, MeasurementList measurementList, String name) {
        RunningStatistics stats = StatisticsHelper.computeRunningStatistics((SimpleImage)img);
        measurementList.put(name + " Mean", stats.getMean());
        measurementList.put(name + " Min", stats.getMin());
        measurementList.put(name + " Max", stats.getMax());
        measurementList.put(name + " Range", stats.getRange());
        measurementList.put(name + " Std.dev.", stats.getStdDev());
    }

    static void addLocalBinaryFeatures(double[] histogram, MeasurementList measurementList, String name) {
        for (int i = 0; i < histogram.length; ++i) {
            measurementList.put(String.format("%s LBP %d", name, i + 1), histogram[i]);
        }
    }

    public ParameterList getDefaultParameterList(ImageData<BufferedImage> imageData) {
        boolean hasMicrons = imageData.getServer().getPixelCalibration().hasPixelSizeMicrons();
        ((Parameter)this.params.getParameters().get("tileSizeMicrons")).setHidden(!hasMicrons);
        ((Parameter)this.params.getParameters().get("tileSizePx")).setHidden(hasMicrons);
        return this.params;
    }

    public String getName() {
        return "Add Local Binary Pattern features";
    }

    public String getLastResultsDescription() {
        return "";
    }

    public String getDescription() {
        return "Add Local Binary Pattern features to existing object measurements";
    }

    protected Collection<PathObject> getParentObjects(ImageData<BufferedImage> imageData) {
        return imageData.getHierarchy().getDetectionObjects();
    }

    public Collection<Class<? extends PathObject>> getSupportedParentObjectClasses() {
        ArrayList<Class<? extends PathObject>> parents = new ArrayList<Class<? extends PathObject>>();
        parents.add(PathDetectionObject.class);
        return parents;
    }

    static class LBFRunnable
    implements Runnable {
        private static Logger logger = LoggerFactory.getLogger(LBFRunnable.class);
        private ImageServer<BufferedImage> server;
        private ParameterList params;
        private PathObject parentObject;
        private ColorDeconvolutionStains stains;

        public LBFRunnable(ImageServer<BufferedImage> server, PathObject parentObject, ParameterList params, ColorDeconvolutionStains stains) {
            this.server = server;
            this.parentObject = parentObject;
            this.params = params;
            this.stains = stains;
        }

        @Override
        public void run() {
            try {
                LocalBinaryPatternsPlugin.processObject(this.parentObject, this.params, this.server, this.stains);
            }
            catch (InterruptedException e) {
                logger.warn("Processing interrupted!", (Throwable)e);
            }
            catch (IOException e) {
                logger.warn("Error processing " + String.valueOf(this.parentObject), (Throwable)e);
            }
            finally {
                this.parentObject.getMeasurementList().close();
                this.server = null;
                this.params = null;
            }
        }

        public String toString() {
            return "Local Binary Pattern features";
        }
    }
}

