/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.algorithms;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.IOException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.analysis.features.CoocurranceMatrices;
import qupath.lib.analysis.features.HaralickFeatureComputer;
import qupath.lib.analysis.features.HaralickFeatures;
import qupath.lib.analysis.images.SimpleImage;
import qupath.lib.analysis.images.SimpleImages;
import qupath.lib.analysis.images.SimpleModifiableImage;
import qupath.lib.analysis.stats.RunningStatistics;
import qupath.lib.analysis.stats.StatisticsHelper;
import qupath.lib.awt.common.BufferedImageTools;
import qupath.lib.color.ColorDeconvolutionStains;
import qupath.lib.color.ColorTransformer;
import qupath.lib.common.GeneralTools;
import qupath.lib.geom.ImmutableDimension;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.objects.PathAnnotationObject;
import qupath.lib.objects.PathCellObject;
import qupath.lib.objects.PathDetectionObject;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.TMACoreObject;
import qupath.lib.plugins.AbstractInteractivePlugin;
import qupath.lib.plugins.TaskRunner;
import qupath.lib.plugins.parameters.Parameter;
import qupath.lib.plugins.parameters.ParameterList;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.RoiTools;
import qupath.lib.roi.interfaces.ROI;

public class IntensityFeaturesPlugin
extends AbstractInteractivePlugin<BufferedImage> {
    private static final Logger logger = LoggerFactory.getLogger(IntensityFeaturesPlugin.class);
    private boolean parametersInitialized = false;
    private static Map<Integer, BasicChannel> channelMap = new HashMap<Integer, BasicChannel>();
    private static List<FeatureComputerBuilder> builders = Arrays.asList(new BasicFeatureComputerBuilder(), new MedianFeatureComputerBuilder(), new HaralickFeatureComputerBuilder());

    static synchronized List<FeatureColorTransform> getBasicChannelTransforms(int nChannels) {
        ArrayList<FeatureColorTransform> list = new ArrayList<FeatureColorTransform>();
        for (int i = 0; i < nChannels; ++i) {
            BasicChannel channel = channelMap.get(i);
            if (channel == null) {
                channel = new BasicChannel(i);
                channelMap.put(i, channel);
            }
            list.add(channel);
        }
        return list;
    }

    public boolean runPlugin(TaskRunner taskRunner, ImageData<BufferedImage> imageData, String arg) {
        boolean success = super.runPlugin(taskRunner, imageData, arg);
        imageData.getHierarchy().fireHierarchyChangedEvent((Object)this);
        return success;
    }

    private static ImmutableDimension getPreferredTileSizePixels(ImageServer<BufferedImage> server, ParameterList params) {
        int tileHeight;
        int tileWidth;
        PixelCalibration cal = server.getPixelCalibration();
        if (cal.hasPixelSizeMicrons()) {
            double tileSize = params.getDoubleParameterValue("tileSizeMicrons");
            tileWidth = (int)Math.round(tileSize / cal.getPixelWidthMicrons());
            tileHeight = (int)Math.round(tileSize / cal.getPixelHeightMicrons());
        } else {
            tileHeight = tileWidth = (int)Math.round(params.getDoubleParameterValue("tileSizePixels"));
        }
        return ImmutableDimension.getInstance((int)tileWidth, (int)tileHeight);
    }

    static String getDiameterString(ImageServer<BufferedImage> server, ParameterList params) {
        RegionType regionType = (RegionType)((Object)params.getChoiceParameterValue("region"));
        PixelCalibration cal = server.getPixelCalibration();
        String shape = regionType == RegionType.SQUARE ? "Square" : (regionType == RegionType.CIRCLE ? "Circle" : "ROI");
        String unit = cal.hasPixelSizeMicrons() ? GeneralTools.micrometerSymbol() : "px";
        double pixelSize = cal.hasPixelSizeMicrons() ? params.getDoubleParameterValue("pixelSizeMicrons") : params.getDoubleParameterValue("downsample");
        double regionSize = cal.hasPixelSizeMicrons() ? params.getDoubleParameterValue("tileSizeMicrons") : params.getDoubleParameterValue("tileSizePixels");
        if (regionType == RegionType.ROI) {
            return String.format("ROI: %.2f %s per pixel", pixelSize, unit);
        }
        if (regionType == RegionType.NUCLEUS) {
            return String.format("Nucleus: %.2f %s per pixel", pixelSize, unit);
        }
        return String.format("%s: Diameter %.1f %s: %.2f %s per pixel", shape, regionSize, unit, pixelSize, unit);
    }

    protected void addRunnableTasks(ImageData<BufferedImage> imageData, PathObject parentObject, List<Runnable> tasks) {
        ParameterList params = this.getParameterList(imageData);
        ImageServer server = imageData.getServer();
        PixelCalibration cal = server.getPixelCalibration();
        double downsample = IntensityFeaturesPlugin.calculateDownsample(cal, params);
        if (downsample <= 0.0) {
            throw new IllegalArgumentException("Effective downsample must be > 0 (requested value " + GeneralTools.formatNumber((double)downsample, (int)1) + ")");
        }
        tasks.add(new IntensityFeatureRunnable(imageData, parentObject, params));
    }

    static double calculateDownsample(PixelCalibration cal, ParameterList params) {
        if (cal.hasPixelSizeMicrons()) {
            return params.getDoubleParameterValue("pixelSizeMicrons") / cal.getAveragedPixelSizeMicrons();
        }
        return params.getDoubleParameterValue("downsample");
    }

    static boolean processObject(PathObject pathObject, ParameterList params, ImageData<BufferedImage> imageData) throws IOException {
        RegionType regionType;
        ImageServer server = imageData.getServer();
        ColorDeconvolutionStains stains = imageData.getColorDeconvolutionStains();
        PixelCalibration cal = server.getPixelCalibration();
        double downsample = IntensityFeaturesPlugin.calculateDownsample(cal, params);
        if (downsample <= 0.0) {
            logger.warn("Effective downsample must be > 0 (requested value {})", (Object)downsample);
        }
        boolean useROI = (regionType = (RegionType)((Object)params.getChoiceParameterValue("region"))) == RegionType.ROI || regionType == RegionType.NUCLEUS;
        ROI roi = null;
        if (regionType == RegionType.NUCLEUS) {
            if (pathObject instanceof PathCellObject) {
                roi = ((PathCellObject)pathObject).getNucleusROI();
            }
        } else {
            roi = pathObject.getROI();
        }
        if (roi == null) {
            return false;
        }
        LinkedHashMap map = new LinkedHashMap();
        if (server.isRGB()) {
            for (FeatureColorTransformEnum transform : FeatureColorTransformEnum.values()) {
                ArrayList<FeatureComputer> arrayList = new ArrayList<FeatureComputer>();
                map.put(transform, arrayList);
                for (FeatureComputerBuilder builder : builders) {
                    arrayList.add(builder.build());
                }
            }
        } else {
            for (FeatureColorTransform transform : IntensityFeaturesPlugin.getBasicChannelTransforms(server.nChannels())) {
                ArrayList<FeatureComputer> list = new ArrayList<FeatureComputer>();
                map.put(transform, list);
                for (FeatureComputerBuilder featureComputerBuilder : builders) {
                    list.add(featureComputerBuilder.build());
                }
            }
        }
        String prefix = IntensityFeaturesPlugin.getDiameterString((ImageServer<BufferedImage>)server, params);
        ImmutableDimension sizePreferred = ImmutableDimension.getInstance((int)((int)(2000.0 * downsample)), (int)((int)(2000.0 * downsample)));
        Collection rois = RoiTools.computeTiledROIs((ROI)roi, (ImmutableDimension)sizePreferred, (ImmutableDimension)sizePreferred, (boolean)false, (int)0);
        if (rois.size() > 1) {
            logger.info("Splitting {} into {} tiles for intensity measurements", (Object)roi, (Object)rois.size());
        }
        for (ROI rOI : rois) {
            boolean isRGB;
            RegionRequest region;
            if (Thread.currentThread().isInterrupted()) {
                logger.warn("Measurement skipped - thread interrupted!");
                return false;
            }
            if (useROI) {
                region = RegionRequest.createInstance((String)server.getPath(), (double)downsample, (ROI)rOI);
            } else {
                ImmutableDimension size = IntensityFeaturesPlugin.getPreferredTileSizePixels((ImageServer<BufferedImage>)server, params);
                int xStart = (int)((double)Math.round(rOI.getCentroidX() / downsample) * downsample) - size.width / 2;
                int yStart = (int)((double)Math.round(rOI.getCentroidY() / downsample) * downsample) - size.height / 2;
                int width = Math.min(server.getWidth(), xStart + size.width) - xStart;
                int height = Math.min(server.getHeight(), yStart + size.height) - yStart;
                region = RegionRequest.createInstance((String)server.getPath(), (double)downsample, (int)xStart, (int)yStart, (int)width, (int)height, (int)rOI.getT(), (int)rOI.getZ());
            }
            BufferedImage img = (BufferedImage)server.readRegion(region);
            if (img == null) {
                logger.error("Could not read image - unable to compute intensity features for {}", (Object)pathObject);
                return false;
            }
            byte[] maskBytes = null;
            if (useROI && img.getWidth() * img.getHeight() > 1) {
                BufferedImage imgMask = BufferedImageTools.createROIMask((int)img.getWidth(), (int)img.getHeight(), (ROI)rOI, (RegionRequest)region);
                maskBytes = ((DataBufferByte)imgMask.getRaster().getDataBuffer()).getData();
            }
            List<FeatureColorTransform> transforms = (isRGB = server.isRGB()) ? Arrays.asList(FeatureColorTransformEnum.values()) : IntensityFeaturesPlugin.getBasicChannelTransforms(server.nChannels());
            int w = img.getWidth();
            int h = img.getHeight();
            int[] rgbBuffer = isRGB ? img.getRGB(0, 0, w, h, null, 0, w) : null;
            float[] pixels = null;
            for (FeatureColorTransform transform : transforms) {
                if (!params.containsKey((Object)transform.getKey()) || !Boolean.TRUE.equals(params.getBooleanParameterValue(transform.getKey()))) continue;
                pixels = transform.getTransformedPixels(img, rgbBuffer, stains, pixels);
                SimpleModifiableImage pixelImage = SimpleImages.createFloatImage((float[])pixels, (int)w, (int)h);
                if (maskBytes != null) {
                    for (int i = 0; i < pixels.length; ++i) {
                        if (maskBytes[i] != 0) continue;
                        pixelImage.setValue(i % w, i / w, Float.NaN);
                    }
                } else if (regionType == RegionType.CIRCLE) {
                    double cx = (w - 1) / 2;
                    double cy = (h - 1) / 2;
                    double radius = (double)Math.max(w, h) * 0.5;
                    double distThreshold = radius * radius;
                    for (int y = 0; y < h; ++y) {
                        for (int x = 0; x < w; ++x) {
                            if (!((cx - (double)x) * (cx - (double)x) + (cy - (double)y) * (cy - (double)y) > distThreshold)) continue;
                            pixelImage.setValue(x, y, Float.NaN);
                        }
                    }
                }
                for (FeatureComputer computer : (List)map.get(transform)) {
                    computer.updateFeatures((SimpleImage)pixelImage, transform, params);
                }
            }
        }
        for (Map.Entry entry : map.entrySet()) {
            String name = prefix + ": " + ((FeatureColorTransform)entry.getKey()).getName(imageData, false) + ":";
            for (FeatureComputer computer : (List)entry.getValue()) {
                computer.addMeasurements(pathObject, name, params);
            }
        }
        pathObject.getMeasurementList().close();
        if (pathObject instanceof PathAnnotationObject) {
            ((PathAnnotationObject)pathObject).setLocked(true);
        } else if (pathObject instanceof TMACoreObject) {
            ((TMACoreObject)pathObject).setLocked(true);
        }
        return true;
    }

    public ParameterList getDefaultParameterList(ImageData<BufferedImage> imageData) {
        if (!this.parametersInitialized) {
            this.params = new ParameterList();
            this.params.addTitleParameter("Resolution");
            this.params.addDoubleParameter("downsample", "Downsample", 1.0, null, "Amount to downsample the image before calculating textures; choose 1 to use full resolution, or a higher value to use a smaller image").addDoubleParameter("pixelSizeMicrons", "Preferred pixel size", 2.0, GeneralTools.micrometerSymbol(), "Preferred pixel size of the image used to calculate the textures - higher values means coarser (lower resolution) images");
            this.params.addTitleParameter("Regions");
            this.params.addChoiceParameter("region", "Region", (Object)RegionType.ROI, Arrays.asList(RegionType.values()), "The region within which to calculate the features");
            this.params.addDoubleParameter("tileSizeMicrons", "Tile diameter", 25.0, GeneralTools.micrometerSymbol(), "Diameter of tile around the object centroid used to calculate textures.\nOnly matters if tiles are being used (i.e. the region parameter isn't ROI).");
            this.params.addDoubleParameter("tileSizePixels", "Tile diameter", 200.0, "px (full resolution image)", "Diameter of tile around the object centroid used to calculate textures.\nOnly matters if tiles are being used (i.e. the region parameter isn't ROI).");
            boolean hasMicrons = imageData.getServer().getPixelCalibration().hasPixelSizeMicrons();
            ((Parameter)this.params.getParameters().get("pixelSizeMicrons")).setHidden(!hasMicrons);
            ((Parameter)this.params.getParameters().get("downsample")).setHidden(hasMicrons);
            ((Parameter)this.params.getParameters().get("tileSizeMicrons")).setHidden(!hasMicrons);
            ((Parameter)this.params.getParameters().get("tileSizePixels")).setHidden(hasMicrons);
            this.params.addTitleParameter("Channels/Color transforms");
            if (imageData.getServer().isRGB()) {
                for (FeatureColorTransformEnum transform : FeatureColorTransformEnum.values()) {
                    if (!transform.supportsImage(imageData)) continue;
                    this.params.addBooleanParameter(transform.getKey(), transform.getPrompt(imageData), false);
                }
            } else {
                for (FeatureColorTransform transform : IntensityFeaturesPlugin.getBasicChannelTransforms(imageData.getServer().nChannels())) {
                    this.params.addBooleanParameter(transform.getKey(), transform.getPrompt(imageData), false);
                }
            }
            for (FeatureComputerBuilder builder : builders) {
                builder.addParameters(imageData, this.params);
            }
        }
        this.parametersInitialized = true;
        return this.params;
    }

    public String getName() {
        return "Compute intensity features";
    }

    public String getLastResultsDescription() {
        return "";
    }

    public String getDescription() {
        return "Add intensity features to existing object measurements";
    }

    protected Collection<PathObject> getParentObjects(ImageData<BufferedImage> imageData) {
        return imageData.getHierarchy().getSelectionModel().getSelectedObjects();
    }

    public Collection<Class<? extends PathObject>> getSupportedParentObjectClasses() {
        ArrayList<Class<? extends PathObject>> parents = new ArrayList<Class<? extends PathObject>>();
        parents.add(PathCellObject.class);
        parents.add(PathDetectionObject.class);
        parents.add(PathAnnotationObject.class);
        parents.add(TMACoreObject.class);
        return parents;
    }

    public boolean alwaysPromptForObjects() {
        return true;
    }

    static class BasicChannel
    implements FeatureColorTransform {
        private int channel;

        BasicChannel(int channel) {
            this.channel = channel;
        }

        @Override
        public String getPrompt(ImageData<?> imageData) {
            return this.getImageChannelName(imageData);
        }

        @Override
        public float[] getTransformedPixels(BufferedImage img, int[] buf, ColorDeconvolutionStains stains, float[] pixels) {
            return img.getRaster().getSamples(0, 0, img.getWidth(), img.getHeight(), this.channel, pixels);
        }

        @Override
        public boolean supportsImage(ImageData<?> imageData) {
            return !imageData.getServer().isRGB() && this.channel >= 0 && this.channel < imageData.getServer().nChannels();
        }

        @Override
        public String getKey() {
            return "channel" + (this.channel + 1);
        }

        @Override
        public double[] getHaralickMinMax() {
            return null;
        }

        private String getImageChannelName(ImageData<?> imageData) {
            if (imageData == null) {
                return this.getSimpleChannelName();
            }
            return imageData.getServer().getChannel(this.channel).getName();
        }

        private String getSimpleChannelName() {
            return "Channel " + (this.channel + 1);
        }

        @Override
        public String getName(ImageData<?> imageData, boolean useSimpleNames) {
            String simpleName = this.getSimpleChannelName();
            if (useSimpleNames) {
                return simpleName;
            }
            String name = this.getImageChannelName(imageData);
            if (name == null || name.isBlank()) {
                return simpleName;
            }
            return name;
        }
    }

    static enum RegionType {
        ROI,
        SQUARE,
        CIRCLE,
        NUCLEUS;


        public String toString() {
            switch (this.ordinal()) {
                case 2: {
                    return "Circular tiles";
                }
                case 0: {
                    return "ROI";
                }
                case 1: {
                    return "Square tiles";
                }
                case 3: {
                    return "Cell nucleus";
                }
            }
            return "Unknown";
        }
    }

    static class IntensityFeatureRunnable
    implements Runnable {
        private ImageData<BufferedImage> imageData;
        private ParameterList params;
        private PathObject parentObject;

        public IntensityFeatureRunnable(ImageData<BufferedImage> imageData, PathObject parentObject, ParameterList params) {
            this.imageData = imageData;
            this.parentObject = parentObject;
            this.params = params;
        }

        @Override
        public void run() {
            try {
                IntensityFeaturesPlugin.processObject(this.parentObject, this.params, this.imageData);
            }
            catch (IOException e) {
                logger.error("Unable to process " + String.valueOf(this.parentObject), (Throwable)e);
            }
            finally {
                this.parentObject.getMeasurementList().close();
                this.imageData = null;
                this.params = null;
            }
        }

        public String toString() {
            return "Intensity measurements";
        }
    }

    static enum FeatureColorTransformEnum implements FeatureColorTransform
    {
        OD("colorOD", "Optical density sum"),
        STAIN_1("colorStain1", "Color Deconvolution Stain 1"),
        STAIN_2("colorStain2", "Color Deconvolution Stain 2"),
        STAIN_3("colorStain3", "Color Deconvolution Stain 3"),
        RED("colorRed", "Red"),
        GREEN("colorGreen", "Green"),
        BLUE("colorBlue", "Blue"),
        HUE("colorHue", "Hue (mean only)"),
        SATURATION("colorSaturation", "Saturation"),
        BRIGHTNESS("colorBrightness", "Brightness");

        private String key;
        private String prompt;

        private FeatureColorTransformEnum(String key, String prompt) {
            this.key = key;
            this.prompt = prompt;
        }

        @Override
        public String getPrompt(ImageData<?> imageData) {
            ColorDeconvolutionStains stains;
            ColorDeconvolutionStains colorDeconvolutionStains = stains = imageData == null ? null : imageData.getColorDeconvolutionStains();
            if (stains != null) {
                switch (this.ordinal()) {
                    case 1: {
                        return stains.getStain(1).getName() + " (color deconvolved)";
                    }
                    case 2: {
                        return stains.getStain(2).getName() + " (color deconvolved)";
                    }
                    case 3: {
                        return stains.getStain(3).getName() + " (color deconvolved)";
                    }
                }
            }
            return this.prompt;
        }

        @Override
        public String getKey() {
            return this.key;
        }

        @Override
        public double[] getHaralickMinMax() {
            switch (this.ordinal()) {
                case 7: {
                    return null;
                }
                case 4: 
                case 5: 
                case 6: {
                    return new double[]{0.0, 255.0};
                }
                case 9: {
                    return new double[]{0.0, 1.0};
                }
                case 8: {
                    return new double[]{0.0, 1.0};
                }
                case 0: {
                    return new double[]{0.0, 2.5};
                }
                case 1: 
                case 2: 
                case 3: {
                    return new double[]{0.0, 1.5};
                }
            }
            return null;
        }

        @Override
        public String getName(ImageData<?> imageData, boolean useSimpleNames) {
            ColorDeconvolutionStains stains = imageData == null ? null : imageData.getColorDeconvolutionStains();
            switch (this.ordinal()) {
                case 1: {
                    return stains == null ? "Stain 1" : stains.getStain(1).getName();
                }
                case 2: {
                    return stains == null ? "Stain 2" : stains.getStain(2).getName();
                }
                case 3: {
                    return stains == null ? "Stain 3" : stains.getStain(3).getName();
                }
                case 7: {
                    return "Hue";
                }
                case 0: {
                    return "OD Sum";
                }
            }
            return this.getPrompt(null);
        }

        @Override
        public boolean supportsImage(ImageData<?> imageData) {
            switch (this.ordinal()) {
                case 4: 
                case 5: 
                case 6: 
                case 7: 
                case 8: 
                case 9: {
                    return imageData.getServer().isRGB();
                }
                case 0: 
                case 1: 
                case 2: 
                case 3: {
                    return imageData.isBrightfield() && imageData.getServer().isRGB();
                }
            }
            return false;
        }

        @Override
        public float[] getTransformedPixels(BufferedImage img, int[] buf, ColorDeconvolutionStains stains, float[] pixels) {
            if (pixels == null) {
                pixels = new float[img.getWidth() * img.getHeight()];
            }
            switch (this.ordinal()) {
                case 9: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Brightness, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 7: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Hue, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 0: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Optical_density_sum, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 4: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Red, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 5: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Green, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 6: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Blue, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 8: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Saturation, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 1: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Stain_1, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 2: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Stain_2, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
                case 3: {
                    return ColorTransformer.getTransformedPixels((int[])buf, (ColorTransformer.ColorTransformMethod)ColorTransformer.ColorTransformMethod.Stain_3, (float[])pixels, (ColorDeconvolutionStains)stains);
                }
            }
            return null;
        }
    }

    static interface FeatureComputerBuilder {
        public void addParameters(ImageData<?> var1, ParameterList var2);

        public FeatureComputer build();
    }

    static interface FeatureComputer {
        public void updateFeatures(SimpleImage var1, FeatureColorTransform var2, ParameterList var3);

        public void addMeasurements(PathObject var1, String var2, ParameterList var3);
    }

    static interface FeatureColorTransform {
        public String getPrompt(ImageData<?> var1);

        public float[] getTransformedPixels(BufferedImage var1, int[] var2, ColorDeconvolutionStains var3, float[] var4);

        public boolean supportsImage(ImageData<?> var1);

        public String getKey();

        public double[] getHaralickMinMax();

        public String getName(ImageData<?> var1, boolean var2);
    }

    static class BasicFeatureComputerBuilder
    implements FeatureComputerBuilder {
        BasicFeatureComputerBuilder() {
        }

        @Override
        public void addParameters(ImageData<?> imageData, ParameterList params) {
            params.addTitleParameter("Basic features");
            for (BasicFeatureComputer.Feature feature : Arrays.asList(BasicFeatureComputer.Feature.MEAN, BasicFeatureComputer.Feature.STD_DEV, BasicFeatureComputer.Feature.MIN_MAX)) {
                params.addBooleanParameter(feature.key, feature.prompt, false, feature.help);
            }
        }

        @Override
        public FeatureComputer build() {
            return new BasicFeatureComputer();
        }
    }

    static class MedianFeatureComputerBuilder
    implements FeatureComputerBuilder {
        private int originalBitsPerPixel;

        MedianFeatureComputerBuilder() {
        }

        @Override
        public void addParameters(ImageData<?> imageData, ParameterList params) {
            this.originalBitsPerPixel = imageData.getServer().getPixelType().getBitsPerPixel();
            if (this.originalBitsPerPixel > 16) {
                return;
            }
            params.addBooleanParameter("doMedian", "Median", false, "Calculate approximate median of pixel values (based on a generated histogram)");
        }

        @Override
        public FeatureComputer build() {
            return new MedianFeatureComputer(this.originalBitsPerPixel);
        }
    }

    static class HaralickFeatureComputerBuilder
    implements FeatureComputerBuilder {
        HaralickFeatureComputerBuilder() {
        }

        @Override
        public void addParameters(ImageData<?> imageData, ParameterList params) {
            params.addTitleParameter("Haralick features");
            params.addBooleanParameter("doHaralick", "Compute Haralick features", false, "Calculate Haralick texture features (13 features in total, non-RGB images require min/max values to be set based on the range of the data)");
            if (!imageData.getServer().isRGB()) {
                params.addDoubleParameter("haralickMin", "Haralick min", Double.NaN, null, "Minimum value used when calculating grayscale cooccurrence matrix for Haralick features -\nThis should be approximately the lowest pixel value in the image for which textures are meaningful.").addDoubleParameter("haralickMax", "Haralick max", Double.NaN, null, "Maximum value used when calculating grayscale cooccurrence matrix for Haralick features -\nThis should be approximately the highest pixel value in the image for which textures are meaningful.");
            }
            params.addIntParameter("haralickDistance", "Haralick distance", 1, null, "Spacing between pixels used in computing the co-occurrence matrix for Haralick textures (default = 1)").addIntParameter("haralickBins", "Haralick number of bins", 32, null, 8.0, 256.0, "Number of intensity bins to use when computing the co-occurrence matrix for Haralick textures (default = 32)");
        }

        @Override
        public FeatureComputer build() {
            return new HaralickFeaturesComp();
        }
    }

    static class CumulativeHistogramFeatureComputer
    implements FeatureComputer {
        private double minBin;
        private double maxBin;
        private long n;
        private int nBins;
        private long[] histogram;

        CumulativeHistogramFeatureComputer() {
        }

        @Override
        public void updateFeatures(SimpleImage img, FeatureColorTransform transform, ParameterList params) {
            if (!Boolean.TRUE.equals(params.getBooleanParameterValue("doCumulativeHistogram")) || transform == null || transform == FeatureColorTransformEnum.HUE || this.histogram != null && this.histogram.length == 0) {
                return;
            }
            if (this.histogram == null) {
                this.minBin = params.getDoubleParameterValue("chMinValue");
                this.maxBin = params.getDoubleParameterValue("chMaxValue");
                this.nBins = params.getIntParameterValue("chBins");
                this.histogram = new long[this.nBins];
            }
            if (this.nBins == 0) {
                return;
            }
            double binWidth = (this.maxBin - this.minBin) / (double)(this.nBins - 1);
            for (int y = 0; y < img.getHeight(); ++y) {
                for (int x = 0; x < img.getWidth(); ++x) {
                    double val = img.getValue(x, y);
                    if (!Double.isFinite(val)) continue;
                    int bin = (int)((val - this.minBin) / binWidth);
                    if (bin >= this.nBins) {
                        int n = this.nBins - 1;
                        this.histogram[n] = this.histogram[n] + 1L;
                    } else if (bin < 0) {
                        this.histogram[0] = this.histogram[0] + 1L;
                    } else {
                        int n = bin;
                        this.histogram[n] = this.histogram[n] + 1L;
                    }
                    ++this.n;
                }
            }
        }

        @Override
        public void addMeasurements(PathObject pathObject, String name, ParameterList params) {
            if (this.histogram == null || this.histogram.length == 0) {
                return;
            }
            double total = 0.0;
            double[] proportions = new double[this.histogram.length];
            for (int i = this.histogram.length - 1; i >= 0; --i) {
                proportions[i] = total += (double)this.histogram[i] / (double)this.n;
            }
            MeasurementList measurementList = pathObject.getMeasurementList();
            double binWidth = (this.maxBin - this.minBin) / (double)(this.nBins - 1);
            NumberFormat formatter = GeneralTools.createFormatter((int)3);
            for (int i = 0; i < this.histogram.length; ++i) {
                double value = this.minBin + (double)i * binWidth;
                measurementList.put(name + " >= " + formatter.format(value), proportions[i]);
            }
        }
    }

    static class CumulativeHistogramFeatureComputerBuilder
    implements FeatureComputerBuilder {
        private int originalBitsPerPixel;

        CumulativeHistogramFeatureComputerBuilder() {
        }

        @Override
        public void addParameters(ImageData<?> imageData, ParameterList params) {
            this.originalBitsPerPixel = imageData.getServer().getPixelType().getBitsPerPixel();
            if (this.originalBitsPerPixel > 16) {
                return;
            }
            params.addTitleParameter("Cumulative histogram");
            params.addBooleanParameter("doCumulativeHistogram", "Cumulative histogram", false);
            params.addDoubleParameter("chMinValue", "Min histogram value", 0.0);
            params.addDoubleParameter("chMaxValue", "Max histogram value", 1.0);
            params.addIntParameter("chBins", "Number of bins", 5);
        }

        @Override
        public FeatureComputer build() {
            return new CumulativeHistogramFeatureComputer();
        }
    }

    static class HueStats {
        private double sinX = 0.0;
        private double cosX = 0.0;

        HueStats() {
        }

        public void update(SimpleImage img) {
            for (int y = 0; y < img.getHeight(); ++y) {
                for (int x = 0; x < img.getWidth(); ++x) {
                    float val = img.getValue(x, y);
                    if (Float.isNaN(val)) continue;
                    double alpha = (double)(val * 2.0f) * Math.PI;
                    this.sinX += Math.sin(alpha);
                    this.cosX += Math.cos(alpha);
                }
            }
        }

        public double getMeanHue() {
            return Math.atan2(this.sinX, this.cosX) / (Math.PI * 2) + 0.5;
        }
    }

    static class HaralickFeaturesComp
    implements FeatureComputer {
        private CoocurranceMatrices matrices;

        HaralickFeaturesComp() {
        }

        @Override
        public void updateFeatures(SimpleImage img, FeatureColorTransform transform, ParameterList params) {
            if (!Boolean.TRUE.equals(params.getBooleanParameterValue("doHaralick"))) {
                return;
            }
            if (transform == FeatureColorTransformEnum.HUE) {
                return;
            }
            double haralickMin = Double.NaN;
            double haralickMax = Double.NaN;
            if (params.containsKey((Object)"haralickMin")) {
                haralickMin = params.getDoubleParameterValue("haralickMin");
            }
            if (params.containsKey((Object)"haralickMax")) {
                haralickMax = params.getDoubleParameterValue("haralickMax");
            }
            double[] minMax = transform.getHaralickMinMax();
            if (Double.isFinite(haralickMin) && Double.isFinite(haralickMax) && haralickMax > haralickMin) {
                logger.trace("Using Haralick min/max {}, {}", (Object)haralickMin, (Object)haralickMax);
                minMax = new double[]{haralickMin, haralickMax};
            } else {
                if (minMax == null) {
                    return;
                }
                logger.trace("Using default Haralick min/max {}, {}", (Object)haralickMin, (Object)haralickMax);
            }
            int d = params.getIntParameterValue("haralickDistance");
            int nBins = params.getIntParameterValue("haralickBins");
            this.matrices = HaralickFeatureComputer.updateCooccurrenceMatrices(this.matrices, img, null, nBins, minMax[0], minMax[1], d);
        }

        @Override
        public void addMeasurements(PathObject pathObject, String name, ParameterList params) {
            if (this.matrices == null) {
                return;
            }
            MeasurementList measurementList = pathObject.getMeasurementList();
            HaralickFeatures haralickFeatures = this.matrices.getMeanFeatures();
            for (int i = 0; i < haralickFeatures.nFeatures(); ++i) {
                measurementList.put(String.format("%s Haralick %s (F%d)", name, haralickFeatures.getFeatureName(i), i), haralickFeatures.getFeature(i));
            }
        }
    }

    static class MedianFeatureComputer
    implements FeatureComputer {
        private int originalBitsPerPixel;
        private double minBin;
        private double maxBin;
        private long n;
        private int nBins;
        private long[] histogram;

        MedianFeatureComputer(int originalBitsPerPixel) {
            this.originalBitsPerPixel = originalBitsPerPixel;
        }

        @Override
        public void updateFeatures(SimpleImage img, FeatureColorTransform transform, ParameterList params) {
            block16: {
                block18: {
                    block17: {
                        if (transform == null || transform == FeatureColorTransformEnum.HUE || this.histogram != null && this.histogram.length == 0) {
                            return;
                        }
                        if (this.histogram != null) break block16;
                        if (!(transform instanceof FeatureColorTransformEnum)) break block17;
                        switch (((FeatureColorTransformEnum)transform).ordinal()) {
                            case 4: 
                            case 5: 
                            case 6: {
                                this.nBins = 256;
                                this.minBin = 0.0;
                                this.maxBin = 255.0;
                                break block18;
                            }
                            case 8: 
                            case 9: {
                                this.nBins = 1001;
                                this.minBin = 0.0;
                                this.maxBin = 1.0;
                                break block18;
                            }
                            case 7: {
                                break block18;
                            }
                            case 0: 
                            case 1: 
                            case 2: 
                            case 3: {
                                this.nBins = 4001;
                                this.minBin = 0.0;
                                this.maxBin = 4.0;
                                break block18;
                            }
                            default: {
                                this.histogram = new long[0];
                                return;
                            }
                        }
                    }
                    if (this.originalBitsPerPixel <= 16) {
                        this.nBins = (int)Math.pow(2.0, this.originalBitsPerPixel);
                        this.minBin = 0.0;
                        this.maxBin = this.nBins - 1;
                    } else {
                        this.histogram = new long[0];
                        return;
                    }
                }
                this.histogram = new long[this.nBins];
            }
            if (this.nBins == 0) {
                return;
            }
            double binWidth = (this.maxBin - this.minBin) / (double)(this.nBins - 1);
            for (int y = 0; y < img.getHeight(); ++y) {
                for (int x = 0; x < img.getWidth(); ++x) {
                    double val = img.getValue(x, y);
                    if (!Double.isFinite(val)) continue;
                    int bin = (int)((val - this.minBin) / binWidth);
                    if (bin >= this.nBins) {
                        int n = this.nBins - 1;
                        this.histogram[n] = this.histogram[n] + 1L;
                    } else if (bin < 0) {
                        this.histogram[0] = this.histogram[0] + 1L;
                    } else {
                        int n = bin;
                        this.histogram[n] = this.histogram[n] + 1L;
                    }
                    ++this.n;
                }
            }
        }

        @Override
        public void addMeasurements(PathObject pathObject, String name, ParameterList params) {
            boolean doMedian;
            boolean bl = doMedian = params.containsKey((Object)"doMedian") && Boolean.TRUE.equals(params.getBooleanParameterValue("doMedian"));
            if (!doMedian || this.histogram == null || this.histogram.length == 0) {
                return;
            }
            double median = Double.NaN;
            double halfway = (double)this.n / 2.0;
            if (this.n > 0L) {
                int bin;
                long sum = 0L;
                for (bin = 0; bin < this.histogram.length && !((double)(sum += this.histogram[bin]) >= halfway); ++bin) {
                }
                if (bin == this.histogram.length) {
                    median = this.maxBin;
                } else if (bin == 0) {
                    median = this.minBin;
                } else {
                    double binWidth = (this.maxBin - this.minBin) / (double)(this.nBins - 1);
                    median = this.minBin + (double)bin * binWidth + binWidth / 2.0;
                }
            }
            MeasurementList measurementList = pathObject.getMeasurementList();
            measurementList.put(name + " Median", median);
        }
    }

    static class BasicFeatureComputer
    implements FeatureComputer {
        private RunningStatistics stats;
        private HueStats hueStats;

        BasicFeatureComputer() {
        }

        @Override
        public void updateFeatures(SimpleImage img, FeatureColorTransform transform, ParameterList params) {
            boolean requireFeatures = false;
            for (Feature feature : Feature.values()) {
                if (!Boolean.TRUE.equals(params.getBooleanParameterValue(feature.key))) continue;
                requireFeatures = true;
                break;
            }
            if (!requireFeatures) {
                return;
            }
            if (transform == FeatureColorTransformEnum.HUE) {
                if (this.hueStats == null) {
                    this.hueStats = new HueStats();
                }
                this.hueStats.update(img);
                return;
            }
            if (this.stats == null) {
                this.stats = new RunningStatistics();
            }
            StatisticsHelper.updateRunningStatistics((RunningStatistics)this.stats, (SimpleImage)img);
        }

        @Override
        public void addMeasurements(PathObject pathObject, String name, ParameterList params) {
            if (this.hueStats != null) {
                pathObject.getMeasurementList().put(name + " Mean", this.hueStats.getMeanHue());
                return;
            }
            if (this.stats == null) {
                return;
            }
            MeasurementList measurementList = pathObject.getMeasurementList();
            if (params.getBooleanParameterValue(Feature.MEAN.key).booleanValue()) {
                measurementList.put(name + " Mean", this.stats.getMean());
            }
            if (params.getBooleanParameterValue(Feature.STD_DEV.key).booleanValue()) {
                measurementList.put(name + " Std.dev.", this.stats.getStdDev());
            }
            if (params.getBooleanParameterValue(Feature.MIN_MAX.key).booleanValue()) {
                measurementList.put(name + " Min", this.stats.getMin());
                measurementList.put(name + " Max", this.stats.getMax());
            }
            logger.trace("Measured pixel count: {}", (Object)this.stats.size());
        }

        static enum Feature {
            MEAN("doMean", "Mean", "Compute mean intensity"),
            STD_DEV("doStdDev", "Standard deviation", "Compute standard deviation of intensities"),
            MIN_MAX("doMinMax", "Min & Max", "Compute minimum & maximum of intensities");

            private String key;
            private String prompt;
            private String help;

            private Feature(String key, String prompt, String help) {
                this.key = key;
                this.prompt = prompt;
                this.help = help;
            }
        }
    }
}

