/*
 * Decompiled with CFR 0.152.
 */
package qupath.imagej.detect.cells;

import ij.CompositeImage;
import ij.ImagePlus;
import ij.ImageStack;
import ij.Prefs;
import ij.gui.Overlay;
import ij.gui.PolygonRoi;
import ij.gui.Roi;
import ij.measure.Calibration;
import ij.plugin.ContrastEnhancer;
import ij.plugin.filter.EDM;
import ij.plugin.filter.RankFilters;
import ij.process.ByteProcessor;
import ij.process.ColorProcessor;
import ij.process.FloatPolygon;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;
import ij.process.ImageStatistics;
import ij.process.ShortProcessor;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.imagej.detect.cells.ObjectMeasurements;
import qupath.imagej.processing.MorphologicalReconstruction;
import qupath.imagej.processing.RoiLabeling;
import qupath.imagej.processing.SimpleThresholding;
import qupath.imagej.processing.Watershed;
import qupath.imagej.tools.IJTools;
import qupath.imagej.tools.PixelImageIJ;
import qupath.lib.analysis.images.SimpleImage;
import qupath.lib.analysis.stats.RunningStatistics;
import qupath.lib.analysis.stats.StatisticsHelper;
import qupath.lib.color.ColorDeconvolutionStains;
import qupath.lib.color.StainVector;
import qupath.lib.common.GeneralTools;
import qupath.lib.images.ImageData;
import qupath.lib.images.PathImage;
import qupath.lib.images.servers.ImageChannel;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.images.servers.ServerTools;
import qupath.lib.measurements.MeasurementList;
import qupath.lib.measurements.MeasurementListFactory;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.PathObjectTools;
import qupath.lib.objects.PathObjects;
import qupath.lib.objects.classes.PathClass;
import qupath.lib.plugins.AbstractTileableDetectionPlugin;
import qupath.lib.plugins.ObjectDetector;
import qupath.lib.plugins.parameters.DoubleParameter;
import qupath.lib.plugins.parameters.Parameter;
import qupath.lib.plugins.parameters.ParameterList;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.PolygonROI;
import qupath.lib.roi.ShapeSimplifier;
import qupath.lib.roi.interfaces.ROI;

public class WatershedCellDetection
extends AbstractTileableDetectionPlugin<BufferedImage> {
    protected boolean parametersInitialized = false;
    private static boolean debugMode = false;
    private static String[] micronParameters = new String[]{"requestedPixelSizeMicrons", "backgroundRadiusMicrons", "medianRadiusMicrons", "sigmaMicrons", "minAreaMicrons", "maxAreaMicrons", "cellExpansionMicrons"};
    private static String[] pixelParameters = new String[]{"backgroundRadius", "medianRadius", "sigma", "minArea", "maxArea", "cellExpansion"};
    private static String[] fluorescenceParameters = new String[]{"detectionImage"};
    private static String[] brightfieldParameters = new String[]{"detectionImageBrightfield", "maxBackground"};
    private transient CellDetector detector;
    private static final Logger logger = LoggerFactory.getLogger(WatershedCellDetection.class);
    static String IMAGE_OPTICAL_DENSITY = "Optical density sum";
    static String IMAGE_HEMATOXYLIN = "Hematoxylin OD";
    ParameterList params;

    public static void setDebugMode(boolean debug) {
        debugMode = debug;
    }

    public static boolean getDebugMode() {
        return debugMode;
    }

    private ParameterList buildParameterList(ImageData<BufferedImage> imageData) {
        ParameterList params = new ParameterList();
        String microns = "\u00b5m";
        params.addTitleParameter("Setup parameters");
        String defaultChannel = null;
        ArrayList<String> channelNames = new ArrayList<String>();
        String[] nucleusGuesses = new String[]{"dapi", "hoechst", "nucleus", "nuclei", "nuclear", "hematoxylin", "haematoxylin"};
        for (ImageChannel channel : imageData.getServerMetadata().getChannels()) {
            String name = channel.getName();
            channelNames.add(name);
            if (defaultChannel != null) continue;
            String lower = name.toLowerCase();
            for (String guess : nucleusGuesses) {
                if (!lower.contains(guess)) continue;
                defaultChannel = name;
            }
        }
        if (defaultChannel == null) {
            defaultChannel = (String)channelNames.get(0);
        }
        if (channelNames.size() != new HashSet(channelNames).size()) {
            logger.warn("Image contains duplicate channel names! This may be confusing for detection and analysis.");
        }
        params.addChoiceParameter("detectionImage", "Detection channel", (Object)defaultChannel, channelNames, "Choose the channel that should be used for nucleus detection (e.g. DAPI)");
        params.addChoiceParameter("detectionImageBrightfield", "Detection image", (Object)IMAGE_HEMATOXYLIN, Arrays.asList(IMAGE_HEMATOXYLIN, IMAGE_OPTICAL_DENSITY), "Transformed image to which to apply the detection");
        params.addDoubleParameter("requestedPixelSizeMicrons", "Requested pixel size", 0.5, microns, "Choose pixel size at which detection will be performed - higher values are likely to be faster, but may be less accurate; set <= 0 to use the full image resolution");
        params.addTitleParameter("Nucleus parameters");
        params.addDoubleParameter("backgroundRadiusMicrons", "Background radius", 8.0, microns, "Radius for background estimation, should be > the largest nucleus radius, or <= 0 to turn off background subtraction");
        params.addBooleanParameter("backgroundByReconstruction", "Use opening by reconstruction", true, "Use opening-by-reconstruction for background estimation (default is 'Yes').\nOpening by reconstruction tends to give a 'better' background estimate, because it incorporates more information across the image tile used for cell detection.\n*However*, in some cases (e.g. images with prominent folds, background staining, or other artefacts)  this can cause problems, with the background estimate varying substantially between tiles.\nOpening by reconstruction was always used in QuPath before v0.4.0, but now it is optional.");
        params.addDoubleParameter("medianRadiusMicrons", "Median filter radius", 0.0, microns, "Radius of median filter used to reduce image texture (optional)");
        params.addDoubleParameter("sigmaMicrons", "Sigma", 1.5, microns, "Sigma value for Gaussian filter used to reduce noise; increasing the value stops nuclei being fragmented, but may reduce the accuracy of boundaries");
        params.addDoubleParameter("minAreaMicrons", "Minimum area", 10.0, microns + "^2", "Detected nuclei with an area < minimum area will be discarded");
        params.addDoubleParameter("maxAreaMicrons", "Maximum area", 400.0, microns + "^2", "Detected nuclei with an area > maximum area will be discarded");
        params.addDoubleParameter("backgroundRadius", "Background radius", 15.0, "px", "Radius for background estimation, should be > the largest nucleus radius, or <= 0 to turn off background subtraction");
        params.addDoubleParameter("medianRadius", "Median filter radius", 0.0, "px", "Radius of median filter used to reduce image texture (optional)");
        params.addDoubleParameter("sigma", "Sigma", 3.0, "px", "Sigma value for Gaussian filter used to reduce noise; increasing the value stops nuclei being fragmented, but may reduce the accuracy of boundaries");
        params.addDoubleParameter("minArea", "Minimum area", 10.0, "px^2", "Detected nuclei with an area < minimum area will be discarded");
        params.addDoubleParameter("maxArea", "Maximum area", 1000.0, "px^2", "Detected nuclei with an area > maximum area will be discarded");
        params.addTitleParameter("Intensity parameters");
        params.addDoubleParameter("threshold", "Threshold", 0.1, null, "Intensity threshold - detected nuclei must have a mean intensity >= threshold");
        params.addDoubleParameter("maxBackground", "Max background intensity", 2.0, null, "If background radius > 0, detected nuclei occurring on a background > max background intensity will be discarded");
        params.addBooleanParameter("watershedPostProcess", "Split by shape", true, "Split merged detected nuclei based on shape ('roundness')");
        params.addBooleanParameter("excludeDAB", "Exclude DAB (membrane staining)", false, "Set to 'true' if regions of high DAB staining should not be considered nuclei; useful if DAB stains cell membranes");
        params.addTitleParameter("Cell parameters");
        params.addDoubleParameter("cellExpansionMicrons", "Cell expansion", 5.0, microns, 0.0, 25.0, "Amount by which to expand detected nuclei to approximate the full cell area");
        params.addDoubleParameter("cellExpansion", "Cell expansion", 5.0, "px", "Amount by which to expand detected nuclei to approximate the full cell area");
        params.addBooleanParameter("includeNuclei", "Include cell nucleus", true, "If cell expansion is used, optionally include/exclude the nuclei within the detected cells");
        params.addTitleParameter("General parameters");
        params.addBooleanParameter("smoothBoundaries", "Smooth boundaries", true, "Smooth the detected nucleus/cell boundaries");
        params.addBooleanParameter("makeMeasurements", "Make measurements", true, "Add default shape & intensity measurements during detection");
        return params;
    }

    protected boolean parseArgument(ImageData<BufferedImage> imageData, String arg) {
        Map map;
        if (arg != null && (map = GeneralTools.parseArgStringValues((String)arg)).containsKey("detectionImageFluorescence")) {
            throw new IllegalArgumentException("'detectionImageFluorescence' is not supported in this version of QuPath - use 'detectionImage' instead");
        }
        return super.parseArgument(imageData, arg);
    }

    public ParameterList getDefaultParameterList(ImageData<BufferedImage> imageData) {
        if (!this.parametersInitialized) {
            this.params = this.buildParameterList(imageData);
        }
        Map map = this.params.getParameters();
        boolean pixelSizeKnown = imageData.getServer() != null && imageData.getServer().getPixelCalibration().hasPixelSizeMicrons();
        for (String name : micronParameters) {
            ((Parameter)map.get(name)).setHidden(!pixelSizeKnown);
        }
        for (String name : pixelParameters) {
            ((Parameter)map.get(name)).setHidden(pixelSizeKnown);
        }
        this.params.setHiddenParameters(!pixelSizeKnown, micronParameters);
        this.params.setHiddenParameters(pixelSizeKnown, pixelParameters);
        boolean isBrightfield = imageData.isBrightfield();
        this.params.setHiddenParameters(!isBrightfield, brightfieldParameters);
        this.params.setHiddenParameters(isBrightfield, fluorescenceParameters);
        if (!isBrightfield) {
            if (imageData.getServer().getPixelType().getBitsPerPixel() > 8) {
                ((DoubleParameter)this.params.getParameters().get("threshold")).setValue(Double.valueOf(100.0));
            } else {
                ((DoubleParameter)this.params.getParameters().get("threshold")).setValue(Double.valueOf(25.0));
            }
        }
        ((Parameter)map.get("excludeDAB")).setHidden(imageData.getColorDeconvolutionStains() == null || !imageData.getColorDeconvolutionStains().isH_DAB());
        return this.params;
    }

    public String getName() {
        return "Cell detection";
    }

    public String getLastResultsDescription() {
        return this.detector == null ? "" : this.detector.getLastResultsDescription();
    }

    public String getDescription() {
        return "Default cell detection algorithm for brightfield images with nuclear or cytoplasmic staining";
    }

    protected double getPreferredPixelSizeMicrons(ImageData<BufferedImage> imageData, ParameterList params) {
        return CellDetector.getPreferredPixelSizeMicrons(imageData, params);
    }

    protected ObjectDetector<BufferedImage> createDetector(ImageData<BufferedImage> imageData, ParameterList params) {
        return new CellDetector();
    }

    protected int getTileOverlap(ImageData<BufferedImage> imageData, ParameterList params) {
        double nucleusRadiusMicrons;
        double pxSize = imageData.getServer().getPixelCalibration().getAveragedPixelSizeMicrons();
        if (!Double.isFinite(pxSize)) {
            return params.getDoubleParameterValue("cellExpansion") > 0.0 ? 25 : 10;
        }
        double expansionMicrons = nucleusRadiusMicrons = 10.0;
        double cellExpansion = params.getDoubleParameterValue("cellExpansionMicrons");
        if (cellExpansion > 0.0) {
            expansionMicrons += params.getDoubleParameterValue("cellExpansionMicrons").doubleValue();
        }
        int overlap = (int)(expansionMicrons / pxSize * 2.0);
        return overlap;
    }

    static class CellDetector
    implements ObjectDetector<BufferedImage> {
        private String lastServerPath = null;
        private ROI pathROI;
        private List<PathObject> pathObjects = null;
        private boolean nucleiClassified = false;

        CellDetector() {
        }

        public static double getPreferredPixelSizeMicrons(ImageData<BufferedImage> imageData, ParameterList params) {
            PixelCalibration cal = imageData.getServer().getPixelCalibration();
            if (cal.hasPixelSizeMicrons()) {
                double requestedPixelSize = params.getDoubleParameterValue("requestedPixelSizeMicrons");
                double averagedPixelSize = cal.getAveragedPixelSizeMicrons();
                if (requestedPixelSize < 0.0) {
                    requestedPixelSize = averagedPixelSize * -requestedPixelSize;
                }
                requestedPixelSize = Math.max(requestedPixelSize, averagedPixelSize);
                return requestedPixelSize;
            }
            return Double.NaN;
        }

        public Collection<PathObject> runDetection(ImageData<BufferedImage> imageData, ParameterList params, ROI pathROI) throws IOException {
            double cellExpansion;
            double maxArea;
            double minArea;
            double sigma;
            double medianRadius;
            double backgroundRadius;
            if (pathROI == null) {
                throw new IOException("Cell detection requires a ROI!");
            }
            PathImage<ImagePlus> pathImage = null;
            if (this.lastServerPath == null || !this.lastServerPath.equals(imageData.getServerPath()) || pathImage == null || !pathROI.equals((Object)this.pathROI)) {
                ImageServer server = imageData.getServer();
                this.lastServerPath = imageData.getServerPath();
                double downsample = ServerTools.getDownsampleFactor((ImageServer)server, (double)CellDetector.getPreferredPixelSizeMicrons(imageData, params));
                RegionRequest request = RegionRequest.createInstance((String)server.getPath(), (double)downsample, (ROI)pathROI);
                pathImage = IJTools.convertToImagePlus((ImageServer<BufferedImage>)server, request);
                logger.trace("Cell detection with downsample: {}", (Object)pathImage.getDownsampleFactor());
                this.pathROI = pathROI;
            }
            boolean isBrightfield = imageData.isBrightfield();
            ImageProcessor ip = ((ImagePlus)pathImage.getImage()).getProcessor();
            FloatProcessor fpDetection = null;
            ColorDeconvolutionStains stains = imageData.getColorDeconvolutionStains();
            LinkedHashMap<String, FloatProcessor> channels = new LinkedHashMap<String, FloatProcessor>();
            LinkedHashMap<String, FloatProcessor> channelsCell = new LinkedHashMap<String, FloatProcessor>();
            Roi roi = null;
            if (pathROI != null) {
                roi = IJTools.convertToIJRoi(pathROI, pathImage);
            }
            if (stains != null && isBrightfield) {
                FloatProcessor[] fps;
                if (ip instanceof ColorProcessor) {
                    ColorProcessor cp = (ColorProcessor)ip;
                    fps = IJTools.colorDeconvolve(cp, stains);
                } else if (((ImagePlus)pathImage.getImage()).getNChannels() == 3) {
                    ImagePlus imp = (ImagePlus)pathImage.getImage();
                    fps = IJTools.colorDeconvolve(imp.getStack().getProcessor(1), imp.getStack().getProcessor(2), imp.getStack().getProcessor(3), stains);
                } else {
                    throw new IllegalArgumentException("Unsupported image for color deconvolution: " + String.valueOf(pathImage.getImage()));
                }
                for (int i = 0; i < 3; ++i) {
                    StainVector stain = stains.getStain(i + 1);
                    if (stain.isResidual()) continue;
                    channels.put(stain.getName() + " OD", fps[i]);
                    channelsCell.put(stain.getName() + " OD", fps[i]);
                }
                if (!((Parameter)params.getParameters().get("detectionImageBrightfield")).isHidden()) {
                    String stainChoice = (String)params.getChoiceParameterValue("detectionImageBrightfield");
                    if (stainChoice.equals(IMAGE_OPTICAL_DENSITY)) {
                        fpDetection = IJTools.convertToOpticalDensitySum((ColorProcessor)ip, stains.getMaxRed(), stains.getMaxGreen(), stains.getMaxBlue());
                    } else if (stainChoice.equals(IMAGE_HEMATOXYLIN)) {
                        for (int i = 0; i < 3; ++i) {
                            if (!ColorDeconvolutionStains.isHematoxylin((StainVector)stains.getStain(i + 1))) continue;
                            fpDetection = (FloatProcessor)fps[i].duplicate();
                            if (i <= 0) continue;
                            logger.warn("Hematoxylin expected to be stain 1, but here it is stain {}", (Object)(i + 1));
                        }
                        if (fpDetection == null) {
                            logger.warn("Hematoxylin stain not found! The first stain will be used by default ({}).", (Object)stains.getStain(1).getName());
                            fpDetection = (FloatProcessor)fps[0].duplicate();
                        }
                    } else {
                        for (int i = 0; i < 3; ++i) {
                            String currentStainName = stains.getStain(i + 1).getName();
                            if (!stainChoice.equals(currentStainName) && !stainChoice.equals(currentStainName + " OD")) continue;
                            fpDetection = (FloatProcessor)fps[i].duplicate();
                            logger.warn("Using stain {} for cell detection", (Object)currentStainName);
                        }
                        if (fpDetection == null) {
                            logger.warn("Unknown detection channel {}, I will use the first stain", (Object)stainChoice);
                            fpDetection = (FloatProcessor)fps[0].duplicate();
                        }
                    }
                }
            }
            if (fpDetection == null) {
                List imageChannels = imageData.getServerMetadata().getChannels();
                if (ip instanceof ColorProcessor) {
                    for (int c = 0; c < 3; ++c) {
                        String name = ((ImageChannel)imageChannels.get(c)).getName();
                        channels.put(name, ((ColorProcessor)ip).toFloat(c, null));
                    }
                } else {
                    ImagePlus imp = (ImagePlus)pathImage.getImage();
                    for (int c = 1; c <= imp.getNChannels(); ++c) {
                        String name = ((ImageChannel)imageChannels.get(c - 1)).getName();
                        if (channels.containsKey(name)) {
                            logger.warn("Channel with duplicate name '{}' - will be skipped", (Object)name);
                            continue;
                        }
                        channels.put(name, imp.getStack().getProcessor(imp.getStackIndex(c, 0, 0)).convertToFloatProcessor());
                    }
                }
                channelsCell.putAll(channels);
                if (!isBrightfield) {
                    String detectionChannelName = (String)params.getChoiceParameterValue("detectionImage");
                    fpDetection = (FloatProcessor)channels.get(detectionChannelName);
                } else {
                    throw new IllegalArgumentException("No valid detection channel is selected!");
                }
            }
            WatershedCellDetector detector2 = new WatershedCellDetector(fpDetection, channels, channelsCell, roi, pathImage);
            if (this.pathObjects == null) {
                this.pathObjects = new ArrayList<PathObject>();
            } else {
                this.pathObjects.clear();
            }
            if (pathImage.getPixelCalibration().hasPixelSizeMicrons()) {
                double pixelSize = pathImage.getPixelCalibration().getAveragedPixelSizeMicrons();
                backgroundRadius = params.getDoubleParameterValue("backgroundRadiusMicrons") / pixelSize;
                medianRadius = params.getDoubleParameterValue("medianRadiusMicrons") / pixelSize;
                sigma = params.getDoubleParameterValue("sigmaMicrons") / pixelSize;
                minArea = params.getDoubleParameterValue("minAreaMicrons") / (pixelSize * pixelSize);
                maxArea = params.getDoubleParameterValue("maxAreaMicrons") / (pixelSize * pixelSize);
                cellExpansion = params.getDoubleParameterValue("cellExpansionMicrons") / pixelSize;
            } else {
                backgroundRadius = params.getDoubleParameterValue("backgroundRadius");
                medianRadius = params.getDoubleParameterValue("medianRadius");
                sigma = params.getDoubleParameterValue("sigma");
                minArea = params.getDoubleParameterValue("minArea");
                maxArea = params.getDoubleParameterValue("maxArea");
                cellExpansion = params.getDoubleParameterValue("cellExpansion");
            }
            detector2.runDetection(backgroundRadius, isBrightfield ? params.getDoubleParameterValue("maxBackground") : Double.NEGATIVE_INFINITY, medianRadius, sigma, params.getDoubleParameterValue("threshold"), minArea, maxArea, true, params.getBooleanParameterValue("watershedPostProcess"), params.getBooleanParameterValue("excludeDAB"), cellExpansion, params.getBooleanParameterValue("smoothBoundaries"), params.getBooleanParameterValue("includeNuclei"), params.getBooleanParameterValue("makeMeasurements"), pathROI.getZ(), pathROI.getT(), params.getBooleanParameterValue("backgroundByReconstruction"));
            this.pathObjects.addAll(detector2.getPathObjects());
            return this.pathObjects;
        }

        public String getLastResultsDescription() {
            if (this.pathObjects == null) {
                return null;
            }
            int nDetections = this.pathObjects.size();
            if (nDetections == 1) {
                return "1 nucleus detected";
            }
            String s = String.format("%d nuclei detected", nDetections);
            if (this.nucleiClassified) {
                int nPositive = PathObjectTools.countObjectsWithClass(this.pathObjects, (PathClass)PathClass.getNegative(null), (boolean)false);
                int nNegative = PathObjectTools.countObjectsWithClass(this.pathObjects, (PathClass)PathClass.getPositive(null), (boolean)false);
                return String.format("%s (%.3f%% positive)", s, (double)nPositive * 100.0 / (double)(nPositive + nNegative));
            }
            return s;
        }
    }

    static class WatershedCellDetector {
        private ImagePlus impDebug;
        private boolean refineBoundary = true;
        private double backgroundRadius = 15.0;
        private double maxBackground = 0.3;
        private int z = 0;
        private int t = 0;
        private boolean lastRunCompleted = false;
        private boolean includeNuclei = true;
        private double cellExpansion = 0.0;
        private double minArea = 0.0;
        private double maxArea = 0.0;
        private double medianRadius = 2.0;
        private double sigma = 2.5;
        private double threshold = 0.3;
        private boolean mergeAll = true;
        private boolean watershedPostProcess = true;
        private boolean excludeDAB = false;
        private boolean smoothBoundaries = false;
        private boolean backgroundByReconstruction = true;
        private boolean makeMeasurements = true;
        private Roi roi = null;
        private FloatProcessor fpDetection = null;
        private Map<String, FloatProcessor> channels = new LinkedHashMap<String, FloatProcessor>();
        private Map<String, FloatProcessor> channelsCell = new LinkedHashMap<String, FloatProcessor>();
        private ImageProcessor ipToMeasure = null;
        private List<PolygonRoi> rois = null;
        private ByteProcessor bpLoG = null;
        private List<PolygonRoi> roisNuclei = new ArrayList<PolygonRoi>();
        private List<PathObject> pathObjects = new ArrayList<PathObject>();
        private PathImage<ImagePlus> pathImage = null;

        public WatershedCellDetector(FloatProcessor fpDetection, Map<String, FloatProcessor> channels, Map<String, FloatProcessor> channelsCell, Roi roi, PathImage<ImagePlus> pathImage) {
            this.fpDetection = fpDetection;
            if (channels != null) {
                this.channels.putAll(channels);
            }
            if (channelsCell != null) {
                this.channelsCell.putAll(channelsCell);
            }
            this.roi = roi;
            this.pathImage = pathImage;
            Prefs.setThreads((int)1);
        }

        private static ByteProcessor estimateBackground(ImageProcessor ip, ImageProcessor ipBackground, double radius, double maxBackground, boolean openingByReconstruction) {
            if (openingByReconstruction) {
                logger.debug("Estimating background using opening by reconstruction");
            } else {
                logger.debug("Estimating background using simple opening");
            }
            RankFilters rf = new RankFilters();
            ipBackground.setRoi(ip.getRoi());
            rf.rank(ipBackground, radius, 1);
            ByteProcessor bpMask = null;
            if (!Double.isNaN(maxBackground) && maxBackground > 0.0) {
                int i;
                int w = ip.getWidth();
                int h = ip.getHeight();
                for (i = 0; i < w * h; ++i) {
                    if (!((double)ipBackground.getf(i) > maxBackground)) continue;
                    if (bpMask == null) {
                        bpMask = new ByteProcessor(w, h);
                    }
                    bpMask.setf(i, 1.0f);
                }
                if (bpMask != null) {
                    rf.rank(bpMask, radius * 2.0, 2);
                    for (i = 0; i < w * h; ++i) {
                        if (bpMask.getf(i) == 0.0f) continue;
                        ipBackground.setf(i, Float.NEGATIVE_INFINITY);
                    }
                }
            }
            if (openingByReconstruction) {
                MorphologicalReconstruction.morphologicalReconstruction(ipBackground, ip);
            } else {
                rf.rank(ipBackground, radius, 2);
            }
            return bpMask;
        }

        /*
         * WARNING - void declaration
         */
        private void doDetection(boolean regenerateROIs) {
            int width = this.fpDetection.getWidth();
            int height = this.fpDetection.getHeight();
            if (debugMode) {
                ImageStack stack = new ImageStack(width, height);
                stack.addSlice("Input image", this.fpDetection.duplicate());
                this.impDebug = new ImagePlus("Debug stack", stack);
            }
            this.lastRunCompleted = false;
            this.pathObjects.clear();
            ByteProcessor bp = null;
            ByteProcessor bpBackgroundMask = null;
            this.fpDetection.setRoi(this.roi);
            if (regenerateROIs) {
                this.rois = null;
                this.bpLoG = null;
                FloatProcessor fpLoG = (FloatProcessor)this.fpDetection.duplicate();
                RankFilters rankFilters = new RankFilters();
                if (this.medianRadius > 0.0) {
                    rankFilters.rank((ImageProcessor)fpLoG, this.medianRadius, 4);
                    if (debugMode) {
                        this.impDebug.getStack().addSlice("Median filtered", fpLoG.duplicate());
                    }
                }
                if (this.excludeDAB && this.channels.containsKey("Hematoxylin OD") && this.channels.containsKey("DAB OD")) {
                    FloatProcessor floatProcessor = this.channels.get("DAB OD");
                    floatProcessor.setRoi(this.roi);
                    ByteProcessor bpH = SimpleThresholding.greaterThanOrEqual((ImageProcessor)this.channels.get("Hematoxylin OD"), (ImageProcessor)floatProcessor);
                    bpH.multiply(0.00392156862745098);
                    rankFilters.rank((ImageProcessor)bpH, 2.5, 4);
                    rankFilters.rank((ImageProcessor)bpH, 2.5, 2);
                    fpLoG.copyBits((ImageProcessor)bpH, 0, 0, 5);
                    if (debugMode) {
                        this.impDebug.getStack().addSlice("DAB excluded", fpLoG.duplicate());
                    }
                }
                if (this.backgroundRadius > 0.0) {
                    ImageProcessor imageProcessor = fpLoG.duplicate();
                    bpBackgroundMask = WatershedCellDetector.estimateBackground((ImageProcessor)fpLoG, imageProcessor, this.backgroundRadius, this.maxBackground, this.backgroundByReconstruction);
                    fpLoG.copyBits(imageProcessor, 0, 0, 4);
                    this.ipToMeasure = fpLoG.duplicate();
                    if (debugMode) {
                        this.impDebug.getStack().addSlice("Background estimate", imageProcessor.duplicate());
                        this.impDebug.getStack().addSlice("Background subtracted", fpLoG.duplicate());
                    }
                } else {
                    this.ipToMeasure = this.fpDetection;
                }
                fpLoG.blurGaussian(this.sigma);
                fpLoG.convolve(new float[]{0.0f, -1.0f, 0.0f, -1.0f, 4.0f, -1.0f, 0.0f, -1.0f, 0.0f}, 3, 3);
                if (debugMode) {
                    this.impDebug.getStack().addSlice("Laplacian of Gaussian filtered", fpLoG.duplicate());
                }
                this.bpLoG = SimpleThresholding.thresholdAbove((ImageProcessor)fpLoG, 0.0);
                fpLoG.setRoi(this.roi);
                ImageProcessor imageProcessor = MorphologicalReconstruction.findRegionalMaxima((ImageProcessor)fpLoG, 0.001f, false);
                ImageProcessor ipLabels = RoiLabeling.labelImage(imageProcessor, 0.0f, false);
                Watershed.doWatershed((ImageProcessor)fpLoG, ipLabels, 0.0, false);
                if (debugMode) {
                    this.impDebug.getStack().addSlice("Watershed labels", ipLabels.duplicate());
                }
                ipLabels.setThreshold(0.5, Double.POSITIVE_INFINITY, 2);
                this.rois = RoiLabeling.getFilledPolygonROIs(ipLabels, 4);
                if (Thread.currentThread().isInterrupted()) {
                    return;
                }
            }
            if (bp == null) {
                bp = new ByteProcessor(width, height);
            }
            bp.setValue(255.0);
            for (Roi roi : this.rois) {
                this.ipToMeasure.setRoi(roi);
                double d = this.ipToMeasure.getStatistics().mean;
                if (d <= this.threshold) continue;
                if (bpBackgroundMask != null) {
                    bpBackgroundMask.setRoi(roi);
                    if (bpBackgroundMask.getStatistics().mean > 0.0) continue;
                }
                bp.fill(roi);
            }
            if (Thread.currentThread().isInterrupted()) {
                return;
            }
            bp.setThreshold(127.0, Double.POSITIVE_INFINITY, 2);
            if (this.mergeAll) {
                bp.filter(4);
                bp.copyBits((ImageProcessor)this.bpLoG, 0, 0, 9);
                if (this.watershedPostProcess) {
                    List<PolygonRoi> rois2 = RoiLabeling.getFilledPolygonROIs((ImageProcessor)bp, 4);
                    bp.setValue(255.0);
                    for (Roi roi : rois2) {
                        bp.fill(roi);
                    }
                    new EDM().toWatershed((ImageProcessor)bp);
                }
            }
            if (this.roi != null) {
                RoiLabeling.clearOutside((ImageProcessor)bp, this.roi);
            }
            bp.setThreshold(127.0, Double.POSITIVE_INFINITY, 2);
            if (this.impDebug != null) {
                this.impDebug.getStack().addSlice("Binary", bp.duplicate());
            }
            if (this.refineBoundary && this.sigma > 1.5) {
                FloatProcessor fpBoundaryCleanup = (FloatProcessor)this.fpDetection.duplicate();
                fpBoundaryCleanup.blurGaussian(1.0);
                fpBoundaryCleanup.convolve(new float[]{0.0f, -1.0f, 0.0f, -1.0f, 4.0f, -1.0f, 0.0f, -1.0f, 0.0f}, 3, 3);
                ByteProcessor byteProcessor = SimpleThresholding.thresholdAbove((ImageProcessor)fpBoundaryCleanup, 0.0);
                byteProcessor.copyBits((ImageProcessor)bp, 0, 0, 12);
                bp.filter(3);
                bp.copyBits((ImageProcessor)byteProcessor, 0, 0, 13);
                regenerateROIs = true;
                if (debugMode) {
                    this.impDebug.getStack().addSlice("Refined boundaries", (ImageProcessor)byteProcessor.convertToFloatProcessor());
                }
            }
            this.roisNuclei = RoiLabeling.getFilledPolygonROIs((ImageProcessor)bp, 4);
            if (Thread.currentThread().isInterrupted()) {
                return;
            }
            if (this.minArea > 0.0 || this.maxArea > 0.0) {
                bp.setValue(0.0);
                Iterator<PolygonRoi> iter = this.roisNuclei.iterator();
                while (iter.hasNext()) {
                    Roi roi = (Roi)iter.next();
                    this.ipToMeasure.setRoi(roi);
                    ImageStatistics imageStatistics = ImageStatistics.getStatistics((ImageProcessor)this.ipToMeasure, (int)3, null);
                    double area = imageStatistics.pixelCount;
                    if (!(imageStatistics.mean < this.threshold || this.minArea > 0.0 && area < this.minArea) && (!(this.maxArea > 0.0) || !(area > this.maxArea))) continue;
                    iter.remove();
                    bp.fill(roi);
                }
                this.ipToMeasure.resetRoi();
            }
            ShortProcessor ipLabels = new ShortProcessor(width, height);
            RoiLabeling.labelROIs((ImageProcessor)ipLabels, this.roisNuclei);
            if (debugMode) {
                this.impDebug.getStack().addSlice("Labeled ROIs", (ImageProcessor)ipLabels.convertToFloatProcessor());
            }
            LinkedHashMap<String, List> linkedHashMap = new LinkedHashMap<String, List>();
            if (this.makeMeasurements) {
                PixelImageIJ pixelImageIJ = new PixelImageIJ((ImageProcessor)ipLabels);
                for (String key : this.channels.keySet()) {
                    List statsList = StatisticsHelper.createRunningStatisticsList((int)this.roisNuclei.size());
                    StatisticsHelper.computeRunningStatistics((SimpleImage)new PixelImageIJ((ImageProcessor)this.channels.get(key)), (SimpleImage)pixelImageIJ, (List)statsList);
                    linkedHashMap.put(key, statsList);
                }
            }
            if (Thread.currentThread().isInterrupted()) {
                return;
            }
            double d = this.pathImage.getDownsampleFactor();
            double downsampleSqrt = Math.sqrt(d);
            ArrayList<PathObject> nucleiObjects = new ArrayList<PathObject>();
            Calibration cal = ((ImagePlus)this.pathImage.getImage()).getCalibration();
            ImagePlane plane = ImagePlane.getPlane((int)this.z, (int)this.t);
            for (int i = 0; i < this.roisNuclei.size(); ++i) {
                PolygonRoi rOrig;
                PolygonRoi r = rOrig = this.roisNuclei.get(i);
                if (this.smoothBoundaries) {
                    r = new PolygonRoi(rOrig.getInterpolatedPolygon(1.0, false), 2);
                    r = WatershedCellDetector.smoothPolygonRoi(r);
                    r = new PolygonRoi(r.getInterpolatedPolygon(Math.min(2.0, (double)r.getNCoordinates() * 0.1), false), 2);
                }
                PolygonROI pathROI = IJTools.convertToPolygonROI(r, cal, d, plane);
                if (this.smoothBoundaries) {
                    pathROI = ShapeSimplifier.simplifyPolygon((PolygonROI)pathROI, (double)(downsampleSqrt / 2.0));
                }
                MeasurementList measurementList = MeasurementListFactory.createMeasurementList((int)(this.makeMeasurements ? 30 : 0), (MeasurementList.MeasurementListType)MeasurementList.MeasurementListType.FLOAT);
                if (this.makeMeasurements) {
                    ObjectMeasurements.addShapeStatistics(measurementList, (Roi)r, (ImageProcessor)this.fpDetection, cal, "Nucleus: ");
                    for (String key : this.channels.keySet()) {
                        List statsList = (List)linkedHashMap.get(key);
                        RunningStatistics runningStatistics = (RunningStatistics)statsList.get(i);
                        measurementList.put("Nucleus: " + key + " mean", runningStatistics.getMean());
                        measurementList.put("Nucleus: " + key + " sum", runningStatistics.getSum());
                        measurementList.put("Nucleus: " + key + " std dev", runningStatistics.getStdDev());
                        measurementList.put("Nucleus: " + key + " max", runningStatistics.getMax());
                        measurementList.put("Nucleus: " + key + " min", runningStatistics.getMin());
                        measurementList.put("Nucleus: " + key + " range", runningStatistics.getRange());
                    }
                }
                PathObject pathObject = PathObjects.createDetectionObject((ROI)pathROI, null, (MeasurementList)measurementList);
                nucleiObjects.add(pathObject);
            }
            if (Thread.currentThread().isInterrupted()) {
                return;
            }
            ArrayList<PolygonRoi> roisCellsList = null;
            if (this.cellExpansion > 0.0) {
                void var23_55;
                FloatProcessor fpEDM = new EDM().makeFloatEDM((ImageProcessor)bp, -1, false);
                fpEDM.multiply(-1.0);
                double cellExpansionThreshold = -this.cellExpansion;
                ImageProcessor ipLabelsCells = ipLabels.duplicate();
                Watershed.doWatershed((ImageProcessor)fpEDM, ipLabelsCells, cellExpansionThreshold, false);
                PolygonRoi[] roisCells = RoiLabeling.labelsToFilledROIs(ipLabelsCells, this.roisNuclei.size());
                LinkedHashMap<String, List> statsMapCell = new LinkedHashMap<String, List>();
                if (this.makeMeasurements) {
                    for (String string : this.channelsCell.keySet()) {
                        List statsList = StatisticsHelper.createRunningStatisticsList((int)this.roisNuclei.size());
                        StatisticsHelper.computeRunningStatistics((SimpleImage)new PixelImageIJ((ImageProcessor)this.channelsCell.get(string)), (SimpleImage)new PixelImageIJ(ipLabelsCells), (List)statsList);
                        statsMapCell.put(string, statsList);
                    }
                }
                for (int i = 0; i < ipLabels.getWidth() * ipLabels.getHeight(); ++i) {
                    if (ipLabels.getf(i) == 0.0f) continue;
                    ipLabelsCells.setf(i, 0.0f);
                }
                LinkedHashMap<String, List> statsMapCytoplasm = new LinkedHashMap<String, List>();
                if (this.makeMeasurements) {
                    for (String key : this.channelsCell.keySet()) {
                        List statsList = StatisticsHelper.createRunningStatisticsList((int)this.roisNuclei.size());
                        StatisticsHelper.computeRunningStatistics((SimpleImage)new PixelImageIJ((ImageProcessor)this.channelsCell.get(key)), (SimpleImage)new PixelImageIJ(ipLabelsCells), (List)statsList);
                        statsMapCytoplasm.put(key, statsList);
                    }
                }
                roisCellsList = new ArrayList<PolygonRoi>(roisCells.length);
                boolean bl = false;
                while (var23_55 < roisCells.length) {
                    PolygonRoi r = roisCells[var23_55];
                    if (r != null) {
                        if (this.smoothBoundaries) {
                            r = new PolygonRoi(r.getInterpolatedPolygon(1.0, false), 2);
                            r = WatershedCellDetector.smoothPolygonRoi(r);
                            r = new PolygonRoi(r.getInterpolatedPolygon(Math.min(2.0, (double)r.getNCoordinates() * 0.1), false), 2);
                        }
                        PolygonROI pathROI = IJTools.convertToPolygonROI(r, cal, d, plane);
                        if (this.smoothBoundaries) {
                            pathROI = ShapeSimplifier.simplifyPolygon((PolygonROI)pathROI, (double)(downsampleSqrt / 2.0));
                        }
                        MeasurementList measurementList = null;
                        PathObject nucleus = null;
                        if (this.includeNuclei) {
                            nucleus = (PathObject)nucleiObjects.get((int)var23_55);
                            measurementList = nucleus.getMeasurementList();
                        } else {
                            measurementList = MeasurementListFactory.createMeasurementList((int)(this.makeMeasurements ? 12 : 0), (MeasurementList.MeasurementListType)MeasurementList.MeasurementListType.GENERAL);
                        }
                        if (this.makeMeasurements) {
                            RunningStatistics stats;
                            ObjectMeasurements.addShapeStatistics(measurementList, (Roi)r, (ImageProcessor)this.fpDetection, ((ImagePlus)this.pathImage.getImage()).getCalibration(), "Cell: ");
                            for (String key : this.channelsCell.keySet()) {
                                if (!statsMapCell.containsKey(key)) continue;
                                stats = (RunningStatistics)((List)statsMapCell.get(key)).get((int)var23_55);
                                measurementList.put("Cell: " + key + " mean", stats.getMean());
                                measurementList.put("Cell: " + key + " std dev", stats.getStdDev());
                                measurementList.put("Cell: " + key + " max", stats.getMax());
                                measurementList.put("Cell: " + key + " min", stats.getMin());
                            }
                            for (String key : this.channelsCell.keySet()) {
                                if (!statsMapCytoplasm.containsKey(key)) continue;
                                stats = (RunningStatistics)((List)statsMapCytoplasm.get(key)).get((int)var23_55);
                                measurementList.put("Cytoplasm: " + key + " mean", stats.getMean());
                                measurementList.put("Cytoplasm: " + key + " std dev", stats.getStdDev());
                                measurementList.put("Cytoplasm: " + key + " max", stats.getMax());
                                measurementList.put("Cytoplasm: " + key + " min", stats.getMin());
                            }
                            if (nucleus != null && nucleus.getROI().isArea()) {
                                double nucleusArea = nucleus.getROI().getArea();
                                double cellArea = pathROI.getArea();
                                measurementList.put("Nucleus/Cell area ratio", Math.min(nucleusArea / cellArea, 1.0));
                            }
                        }
                        PathObject pathObject = PathObjects.createCellObject((ROI)pathROI, (ROI)(nucleus == null ? null : nucleus.getROI()), null, (MeasurementList)measurementList);
                        this.pathObjects.add(pathObject);
                        roisCellsList.add(r);
                    }
                    ++var23_55;
                }
            } else {
                this.pathObjects.addAll(nucleiObjects);
            }
            for (PathObject pathObject : this.pathObjects) {
                pathObject.getMeasurementList().close();
            }
            int sizeBefore = this.pathObjects.size();
            this.pathObjects.removeIf(p -> PathObjectTools.getROI((PathObject)p, (boolean)false).isEmpty() || PathObjectTools.getROI((PathObject)p, (boolean)true).isEmpty());
            int sizeAfter = this.pathObjects.size();
            if (sizeBefore != sizeAfter) {
                logger.debug("Filtered out {} invalid cells (empty ROIs)", (Object)(sizeBefore - sizeAfter));
            }
            if (this.impDebug != null) {
                this.impDebug.setDimensions(this.impDebug.getStackSize(), 1, 1);
                if (this.impDebug.getNChannels() <= 8) {
                    this.impDebug = new CompositeImage(this.impDebug, 3);
                    ((CompositeImage)this.impDebug).resetDisplayRanges();
                }
                Overlay overlay = new Overlay();
                for (PolygonRoi r : this.roisNuclei) {
                    Roi r2 = (Roi)r.clone();
                    r2.setStrokeColor(Color.RED);
                    overlay.add(r2);
                }
                this.impDebug.setOverlay(overlay);
                new ContrastEnhancer().stretchHistogram(this.impDebug, 0.04);
                this.impDebug.show();
            }
            this.lastRunCompleted = true;
        }

        private static PolygonRoi smoothPolygonRoi(PolygonRoi r) {
            FloatPolygon poly = r.getFloatPolygon();
            FloatPolygon poly2 = new FloatPolygon();
            int nPoints = poly.npoints;
            for (int i = 0; i < nPoints; i += 2) {
                int iMinus = (i + nPoints - 1) % nPoints;
                int iPlus = (i + 1) % nPoints;
                poly2.addPoint((poly.xpoints[iMinus] + poly.xpoints[iPlus] + poly.xpoints[i]) / 3.0f, (poly.ypoints[iMinus] + poly.ypoints[iPlus] + poly.ypoints[i]) / 3.0f);
            }
            return new PolygonRoi(poly2, 2);
        }

        public List<PathObject> getPathObjects() {
            return this.pathObjects;
        }

        public void runDetection(double backgroundRadius, double maxBackground, double medianRadius, double sigma, double threshold, double minArea, double maxArea, boolean mergeAll, boolean watershedPostProcess, boolean excludeDAB, double cellExpansion, boolean smoothBoundaries, boolean includeNuclei, boolean makeMeasurements, int z, int t, boolean backgroundByReconstruction) {
            boolean updateAnything;
            boolean updateNucleusROIs;
            boolean bl = updateNucleusROIs = this.rois == null || this.bpLoG == null;
            updateNucleusROIs = updateNucleusROIs ? updateNucleusROIs : this.medianRadius != medianRadius;
            this.medianRadius = medianRadius;
            updateNucleusROIs = updateNucleusROIs ? updateNucleusROIs : this.t != t || this.z != z;
            this.z = z;
            this.t = t;
            updateNucleusROIs = updateNucleusROIs ? updateNucleusROIs : this.backgroundRadius != backgroundRadius;
            this.backgroundRadius = backgroundRadius;
            updateNucleusROIs = updateNucleusROIs ? updateNucleusROIs : this.sigma != sigma;
            this.sigma = sigma;
            updateNucleusROIs = updateNucleusROIs ? updateNucleusROIs : this.excludeDAB != excludeDAB;
            this.excludeDAB = excludeDAB;
            boolean bl2 = updateAnything = updateNucleusROIs || !this.lastRunCompleted;
            updateAnything = updateAnything ? updateAnything : this.minArea != minArea;
            this.minArea = minArea;
            updateAnything = updateAnything ? updateAnything : this.maxArea != maxArea;
            this.maxArea = maxArea;
            updateAnything = updateAnything ? updateAnything : this.maxBackground != maxBackground;
            this.maxBackground = maxBackground;
            updateAnything = updateAnything ? updateAnything : this.threshold != threshold;
            this.threshold = threshold;
            updateAnything = updateAnything ? updateAnything : this.mergeAll != mergeAll;
            this.mergeAll = mergeAll;
            updateAnything = updateAnything ? updateAnything : this.watershedPostProcess != watershedPostProcess;
            this.watershedPostProcess = watershedPostProcess;
            updateAnything = updateAnything ? updateAnything : this.cellExpansion != cellExpansion;
            this.cellExpansion = cellExpansion;
            updateAnything = updateAnything ? updateAnything : this.smoothBoundaries != smoothBoundaries;
            this.smoothBoundaries = smoothBoundaries;
            updateAnything = updateAnything ? updateAnything : this.includeNuclei != includeNuclei;
            this.includeNuclei = includeNuclei;
            updateAnything = updateAnything ? updateAnything : this.makeMeasurements != makeMeasurements;
            this.makeMeasurements = makeMeasurements;
            updateAnything = updateAnything ? updateAnything : this.backgroundByReconstruction != backgroundByReconstruction;
            this.backgroundByReconstruction = backgroundByReconstruction;
            this.doDetection(updateNucleusROIs);
        }
    }
}

