/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui.viewer;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.Reader;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javafx.application.Platform;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.event.EventHandler;
import javafx.event.EventType;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.ButtonType;
import javafx.scene.control.Dialog;
import javafx.scene.input.DragEvent;
import javafx.scene.input.Dragboard;
import javafx.scene.input.TransferMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.ext.extensionmanager.gui.ExtensionManager;
import qupath.fx.dialogs.Dialogs;
import qupath.lib.common.GeneralTools;
import qupath.lib.gui.FileCopier;
import qupath.lib.gui.QuPathGUI;
import qupath.lib.gui.commands.Commands;
import qupath.lib.gui.commands.InteractiveObjectImporter;
import qupath.lib.gui.commands.ProjectCommands;
import qupath.lib.gui.localization.QuPathResources;
import qupath.lib.gui.prefs.PathPrefs;
import qupath.lib.gui.prefs.QuPathStyleManager;
import qupath.lib.gui.scripting.DefaultScriptEditor;
import qupath.lib.gui.scripting.ScriptEditor;
import qupath.lib.gui.tma.TMADataIO;
import qupath.lib.gui.viewer.QuPathViewer;
import qupath.lib.images.ImageData;
import qupath.lib.io.GsonTools;
import qupath.lib.io.PathIO;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.objects.hierarchy.TMAGrid;
import qupath.lib.projects.Project;
import qupath.lib.projects.ProjectIO;
import qupath.lib.projects.Projects;

public class DragDropImportListener
implements EventHandler<DragEvent> {
    private static final Logger logger = LoggerFactory.getLogger(DragDropImportListener.class);
    private static final Pattern GITHUB_BASE_PATTERN = Pattern.compile("https://github.com/.*");
    private final QuPathGUI qupath;
    private List<DropHandler<File>> dropHandlers = new ArrayList<DropHandler<File>>();
    private List<DropHandler<JsonElement>> jsonDropHandlers = new ArrayList<DropHandler<JsonElement>>();
    private boolean taskRunning = false;
    private Timer timer = new Timer("drag-drop-timer", true);

    public DragDropImportListener(QuPathGUI qupath) {
        this.qupath = qupath;
    }

    public void setupTarget(Node target) {
        target.setOnDragOver((EventHandler)this);
        target.setOnDragDropped((EventHandler)this);
        target.setOnDragDone((EventHandler)this);
    }

    public void setupTarget(Scene target) {
        target.setOnDragOver((EventHandler)this);
        target.setOnDragDropped((EventHandler)this);
        target.setOnDragDone((EventHandler)this);
    }

    public void handle(DragEvent event) {
        Dragboard dragboard;
        if (this.taskRunning) {
            return;
        }
        EventType type = event.getEventType();
        if (type == DragEvent.DRAG_DONE) {
            logger.debug("Drag-drop done");
            return;
        }
        if (type == DragEvent.DRAG_OVER) {
            dragboard = event.getDragboard();
            if (dragboard.hasFiles() || dragboard.hasUrl() || dragboard.hasString()) {
                event.acceptTransferModes(new TransferMode[]{TransferMode.COPY});
                event.consume();
                return;
            }
        } else if (type != DragEvent.DRAG_DROPPED) {
            logger.warn("Unexpected drag-drop event {}", (Object)event);
            return;
        }
        dragboard = event.getDragboard();
        Object source = event.getSource();
        QuPathViewer viewer = null;
        for (QuPathViewer viewer2 : this.qupath.getAllViewers()) {
            if (viewer2.getView() != source) continue;
            viewer = viewer2;
            break;
        }
        long delay = 0L;
        if (event.getGestureSource() == null) {
            delay = 50L;
            logger.debug("Setting drag-drop delay to {} ms", (Object)delay);
        }
        if (viewer == null && this.qupath.getAllViewers().size() == 1) {
            viewer = this.qupath.getViewer();
        }
        ArrayList files = dragboard.hasFiles() ? new ArrayList(dragboard.getFiles()) : null;
        String url = dragboard.getUrl();
        String string = dragboard.getString();
        QuPathViewer viewer2 = viewer;
        if (files != null || url != null || string != null) {
            this.invokeLater(() -> {
                this.taskRunning = true;
                try {
                    if (files != null) {
                        logger.debug("Files dragged onto {}", source);
                        this.handleFileDrop(viewer2, files);
                    } else if (url != null) {
                        logger.debug("URL {} dragged onto {}", (Object)url, source);
                        this.handleURLDrop(viewer2, url);
                    } else if (string != null) {
                        logger.debug("Text dragged onto {}, treating as a URL", source);
                        this.handleURLDrop(viewer2, string);
                    }
                }
                catch (IOException | InterruptedException | URISyntaxException e) {
                    Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop"), (Throwable)e);
                    logger.error(e.getMessage(), (Throwable)e);
                }
                finally {
                    this.taskRunning = false;
                }
            }, delay);
            event.setDropCompleted(true);
        } else {
            event.setDropCompleted(false);
        }
        event.consume();
    }

    void invokeLater(final Runnable runnable, long millis) {
        assert (Platform.isAccessibilityActive());
        if (millis <= 0L) {
            runnable.run();
        } else {
            TimerTask task = new TimerTask(this){

                @Override
                public void run() {
                    Platform.runLater((Runnable)runnable);
                }
            };
            this.timer.schedule(task, millis);
        }
    }

    public void addFileDropHandler(DropHandler<File> handler) {
        this.dropHandlers.add(handler);
    }

    public void removeFileDropHandler(DropHandler<File> handler) {
        this.dropHandlers.remove(handler);
    }

    public void addJsonDropHandler(DropHandler<JsonElement> handler) {
        this.jsonDropHandlers.add(handler);
    }

    public void removeJsonDropHandler(DropHandler<JsonElement> handler) {
        this.jsonDropHandlers.remove(handler);
    }

    void handleFileDrop(QuPathViewer viewer, List<File> list) throws IOException {
        try {
            this.handleFileDropImpl(viewer, list);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Throwable e) {
            throw new IOException(e);
        }
    }

    void handleURLDrop(QuPathViewer viewer, String url) throws IOException, URISyntaxException, InterruptedException {
        try {
            this.qupath.openImage(viewer, url, false, false);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Throwable e) {
            throw new IOException(e);
        }
    }

    private void handleFileDropImpl(QuPathViewer viewer, List<File> list) throws IOException {
        String[] stringArray;
        boolean maybeZarr;
        if (list == null || list.isEmpty()) {
            logger.warn("No files given!");
            return;
        }
        int nJars = 0;
        int nCss = 0;
        int nJson = 0;
        for (File file : list) {
            String ext = GeneralTools.getExtension((File)file.getCanonicalFile()).orElse("").toLowerCase();
            if (ext.equals(".jar")) {
                ++nJars;
                continue;
            }
            if (ext.equals(".css")) {
                ++nCss;
                continue;
            }
            if (!ext.equals(".json")) continue;
            ++nJson;
        }
        if (nJars == list.size()) {
            ExtensionManager.promptToCopyFilesToExtensionDirectory(list, (ReadOnlyObjectProperty)QuPathGUI.getExtensionCatalogManager().getExtensionDirectoryPath(), () -> Commands.requestUserDirectory(true));
            return;
        }
        if ((list = this.handlePropertiesFiles(list)).isEmpty()) {
            return;
        }
        if (nCss == list.size()) {
            QuPathStyleManager.installStyles(list);
            return;
        }
        if (nJson == list.size()) {
            ArrayList<JsonElement> elements = new ArrayList<JsonElement>();
            Gson gson = GsonTools.getInstance();
            for (File file : list) {
                try {
                    BufferedReader reader = Files.newBufferedReader(file.toPath());
                    try {
                        elements.add((JsonElement)gson.fromJson((Reader)reader, JsonElement.class));
                    }
                    finally {
                        if (reader == null) continue;
                        reader.close();
                    }
                }
                catch (IOException ex) {
                    logger.error("Could not read JSON file {}", (Object)file, (Object)ex);
                }
            }
            for (DropHandler<JsonElement> handler : this.jsonDropHandlers) {
                if (!handler.handleDrop(viewer, elements)) continue;
                return;
            }
        }
        ImageData<BufferedImage> imageData = viewer == null ? null : viewer.getImageData();
        PathObjectHierarchy hierarchy = imageData == null ? null : imageData.getHierarchy();
        boolean singleFile = list.size() == 1;
        Set allUnzippedExtensions = list.stream().flatMap(f -> {
            try {
                return PathIO.unzippedExtensions((Path)f.getCanonicalFile().toPath(), (String[])new String[0]).stream();
            }
            catch (IOException e) {
                logger.debug(e.getLocalizedMessage(), (Throwable)e);
                return Arrays.stream(new String[0]);
            }
        }).collect(Collectors.toSet());
        File file = list.getFirst();
        String fileName = file.getCanonicalFile().getName().toLowerCase();
        if (singleFile && fileName.endsWith(PathPrefs.getSerializationExtension())) {
            if (viewer == null) {
                Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop.loadData"), (String)QuPathResources.getString("DragDrop.loadMessage"));
                return;
            }
            try {
                if (imageData != null) {
                    Dialog dialog = new Dialog();
                    ButtonType btOpen = new ButtonType(QuPathResources.getString("DragDrop.openImage"));
                    ButtonType buttonType = new ButtonType(QuPathResources.getString("DragDrop.importObjects"));
                    dialog.getDialogPane().getButtonTypes().setAll((Object[])new ButtonType[]{btOpen, buttonType, ButtonType.CANCEL});
                    dialog.setTitle(QuPathResources.getString("DragDrop.openData"));
                    dialog.setHeaderText(QuPathResources.getString("DragDrop.chooseData"));
                    dialog.setContentText(QuPathResources.getString("DragDrop.chooseDataOptions"));
                    ButtonType choice = dialog.showAndWait().orElse(ButtonType.CANCEL);
                    if (choice == ButtonType.CANCEL) {
                        return;
                    }
                    if (choice == buttonType) {
                        InteractiveObjectImporter.promptToImportObjectsFromFile(imageData, file);
                        return;
                    }
                }
                this.qupath.openSavedData(viewer, file, false, true);
            }
            catch (Exception e) {
                Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop.loadData"), (Throwable)e);
                logger.error(e.getMessage(), (Throwable)e);
            }
            return;
        }
        boolean bl = maybeZarr = file.isDirectory() && file.getName().toLowerCase().endsWith(".zarr");
        if (singleFile && file.isDirectory() && !maybeZarr) {
            List<File> list2;
            File[] filesInDirectory = file.listFiles(f -> !f.isHidden());
            if (filesInDirectory == null) {
                logger.warn("Could not list files in directory {}", (Object)file);
                filesInDirectory = new File[]{};
            }
            if ((list2 = Arrays.stream(filesInDirectory).filter(f -> f.isFile() && f.getAbsolutePath().toLowerCase().endsWith(ProjectIO.getProjectExtension())).toList()).size() == 1) {
                file = list2.getFirst();
                fileName = file.getName().toLowerCase();
                logger.warn("Selecting project file {}", (Object)file);
            } else if (list2.size() > 1) {
                logger.debug("Multiple project files found in directory {}", (Object)file);
                Object[] fileNames = (String[])list2.stream().map(File::getName).toArray(String[]::new);
                String selectedName = (String)Dialogs.showChoiceDialog((String)QuPathResources.getString("DragDrop.selectProject"), (String)QuPathResources.getString("DragDrop.selectProjectToOpen"), (Object[])fileNames, (Object)fileNames[0]);
                if (selectedName == null) {
                    return;
                }
                file = new File(file, selectedName);
                fileName = file.getName().toLowerCase();
            } else if (filesInDirectory.length == 0) {
                if (Dialogs.showYesNoDialog((String)QuPathResources.getString("DragDrop.createProject"), (String)QuPathResources.getString("DragDrop.createProjectForEmptyDirectory"))) {
                    Project project = Projects.createProject((File)file, BufferedImage.class);
                    this.qupath.setProject((Project<BufferedImage>)project);
                    if (!project.isEmpty()) {
                        project.syncChanges();
                    }
                    return;
                }
                return;
            }
        }
        if (singleFile && fileName.endsWith(ProjectIO.getProjectExtension())) {
            try {
                Project project = ProjectIO.loadProject((File)file, BufferedImage.class);
                this.qupath.setProject((Project<BufferedImage>)project);
            }
            catch (Exception e) {
                logger.error("Could not open as project file - opening in the Script Editor instead", (Throwable)e);
                this.qupath.getScriptEditor().showScript(file);
            }
            return;
        }
        if (PathIO.getObjectFileExtensions((boolean)false).containsAll(allUnzippedExtensions)) {
            if (imageData == null || hierarchy == null) {
                this.qupath.getScriptEditor().showScript(file);
                logger.info("Opening the dragged file in the Script Editor as there is no currently opened image in the viewer");
                return;
            }
            InteractiveObjectImporter.promptToImportObjectsFromFile(imageData, file);
            return;
        }
        if (singleFile && fileName.endsWith(".qptma")) {
            if (hierarchy == null) {
                Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop.TMAGridImport"), (String)QuPathResources.getString("DragDrop.TMAGridImportImageError"));
            } else {
                TMAGrid tmaGrid = TMADataIO.importDearrayedTMAData(file);
                if (tmaGrid != null) {
                    if (hierarchy.isEmpty() || Dialogs.showYesNoDialog((String)QuPathResources.getString("DragDrop.TMAGridImport"), (String)QuPathResources.getString("DragDrop.TMAGridSetHierarchy"))) {
                        hierarchy.setTMAGrid(tmaGrid);
                    }
                } else {
                    Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop.TMAGrid"), (String)String.format(QuPathResources.getString("DragDrop.TMAGridParseError"), file.getName()));
                }
            }
            return;
        }
        for (DropHandler dropHandler : this.dropHandlers) {
            if (!dropHandler.handleDrop(viewer, list)) continue;
            return;
        }
        ScriptEditor scriptEditor = this.qupath.getScriptEditor();
        if (scriptEditor instanceof DefaultScriptEditor && ((DefaultScriptEditor)scriptEditor).supportsFile(file)) {
            scriptEditor.showScript(file);
            return;
        }
        if (singleFile && (file.isFile() || maybeZarr)) {
            if (viewer == null) {
                Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop.openImage"), (String)QuPathResources.getString("DragDrop.specificViewer"));
                return;
            }
            this.qupath.openImage(viewer, file.getAbsolutePath(), true, true);
            return;
        }
        if (this.qupath.getProject() != null && (stringArray = (String[])list.stream().filter(File::isFile).map(File::getAbsolutePath).toArray(String[]::new)).length > 0) {
            ProjectCommands.promptToImportImages(this.qupath, stringArray);
            return;
        }
        if (this.qupath.getProject() == null && list.size() > 1) {
            Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop"), (String)QuPathResources.getString("DragDrop.projectForMultipleFiles"));
            return;
        }
        if (list.size() > 1) {
            Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop"), (String)QuPathResources.getString("DragDrop.couldNotHandleFiles"));
        } else {
            Dialogs.showErrorMessage((String)QuPathResources.getString("DragDrop"), (String)String.format(QuPathResources.getString("DragDrop.couldNotHandleFile"), list.get(0).getName()));
        }
    }

    private List<File> handlePropertiesFiles(List<File> files) {
        List<File> propertyFiles = files.stream().filter(f -> f.getName().toLowerCase().endsWith(".properties")).toList();
        if (propertyFiles.isEmpty()) {
            return files;
        }
        new FileCopier().title("Install localization properties").relativeToUserDirectory().outputPath("localization").inputFiles(propertyFiles).doCopy();
        if (propertyFiles.size() == files.size()) {
            return Collections.emptyList();
        }
        ArrayList<File> remainingFiles = new ArrayList<File>(files);
        remainingFiles.removeAll(propertyFiles);
        return remainingFiles;
    }

    @FunctionalInterface
    public static interface DropHandler<T> {
        public boolean handleDrop(QuPathViewer var1, List<T> var2);
    }
}

