/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui.prefs;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ThreadFactory;
import javafx.application.Application;
import javafx.application.ColorScheme;
import javafx.application.Platform;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.scene.control.ButtonType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.fx.dialogs.Dialogs;
import qupath.fx.utils.FXUtils;
import qupath.lib.common.GeneralTools;
import qupath.lib.common.ThreadTools;
import qupath.lib.gui.UserDirectoryManager;
import qupath.lib.gui.commands.Commands;
import qupath.lib.gui.prefs.PathPrefs;

public class QuPathStyleManager {
    private static final Logger logger = LoggerFactory.getLogger(QuPathStyleManager.class);
    private static final String STYLESHEET_MAIN = "css/main.css";
    private static final String STYLESHEET_DARK = "css/dark.css";
    private static final StyleOption DEFAULT_LIGHT_STYLE = new JavaFXStylesheet("Modena Light", "MODENA");
    private static final StyleOption DEFAULT_DARK_STYLE = new CustomStylesheet("Modena Dark", "Darker version of JavaFX Modena stylesheet", ColorScheme.DARK, "css/dark.css");
    private static final List<String> previouslyAddedStyleSheets = new ArrayList<String>();
    private static final StyleOption DEFAULT_SYSTEM_STYLE = new SystemStylesheet(DEFAULT_LIGHT_STYLE, DEFAULT_DARK_STYLE);
    private static final ReadOnlyObjectProperty<ColorScheme> systemColorScheme = Platform.getPreferences().colorSchemeProperty();
    private static final ObservableList<StyleOption> styles = FXCollections.observableArrayList((Object[])new StyleOption[]{DEFAULT_SYSTEM_STYLE, DEFAULT_LIGHT_STYLE, DEFAULT_DARK_STYLE});
    private static final ObservableList<StyleOption> stylesUnmodifiable = FXCollections.unmodifiableObservableList(styles);
    private static final ObjectProperty<StyleOption> selectedStyle;
    private static CssStylesWatcher watcher;
    private static final ObservableList<Fonts> availableFonts;
    private static final ObjectProperty<Fonts> selectedFont;

    private static StyleOption findByName(String name) {
        return styles.stream().filter(s -> Objects.equals(s.getName(), name)).findFirst().orElse(DEFAULT_LIGHT_STYLE);
    }

    private static void updateStyle() {
        String url;
        if (!Platform.isFxApplicationThread()) {
            Platform.runLater(QuPathStyleManager::updateStyle);
            return;
        }
        StyleOption n = (StyleOption)selectedStyle.get();
        if (n != null) {
            n.setStyle();
        } else {
            Application.setUserAgentStylesheet(null);
        }
        Fonts font = (Fonts)((Object)selectedFont.get());
        if (font != null && (url = font.getURL()) != null) {
            QuPathStyleManager.addStyleSheets(url);
        }
    }

    private static Path getCssDirectory() {
        return UserDirectoryManager.getInstance().getCssStylesPath();
    }

    public static void updateAvailableStyles() {
        StyleOption previouslySelected;
        Path cssPath = QuPathStyleManager.getCssDirectory();
        if (cssPath != null) {
            if (watcher == null) {
                try {
                    watcher = new CssStylesWatcher(cssPath);
                    QuPathStyleManager.watcher.styles.addListener(c -> QuPathStyleManager.updateAvailableStyles());
                }
                catch (Exception e) {
                    logger.warn("Exception searching for css files: {}", (Object)e.getMessage(), (Object)e);
                }
            } else if (!Objects.equals(QuPathStyleManager.watcher.cssPath, cssPath)) {
                watcher.setCssPath(cssPath);
            }
        }
        StyleOption styleOption = previouslySelected = selectedStyle == null ? null : (StyleOption)selectedStyle.get();
        if (watcher == null || QuPathStyleManager.watcher.styles.isEmpty()) {
            styles.setAll((Object[])new StyleOption[]{DEFAULT_SYSTEM_STYLE, DEFAULT_LIGHT_STYLE, DEFAULT_DARK_STYLE});
        } else {
            ArrayList<StyleOption> temp = new ArrayList<StyleOption>();
            temp.add(DEFAULT_SYSTEM_STYLE);
            temp.add(DEFAULT_LIGHT_STYLE);
            temp.add(DEFAULT_DARK_STYLE);
            temp.addAll((Collection<StyleOption>)QuPathStyleManager.watcher.styles);
            styles.setAll(temp);
        }
        if (selectedStyle != null) {
            if (previouslySelected == null || !styles.contains((Object)previouslySelected)) {
                selectedStyle.set((Object)DEFAULT_LIGHT_STYLE);
            } else {
                selectedStyle.set((Object)previouslySelected);
            }
        }
    }

    public static boolean installStyles(Collection<File> list) {
        File dir = Commands.requestUserDirectory(true);
        if (dir == null) {
            return false;
        }
        Path pathCss = QuPathStyleManager.getCssDirectory();
        int nInstalled = 0;
        try {
            if (pathCss != null && !Files.exists(pathCss, new LinkOption[0]) && Files.isDirectory(pathCss.getParent(), new LinkOption[0])) {
                Files.createDirectory(pathCss, new FileAttribute[0]);
            }
            if (!Files.isDirectory(pathCss, new LinkOption[0])) {
                return false;
            }
            Boolean overwriteExisting = null;
            for (File file : list) {
                Path target;
                if (!file.getName().toLowerCase().endsWith(".css")) {
                    logger.warn("Cannot install style for {} - not a .css file!", (Object)file);
                    continue;
                }
                Path source = file.toPath();
                if (Objects.equals(source, target = pathCss.resolve(file.getName()))) {
                    logger.warn("Can't copy CSS - source and target files are the same!");
                    continue;
                }
                if (Files.exists(target, new LinkOption[0])) {
                    if (overwriteExisting == null) {
                        ButtonType response = Dialogs.showYesNoCancelDialog((String)"Install CSS", (String)"Do you want to overwrite existing CSS files?");
                        if (response == ButtonType.YES) {
                            overwriteExisting = Boolean.TRUE;
                        } else if (response == ButtonType.NO) {
                            overwriteExisting = Boolean.FALSE;
                        } else {
                            return false;
                        }
                    }
                    if (!overwriteExisting.booleanValue()) continue;
                }
                logger.info("Copying {} -> {}", (Object)source, (Object)target);
                Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING);
                ++nInstalled;
            }
        }
        catch (IOException e) {
            logger.error("Exception installing CSS files: {}", (Object)e.getLocalizedMessage(), (Object)e);
            return false;
        }
        if (nInstalled > 0) {
            QuPathStyleManager.updateAvailableStyles();
        }
        return true;
    }

    public static void refresh() {
        QuPathStyleManager.updateStyle();
    }

    public static ColorScheme getStyleColorScheme() {
        StyleOption selected = (StyleOption)selectedStyle.get();
        return selected == null ? Platform.getPreferences().getColorScheme() : selected.getColorScheme();
    }

    public static ObservableList<StyleOption> availableStylesProperty() {
        return stylesUnmodifiable;
    }

    public static ObjectProperty<StyleOption> selectedStyleProperty() {
        return selectedStyle;
    }

    public static ObservableList<Fonts> availableFontsProperty() {
        return availableFonts;
    }

    public static ObjectProperty<Fonts> fontProperty() {
        return selectedFont;
    }

    private static void setStyleSheets(String ... urls) {
        Application.setUserAgentStylesheet(null);
        QuPathStyleManager.removePreviousStyleSheets(new String[0]);
        QuPathStyleManager.addStyleSheets(STYLESHEET_MAIN);
        QuPathStyleManager.addStyleSheets(urls);
    }

    private static void removePreviousStyleSheets(String ... urls) {
        if (previouslyAddedStyleSheets.isEmpty()) {
            return;
        }
        try {
            Class<?> cStyleManager = Class.forName("com.sun.javafx.css.StyleManager");
            Object styleManager = cStyleManager.getMethod("getInstance", new Class[0]).invoke(null, new Object[0]);
            Method m = styleManager.getClass().getMethod("removeUserAgentStylesheet", String.class);
            Iterator<String> iterator = previouslyAddedStyleSheets.iterator();
            while (iterator.hasNext()) {
                String url = iterator.next();
                iterator.remove();
                m.invoke(styleManager, url);
                logger.debug("Stylesheet removed {}", (Object)url);
            }
        }
        catch (Exception e) {
            logger.error("Unable to call removeUserAgentStylesheet", (Throwable)e);
        }
    }

    private static void addStyleSheets(String ... urls) {
        try {
            Class<?> cStyleManager = Class.forName("com.sun.javafx.css.StyleManager");
            Object styleManager = cStyleManager.getMethod("getInstance", new Class[0]).invoke(null, new Object[0]);
            Method m = styleManager.getClass().getMethod("addUserAgentStylesheet", String.class);
            for (String url : urls) {
                if (previouslyAddedStyleSheets.contains(url)) continue;
                m.invoke(styleManager, url);
                previouslyAddedStyleSheets.add(url);
                logger.debug("Stylesheet added {}", (Object)url);
            }
        }
        catch (Exception e) {
            logger.error("Unable to call addUserAgentStylesheet", (Throwable)e);
        }
    }

    static {
        availableFonts = FXCollections.unmodifiableObservableList((ObservableList)FXCollections.observableArrayList((Object[])Fonts.values()));
        selectedFont = PathPrefs.createPersistentPreference("selectedFont", GeneralTools.isMac() ? Fonts.SANS_SERIF : Fonts.DEFAULT, Fonts.class);
        QuPathStyleManager.updateAvailableStyles();
        selectedStyle = PathPrefs.createPersistentPreference("qupathStylesheet", DEFAULT_SYSTEM_STYLE, StyleOption::getName, QuPathStyleManager::findByName);
        systemColorScheme.addListener((v, o, n) -> {
            if (selectedStyle.get() == DEFAULT_SYSTEM_STYLE) {
                QuPathStyleManager.updateStyle();
            }
        });
        selectedStyle.addListener((v, o, n) -> QuPathStyleManager.updateStyle());
        selectedFont.addListener((v, o, n) -> QuPathStyleManager.updateStyle());
    }

    public static interface StyleOption {
        public void setStyle();

        public String getDescription();

        public String getName();

        default public ColorScheme getColorScheme() {
            return Platform.getPreferences().getColorScheme();
        }
    }

    public static enum Fonts {
        DEFAULT,
        SANS_SERIF,
        SERIF;


        private String getURL() {
            return switch (this.ordinal()) {
                case 1 -> "css/sans-serif.css";
                case 2 -> "css/serif.css";
                default -> null;
            };
        }

        public String toString() {
            return switch (this.ordinal()) {
                case 1 -> "Sans-serif";
                case 2 -> "Serif";
                default -> "Default";
            };
        }
    }

    private static class CssStylesWatcher
    implements Runnable {
        private static final ThreadFactory THREAD_FACTORY = ThreadTools.createThreadFactory((String)"css-watcher", (boolean)true);
        private Thread thread;
        private Path cssPath;
        private WatchService watcher;
        private ObservableList<StyleOption> styles = FXCollections.observableArrayList();

        private CssStylesWatcher(Path cssPath) {
            this.thread = THREAD_FACTORY.newThread(this);
            try {
                this.watcher = FileSystems.getDefault().newWatchService();
                logger.debug("Watching for changes in {}", (Object)cssPath);
            }
            catch (IOException e) {
                logger.error("Exception setting up CSS watcher: {}", (Object)e.getMessage(), (Object)e);
            }
            this.setCssPath(cssPath);
            this.thread.start();
        }

        private void setCssPath(Path cssPath) {
            if (Objects.equals(this.cssPath, cssPath)) {
                return;
            }
            this.cssPath = cssPath;
            if (Files.isDirectory(cssPath, new LinkOption[0])) {
                try {
                    cssPath.register(this.watcher, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
                    logger.debug("Watching for changes in {}", (Object)cssPath);
                }
                catch (IOException e) {
                    logger.error("Exception setting up CSS watcher: {}", (Object)e.getMessage(), (Object)e);
                }
            }
            this.refreshStylesheets();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            while (this.watcher != null) {
                boolean valid;
                WatchKey key = null;
                CssStylesWatcher cssStylesWatcher = this;
                synchronized (cssStylesWatcher) {
                    try {
                        key = this.watcher.take();
                        if (key == null) {
                            continue;
                        }
                    }
                    catch (InterruptedException interruptedException) {
                        return;
                    }
                }
                for (WatchEvent watchEvent : key.pollEvents()) {
                    if (watchEvent.kind() == StandardWatchEventKinds.OVERFLOW) continue;
                    WatchEvent event = watchEvent;
                    Path basePath = (Path)key.watchable();
                    if (!Objects.equals(this.cssPath, basePath)) continue;
                    Path path = basePath.resolve((Path)event.context());
                    if (watchEvent.kind() == StandardWatchEventKinds.ENTRY_MODIFY && Files.isRegularFile(path, new LinkOption[0])) {
                        try {
                            StyleOption currentStyle = (StyleOption)selectedStyle.get();
                            if (!(currentStyle instanceof CustomStylesheet)) continue;
                            CustomStylesheet currentCustomStyle = (CustomStylesheet)currentStyle;
                            String url = path.toUri().toString();
                            if (!currentCustomStyle.containsUrl(url)) break;
                            logger.info("Refreshing style {}", (Object)currentStyle.getName());
                            QuPathStyleManager.refresh();
                            break;
                        }
                        catch (Exception e) {
                            logger.warn("Exception processing CSS refresh: {}", (Object)e.getMessage(), (Object)e);
                            continue;
                        }
                    }
                    this.refreshStylesheets();
                }
                if (valid = key.reset()) continue;
                break;
            }
        }

        private void refreshStylesheets() {
            try {
                if (Files.isDirectory(this.cssPath, new LinkOption[0])) {
                    List<CustomStylesheet> newStyles = Files.list(this.cssPath).filter(p -> Files.isRegularFile(p, new LinkOption[0]) && p.getFileName().toString().toLowerCase().endsWith(".css")).map(CustomStylesheet::new).sorted(Comparator.comparing(StyleOption::getName)).toList();
                    FXUtils.runOnApplicationThread(() -> this.styles.setAll((Collection)newStyles));
                    return;
                }
            }
            catch (IOException e) {
                logger.warn(e.getLocalizedMessage(), (Throwable)e);
            }
            FXUtils.runOnApplicationThread(() -> this.styles.clear());
        }
    }

    static class JavaFXStylesheet
    implements StyleOption {
        private String name;
        private String cssName;

        JavaFXStylesheet(String name, String cssName) {
            this.name = name;
            this.cssName = cssName;
        }

        @Override
        public void setStyle() {
            Application.setUserAgentStylesheet((String)this.cssName);
            QuPathStyleManager.removePreviousStyleSheets(this.cssName);
            QuPathStyleManager.addStyleSheets(QuPathStyleManager.STYLESHEET_MAIN);
        }

        @Override
        public String getDescription() {
            return "Built-in JavaFX stylesheet " + this.cssName;
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public ColorScheme getColorScheme() {
            return ColorScheme.LIGHT;
        }

        public String toString() {
            return this.getName();
        }

        public int hashCode() {
            return Objects.hash(this.cssName, this.name);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            JavaFXStylesheet other = (JavaFXStylesheet)obj;
            return Objects.equals(this.cssName, other.cssName) && Objects.equals(this.name, other.name);
        }
    }

    static class CustomStylesheet
    implements StyleOption {
        private final String name;
        private final String description;
        private final String[] urls;
        private final ColorScheme colorScheme;

        CustomStylesheet(String name, String description, ColorScheme colorScheme, String ... urls) {
            this.name = name;
            this.description = description;
            this.urls = (String[])urls.clone();
            this.colorScheme = colorScheme;
        }

        CustomStylesheet(Path path) {
            this(GeneralTools.getNameWithoutExtension((File)path.toFile()), path.toString(), null, path.toUri().toString());
        }

        @Override
        public void setStyle() {
            QuPathStyleManager.setStyleSheets(this.urls);
        }

        @Override
        public String getDescription() {
            return this.description;
        }

        @Override
        public String getName() {
            return this.name;
        }

        public String toString() {
            return this.getName();
        }

        @Override
        public ColorScheme getColorScheme() {
            if (this.colorScheme != null) {
                return this.colorScheme;
            }
            String name = this.getName().toLowerCase();
            if (name.contains("dark")) {
                return ColorScheme.DARK;
            }
            if (name.contains("light")) {
                return ColorScheme.LIGHT;
            }
            return Platform.getPreferences().getColorScheme();
        }

        private boolean containsUrl(String url) {
            for (String css : this.urls) {
                if (!Objects.equals(url, css)) continue;
                return true;
            }
            return false;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + Arrays.hashCode(this.urls);
            result = 31 * result + Objects.hash(this.description, this.name);
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            CustomStylesheet other = (CustomStylesheet)obj;
            return Objects.equals(this.description, other.description) && Objects.equals(this.name, other.name) && Arrays.equals(this.urls, other.urls);
        }
    }

    static class SystemStylesheet
    implements StyleOption {
        private final StyleOption defaultLight;
        private final StyleOption defaultDark;

        private SystemStylesheet(StyleOption defaultLight, StyleOption defaultDark) {
            this.defaultLight = defaultLight;
            this.defaultDark = defaultDark;
        }

        @Override
        public void setStyle() {
            if (Platform.getPreferences().getColorScheme() == ColorScheme.DARK) {
                this.defaultDark.setStyle();
            } else {
                this.defaultLight.setStyle();
            }
        }

        @Override
        public String getDescription() {
            return "Use a style based on the system-wide light/dark setting";
        }

        @Override
        public String getName() {
            return "System theme";
        }

        public String toString() {
            return this.getName();
        }
    }
}

