/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui.images.stores;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.Weigher;
import java.awt.Shape;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.common.ThreadTools;
import qupath.lib.gui.images.stores.ImageRegionStore;
import qupath.lib.gui.images.stores.ImageRegionStoreHelpers;
import qupath.lib.gui.images.stores.SizeEstimator;
import qupath.lib.gui.images.stores.TileListener;
import qupath.lib.gui.images.stores.TileWorker;
import qupath.lib.images.servers.GeneratingImageServer;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.RegionRequest;

abstract class AbstractImageRegionStore<T>
implements ImageRegionStore<T> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractImageRegionStore.class);
    private static final int DEFAULT_THUMBNAIL_WIDTH = 1024;
    private final List<TileWorker<T>> workers = Collections.synchronizedList(new ArrayList());
    private final Map<RegionRequest, TileWorker<T>> waitingMap = new ConcurrentHashMap<RegionRequest, TileWorker<T>>();
    private boolean clearingCache = false;
    protected List<TileListener<T>> tileListeners = Collections.synchronizedList(new ArrayList());
    protected Map<RegionRequest, T> cache;
    private int maxThumbnailSize;
    private int minThumbnailSize = 16;
    private long tileCacheSizeBytes;
    private final TileRequestManager manager = new TileRequestManager(10);
    private final ExecutorService pool = Executors.newFixedThreadPool(Math.max(8, Math.min(Runtime.getRuntime().availableProcessors() * 4, 32)), ThreadTools.createThreadFactory((String)"region-store-", (boolean)false));
    private final ExecutorService poolLocal = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors(), ThreadTools.createThreadFactory((String)"region-store-local-", (boolean)false));

    protected AbstractImageRegionStore(SizeEstimator<T> sizeEstimator, int thumbnailSize, long tileCacheSizeBytes) {
        this.maxThumbnailSize = thumbnailSize;
        this.tileCacheSizeBytes = tileCacheSizeBytes;
        Weigher weigher = (r, t) -> (int)Long.min(Integer.MAX_VALUE, sizeEstimator.getApproxImageSize(t) / 1024L);
        long maxWeight = Long.max(1L, tileCacheSizeBytes / 1024L);
        int concurrencyLevel = 1;
        Cache originalCache = CacheBuilder.newBuilder().weigher(weigher).maximumWeight(maxWeight).softValues().concurrencyLevel(concurrencyLevel).removalListener(n -> {
            if (n.getCause() == RemovalCause.COLLECTED) {
                logger.debug("Cached tile collected: {} (cache size={})", n.getKey(), (Object)this.cache.size());
            } else {
                logger.trace("Cached tile removed due to {}: {} (cache size={})", new Object[]{n.getCause(), n.getKey(), this.cache.size()});
            }
        }).build();
        this.cache = originalCache.asMap();
    }

    protected AbstractImageRegionStore(SizeEstimator<T> sizeEstimator, long tileCacheSizeBytes) {
        this(sizeEstimator, 1024, tileCacheSizeBytes);
    }

    public long getTileCacheSize() {
        return this.tileCacheSizeBytes;
    }

    double calculateThumbnailDownsample(int width, int height) {
        double maxDim = Math.max(width, height);
        double minDim = Math.min(width, height);
        if (minDim > (double)this.minThumbnailSize) {
            double maxDownsample = minDim / (double)this.minThumbnailSize;
            return Math.max(1.0, Math.min(maxDim / (double)this.maxThumbnailSize, maxDownsample));
        }
        return 1.0;
    }

    RegionRequest getThumbnailRequest(ImageServer<T> server, int zPosition, int tPosition) {
        double downsample = 1.0;
        if (AbstractImageRegionStore.isPyramidalImageServer(server)) {
            downsample = this.calculateThumbnailDownsample(server.getWidth(), server.getHeight());
        }
        downsample = Math.max(downsample, 1.0);
        return RegionRequest.createInstance((String)server.getPath(), (double)downsample, (int)0, (int)0, (int)server.getWidth(), (int)server.getHeight(), (int)zPosition, (int)tPosition);
    }

    @Override
    public T getCachedThumbnail(ImageServer<T> server, int zPosition, int tPosition) {
        RegionRequest request = this.getThumbnailRequest(server, zPosition, tPosition);
        return this.cache.get(request);
    }

    protected void registerRequest(TileListener<T> tileListener, ImageServer<T> server, Shape clipShape, double downsampleFactor, int zPosition, int tPosition) {
        this.manager.registerRequest(tileListener, server, clipShape, downsampleFactor, zPosition, tPosition);
    }

    @Override
    public void addTileListener(TileListener<T> listener) {
        this.tileListeners.add(listener);
    }

    public Map<RegionRequest, T> getCache() {
        return this.cache;
    }

    @Override
    public void removeTileListener(TileListener<T> listener) {
        this.tileListeners.remove(listener);
    }

    @Override
    public T getCachedTile(ImageServer<T> server, RegionRequest request) {
        return this.cache.get(request);
    }

    public synchronized Map<RegionRequest, T> getCachedTilesForServer(ImageServer<T> server) {
        HashMap<RegionRequest, T> tiles = new HashMap<RegionRequest, T>();
        String serverPath = server.getPath();
        for (Map.Entry<RegionRequest, T> entry : this.cache.entrySet()) {
            if (entry.getValue() == null || !entry.getKey().getPath().equals(serverPath)) continue;
            tiles.put(entry.getKey(), entry.getValue());
        }
        return tiles;
    }

    private static boolean isPyramidalImageServer(ImageServer<?> server) {
        return server.nResolutions() > 1;
    }

    protected void workerComplete(TileWorker<T> worker) {
        this.workers.remove(worker);
        this.manager.taskCompleted(worker);
        if (worker.isCancelled() || !this.stopWaiting(worker.getRequest())) {
            return;
        }
        try {
            Object imgNew = worker.get();
            if (imgNew == null) {
                return;
            }
            RegionRequest request = worker.getRequest();
            worker.getRequestedCache().put(request, imgNew);
            ArrayList<TileListener<T>> myTileListeners = new ArrayList<TileListener<T>>(this.tileListeners);
            for (TileListener tileListener : myTileListeners) {
                tileListener.tileAvailable(request.getPath(), (ImageRegion)request, imgNew);
            }
        }
        catch (InterruptedException e) {
            logger.warn("Tile request interrupted", (Throwable)e);
        }
        catch (ExecutionException e) {
            logger.warn("Tile request exception", (Throwable)e);
        }
    }

    protected T getCachedRegion(ImageServer<T> server, RegionRequest request) {
        if (server == null) {
            return null;
        }
        Object result = this.requestImageTile(server, request, this.cache, false);
        if (result != null && !(result instanceof TileWorker)) {
            Object img = result;
            return (T)img;
        }
        return null;
    }

    protected boolean stopWaiting(RegionRequest request) {
        if (this.clearingCache) {
            logger.warn("Stop waiting called while clearing cache: {}", (Object)Thread.currentThread());
            return this.waitingMap.remove(request) != null;
        }
        return this.waitingMap.remove(request) != null;
    }

    protected synchronized Object requestImageTile(ImageServer<T> server, RegionRequest request, Map<RegionRequest, T> cache, boolean ensureTileReturned) {
        T img = cache.get(request);
        if (img != null) {
            return img;
        }
        if (cache.containsKey(request)) {
            return null;
        }
        if (server.isEmptyRegion(request)) {
            return null;
        }
        TileWorker<T> worker = null;
        worker = this.waitingMap.get(request);
        if (worker != null && worker.isCancelled()) {
            this.workers.remove(worker);
            worker = null;
        }
        if (worker == null) {
            worker = this.createTileWorker(server, request, cache, ensureTileReturned);
            this.workers.add(worker);
            if (server instanceof GeneratingImageServer) {
                if (this.poolLocal.isShutdown()) {
                    return null;
                }
                this.poolLocal.execute(worker);
            } else {
                if (this.pool.isShutdown()) {
                    return null;
                }
                this.pool.execute(worker);
            }
            this.waitingMap.put(request, worker);
        }
        return worker;
    }

    protected TileWorker<T> createTileWorker(ImageServer<T> server, RegionRequest request, Map<RegionRequest, T> cache, boolean ensureTileReturned) {
        return new DefaultTileWorker(server, request, cache, ensureTileReturned);
    }

    @Override
    public synchronized T getThumbnail(ImageServer<T> server, int zPosition, int tPosition, boolean addToCache) {
        RegionRequest request = this.getThumbnailRequest(server, zPosition, tPosition);
        Object result = this.requestImageTile(server, request, this.cache, true);
        if (!(result instanceof TileWorker)) {
            return (T)result;
        }
        logger.debug("Thumbnail request for {}, ({}, {})", new Object[]{server, zPosition, tPosition});
        TileWorker worker = (TileWorker)result;
        try {
            return (T)worker.get();
        }
        catch (InterruptedException e) {
            logger.error(e.getLocalizedMessage());
        }
        catch (ExecutionException e) {
            logger.error(e.getLocalizedMessage());
        }
        try {
            logger.warn("Fallback to requesting thumbnail directly...");
            return (T)server.readRegion(request);
        }
        catch (IOException e) {
            logger.error("Unable to obtain thumbnail for {}", (Object)request, (Object)e);
            return null;
        }
    }

    public synchronized void clearCache() {
        this.clearCache(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void clearCache(boolean stopWaiting) {
        this.clearingCache = true;
        try {
            if (stopWaiting) {
                for (TileWorker worker : (TileWorker[])this.waitingMap.values().toArray(TileWorker[]::new)) {
                    worker.cancel(true);
                }
                this.waitingMap.clear();
                this.workers.clear();
            }
            this.cache.clear();
        }
        finally {
            this.clearingCache = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void clearCacheForServer(ImageServer<T> server) {
        this.clearingCache = true;
        try {
            if (!this.waitingMap.isEmpty()) {
                String serverPath = server.getPath();
                logger.trace("Waiting map size before server cache cleared: {}", (Object)this.waitingMap.size());
                Iterator<Map.Entry<RegionRequest, TileWorker<T>>> iter = this.waitingMap.entrySet().iterator();
                while (iter.hasNext()) {
                    logger.trace("Waiting map size during server clear: {}", (Object)this.waitingMap.size());
                    Map.Entry<RegionRequest, TileWorker<T>> entry = iter.next();
                    if (!serverPath.equals(entry.getKey().getPath())) continue;
                    logger.trace("Removing entry from waiting map for thread  {}", (Object)Thread.currentThread().threadId());
                    iter.remove();
                    entry.getValue().cancel(true);
                    this.workers.remove(entry.getValue());
                }
            }
            this.clearCacheForServer(this.cache, server);
        }
        finally {
            this.clearingCache = false;
        }
    }

    @Override
    public synchronized void clearCacheForRequestOverlap(RegionRequest request) {
        if (!this.waitingMap.isEmpty()) {
            Iterator<Map.Entry<RegionRequest, TileWorker<T>>> iter = this.waitingMap.entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry<RegionRequest, TileWorker<T>> entry = iter.next();
                if (!request.overlapsRequest(entry.getKey())) continue;
                iter.remove();
                entry.getValue().cancel(true);
                this.workers.remove(entry.getValue());
            }
        }
        this.clearCacheForRequestOverlap(this.cache, request);
    }

    private synchronized void clearCacheForServer(Map<RegionRequest, T> map, ImageServer<?> server) {
        String serverPath = server.getPath();
        map.entrySet().removeIf(entry -> serverPath.equals(((RegionRequest)entry.getKey()).getPath()));
    }

    private synchronized void clearCacheForRequestOverlap(Map<RegionRequest, T> map, RegionRequest request) {
        map.entrySet().removeIf(regionRequestTEntry -> request.overlapsRequest((RegionRequest)regionRequestTEntry.getKey()));
    }

    @Override
    public void close() {
        for (TileWorker<T> worker : new ArrayList<TileWorker<T>>(this.workers)) {
            worker.cancel(true);
        }
        this.pool.shutdownNow();
        this.poolLocal.shutdownNow();
        this.cache.clear();
    }

    class TileRequestManager {
        static final int MAX_Z_SEPARATION = 10;
        private List<TileRequestCollection<T>> list = new ArrayList();
        private final TileRequestComparator<T> comparator = new TileRequestComparator();
        private final int nThreads;
        private int busyThreads = 0;
        private final List<TileWorker<T>> requestedWorkers = new ArrayList();

        TileRequestManager(int nThreads) {
            this.nThreads = nThreads;
        }

        public synchronized void registerRequest(TileListener<T> tileListener, ImageServer<T> server, Shape clipShape, double downsampleFactor, int zPosition, int tPosition) {
            Iterator iter = this.list.iterator();
            while (iter.hasNext()) {
                TileRequestCollection temp = iter.next();
                if (temp.tileListener != tileListener) continue;
                if (temp.clipShape.equals(clipShape) && temp.zPosition == zPosition && temp.tPosition == tPosition) {
                    return;
                }
                iter.remove();
                break;
            }
            TileRequestCollection requestCollection = new TileRequestCollection(tileListener, server, clipShape, downsampleFactor, zPosition, tPosition, 10);
            this.list.add(requestCollection);
            Collections.sort(this.list, this.comparator);
            this.assignTasks();
        }

        public synchronized void deregisterRequest(TileListener<T> tileListener) {
            this.list.removeIf(temp -> temp.tileListener == tileListener);
        }

        synchronized void assignTasks() {
            if (this.list.isEmpty()) {
                return;
            }
            int ind = 0;
            while (this.busyThreads < this.nThreads && ind < this.list.size()) {
                TileRequestCollection temp = this.list.get(ind);
                if (!temp.hasMoreTiles()) {
                    this.list.remove(temp);
                    continue;
                }
                RegionRequest request = temp.nextTileRequest();
                if (AbstractImageRegionStore.this.cache.containsKey(request) || AbstractImageRegionStore.this.waitingMap.containsKey(request)) continue;
                TileWorker worker = AbstractImageRegionStore.this.createTileWorker(temp.server, request, AbstractImageRegionStore.this.cache, false);
                logger.trace("Adding {} to waiting map for thread {}", (Object)request, (Object)Thread.currentThread().getId());
                AbstractImageRegionStore.this.waitingMap.put(request, worker);
                if (temp.server instanceof GeneratingImageServer) {
                    if (!AbstractImageRegionStore.this.poolLocal.isShutdown()) {
                        AbstractImageRegionStore.this.poolLocal.execute(worker);
                    }
                } else if (!AbstractImageRegionStore.this.pool.isShutdown()) {
                    AbstractImageRegionStore.this.pool.execute(worker);
                }
                this.requestedWorkers.add(worker);
                ++this.busyThreads;
            }
            Collections.sort(this.list, this.comparator);
        }

        synchronized void taskCompleted(TileWorker<T> worker) {
            if (!this.requestedWorkers.remove(worker)) {
                return;
            }
            --this.busyThreads;
            logger.trace("Number of busy threads: {}", (Object)this.busyThreads);
            Collections.sort(this.list, this.comparator);
            this.assignTasks();
        }
    }

    class DefaultTileWorker
    extends FutureTask<T>
    implements TileWorker<T> {
        private final Map<RegionRequest, T> cache;
        private final RegionRequest request;

        DefaultTileWorker(final ImageServer<T> server, final RegionRequest request, final Map<RegionRequest, T> cache, final boolean ensureTileReturned) {
            super(new Callable<T>(){

                @Override
                public T call() throws Exception {
                    Object imgTile = cache.get(request);
                    if (imgTile != null) {
                        return imgTile;
                    }
                    if (ensureTileReturned) {
                        return server.readRegion(request);
                    }
                    return server.readRegion(request);
                }
            });
            this.request = request;
            this.cache = cache;
        }

        @Override
        public RegionRequest getRequest() {
            return this.request;
        }

        @Override
        public Map<RegionRequest, T> getRequestedCache() {
            return this.cache;
        }

        @Override
        public void done() {
            AbstractImageRegionStore.this.workerComplete(this);
        }
    }

    static class TileRequestComparator<T>
    implements Comparator<TileRequestCollection<T>> {
        TileRequestComparator() {
        }

        @Override
        public int compare(TileRequestCollection<T> r1, TileRequestCollection<T> r2) {
            int zDiff = r1.zSeparation - r2.zSeparation;
            if (zDiff == 0) {
                return (int)(r1.timestamp - r2.timestamp);
            }
            return zDiff;
        }
    }

    static class TileRequestCollection<T> {
        private long timestamp;
        private List<RegionRequest> tileRequests = new ArrayList<RegionRequest>();
        private int zSeparation = 0;
        private int maxZSeparation = 0;
        private TileListener<T> tileListener;
        private ImageServer<T> server;
        private Shape clipShape;
        private double downsampleFactor;
        private int zPosition;
        private int tPosition;

        TileRequestCollection(TileListener<T> tileListener, ImageServer<T> server, Shape clipShape, double downsampleFactor, int zPosition, int tPosition, int maxZSeparation) {
            this.timestamp = System.currentTimeMillis();
            this.tileListener = tileListener;
            this.server = server;
            this.clipShape = clipShape;
            this.downsampleFactor = downsampleFactor;
            this.zPosition = zPosition;
            this.tPosition = tPosition;
            this.zSeparation = 0;
            this.maxZSeparation = server == null ? 0 : Math.min(server.nZSlices() - 1, maxZSeparation);
            this.updateRequests();
        }

        void updateRequests() {
            if (this.zSeparation == 0) {
                this.updateRequestsForZ(this.zPosition, this.downsampleFactor, false);
            } else {
                if (this.zPosition - this.zSeparation >= 0) {
                    this.updateRequestsForZ(this.zPosition - this.zSeparation, this.downsampleFactor * (double)Math.max(5, this.zSeparation * 2), true);
                }
                if (this.zPosition + this.zSeparation < this.server.nZSlices()) {
                    this.updateRequestsForZ(this.zPosition + this.zSeparation, this.downsampleFactor * (double)Math.max(5, this.zSeparation * 2), true);
                }
            }
        }

        void updateRequestsForZ(int z, double downsample, boolean stopBeforeDownsample) {
            if (this.server == null) {
                return;
            }
            boolean firstLoop = true;
            double[] downsamples = this.server.getPreferredDownsamples();
            Arrays.sort(downsamples);
            for (int i = downsamples.length - 1; i >= 0; --i) {
                double d = downsamples[i];
                if (Double.isNaN(d)) continue;
                if (firstLoop || !stopBeforeDownsample || d > this.downsampleFactor) {
                    int sizeBefore = this.tileRequests.size();
                    this.tileRequests = ImageRegionStoreHelpers.getTilesToRequest(this.server, this.clipShape, downsample, z, this.tPosition, this.tileRequests);
                    int sizeAfter = this.tileRequests.size();
                    logger.trace("Requests added: {} - z separation = {}, downsample = {}", new Object[]{sizeAfter - sizeBefore, this.zSeparation, downsample});
                }
                firstLoop = false;
                if (!(d <= this.downsampleFactor)) continue;
                return;
            }
        }

        public boolean hasMoreTiles() {
            return !this.tileRequests.isEmpty();
        }

        public RegionRequest nextTileRequest() {
            int ind = this.tileRequests.size() - 1;
            assert (ind >= 0);
            RegionRequest request = this.tileRequests.remove(ind);
            if (ind == 0 && this.zSeparation < this.maxZSeparation) {
                ++this.zSeparation;
                this.updateRequests();
            }
            return request;
        }
    }
}

