/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui.commands;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;
import java.util.WeakHashMap;
import javafx.beans.property.ReadOnlyObjectWrapper;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonType;
import javafx.scene.control.ContentDisplay;
import javafx.scene.control.TableCell;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.Tooltip;
import javafx.scene.input.Clipboard;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Paint;
import javafx.scene.text.TextAlignment;
import javafx.stage.FileChooser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.fx.dialogs.Dialogs;
import qupath.fx.dialogs.FileChoosers;
import qupath.fx.utils.GridPaneUtils;
import qupath.lib.common.GeneralTools;
import qupath.lib.gui.QuPathGUI;
import qupath.lib.gui.tools.ColorToolsFX;
import qupath.lib.gui.tools.GuiTools;
import qupath.lib.images.ImageData;
import qupath.lib.io.TMAScoreImporter;
import qupath.lib.objects.PathObject;
import qupath.lib.objects.TMACoreObject;
import qupath.lib.objects.hierarchy.PathObjectHierarchy;
import qupath.lib.objects.hierarchy.TMAGrid;

class TMADataImporter {
    private static final Logger logger = LoggerFactory.getLogger(TMADataImporter.class);
    private static String TITLE = "Import TMA data";
    private static Set<QuPathGUI> installedHandlers = Collections.newSetFromMap(new WeakHashMap());

    TMADataImporter() {
    }

    public static synchronized void installDragAndDropHandler(QuPathGUI qupath) {
        if (installedHandlers.contains(qupath)) {
            logger.warn(TITLE + " file drag & drop already installed for this QuPath instance!");
            return;
        }
        installedHandlers.add(qupath);
        qupath.getDefaultDragDropListener().addFileDropHandler((viewer, list) -> {
            if (list.isEmpty()) {
                return false;
            }
            ImageData<BufferedImage> imageData = qupath.getImageData();
            if (imageData == null || imageData.getHierarchy().getTMAGrid() == null) {
                return false;
            }
            File file = (File)list.get(0);
            if (file.getName().toLowerCase().endsWith(".qpmap")) {
                try {
                    CoreInfoGrid grid = new CoreInfoGrid(imageData.getHierarchy().getTMAGrid());
                    boolean success = TMADataImporter.handleImportGrid(grid, GeneralTools.readFileAsString((String)file.getAbsolutePath()));
                    if (success) {
                        grid.synchronizeTMAGridToInfo();
                        imageData.getHierarchy().fireObjectsChangedEvent((Object)grid, (Collection)imageData.getHierarchy().getTMAGrid().getTMACoreList());
                        Dialogs.showInfoNotification((String)TITLE, (String)("TMA grid imported (" + grid.getGridWidth() + "x" + grid.getGridHeight() + ")"));
                        return true;
                    }
                }
                catch (Exception e) {
                    logger.error("Error importing TMA grid", (Throwable)e);
                }
            }
            return false;
        });
    }

    public static void importTMAData(ImageData<?> imageData) {
        if (imageData == null) {
            GuiTools.showNoImageError(TITLE);
            return;
        }
        PathObjectHierarchy hierarchy = imageData.getHierarchy();
        if (hierarchy.getTMAGrid() == null) {
            Dialogs.showErrorMessage((String)TITLE, (String)"No TMA grid has been set for the selected image!");
            return;
        }
        TMAGrid grid = hierarchy.getTMAGrid();
        CoreInfoGrid infoGrid = new CoreInfoGrid(grid);
        TableView table = new TableView();
        table.getItems().setAll(infoGrid.getRows());
        int c2 = 0;
        while (c2 < grid.getGridWidth()) {
            int col = c2++;
            TableColumn tableColumn = new TableColumn();
            tableColumn.setCellValueFactory(column -> new ReadOnlyObjectWrapper((Object)((CoreInfoRow)column.getValue()).get(col)));
            tableColumn.setCellFactory(column -> new CoreInfoTableCell());
            tableColumn.setResizable(false);
            table.getColumns().add((Object)tableColumn);
        }
        table.widthProperty().addListener(c -> {
            Pane headerRow = (Pane)table.lookup("TableHeaderRow");
            headerRow.setMaxHeight(0.0);
            headerRow.setMinHeight(0.0);
            headerRow.setPrefHeight(0.0);
            headerRow.setVisible(false);
        });
        table.getSelectionModel().setCellSelectionEnabled(true);
        BorderPane pane = new BorderPane();
        pane.setCenter((Node)table);
        Button btnImportData = new Button("Import data");
        btnImportData.setTooltip(new Tooltip("Import TMA core data from a tab-delimited or .csv file"));
        btnImportData.setOnAction(e -> {
            if (TMADataImporter.handleImportDataFromFile(infoGrid)) {
                table.refresh();
            }
        });
        Button btnPasteData = new Button("Paste data");
        btnPasteData.setTooltip(new Tooltip("Paste tab-delimited TMA core data from the clipboard"));
        btnPasteData.setOnAction(e -> {
            if (TMADataImporter.handleImportDataFromClipboard(infoGrid)) {
                table.refresh();
            }
        });
        Button btnPasteGrid = new Button("Paste grid");
        btnPasteGrid.setTooltip(new Tooltip("Paste a tab-delimited grid containing TMA core names from the clipboard"));
        btnPasteGrid.setOnAction(e -> {
            if (TMADataImporter.handlePasteGrid(infoGrid)) {
                table.refresh();
            }
        });
        Button btnLoadGrid = new Button("Import grid");
        btnLoadGrid.setTooltip(new Tooltip("Import a grid containing TMA core names from a tab-delimited or .csv file"));
        btnLoadGrid.setOnAction(e -> {
            if (TMADataImporter.handleLoadGridFromFile(infoGrid)) {
                table.refresh();
            }
        });
        GridPane buttonPane = GridPaneUtils.createColumnGridControls((Node[])new Node[]{btnImportData, btnPasteData, btnLoadGrid, btnPasteGrid});
        buttonPane.setHgap(10.0);
        buttonPane.setPadding(new Insets(5.0, 0.0, 5.0, 0.0));
        pane.setBottom((Node)buttonPane);
        if (Dialogs.builder().title(TITLE).content((Node)pane).buttons(new ButtonType[]{ButtonType.APPLY, ButtonType.CANCEL}).showAndWait().orElse(ButtonType.CANCEL) == ButtonType.APPLY) {
            infoGrid.synchronizeTMAGridToInfo();
            hierarchy.fireObjectsChangedEvent((Object)infoGrid, new ArrayList(grid.getTMACoreList()));
            return;
        }
    }

    private static boolean handleImportDataFromClipboard(TMAGrid infoGrid) {
        logger.trace("Importing TMA data from clipboard...");
        if (!Clipboard.getSystemClipboard().hasString()) {
            Dialogs.showErrorMessage((String)TITLE, (String)"No text on clipboard!");
            return false;
        }
        int nScores = TMAScoreImporter.importFromCSV((String)Clipboard.getSystemClipboard().getString(), (PathObjectHierarchy)TMADataImporter.createPseudoHierarchy(infoGrid));
        if (nScores == 1) {
            Dialogs.showMessageDialog((String)TITLE, (String)"Updated 1 core");
        } else {
            Dialogs.showMessageDialog((String)TITLE, (String)("Updated " + nScores + " cores"));
        }
        return nScores > 0;
    }

    private static FileChooser.ExtensionFilter createTextFileExtensionFilter() {
        return FileChoosers.createExtensionFilter((String)"Text file", (String[])new String[]{"*.csv", "*.txt"});
    }

    private static boolean handleImportDataFromFile(TMAGrid infoGrid) {
        logger.trace("Importing TMA data from file...");
        File file = FileChoosers.promptForFile((FileChooser.ExtensionFilter[])new FileChooser.ExtensionFilter[]{TMADataImporter.createTextFileExtensionFilter()});
        if (file == null) {
            return false;
        }
        try {
            int nScores = TMAScoreImporter.importFromCSV((File)file, (PathObjectHierarchy)TMADataImporter.createPseudoHierarchy(infoGrid));
            if (nScores == 1) {
                Dialogs.showMessageDialog((String)TITLE, (String)"Updated 1 core");
            } else {
                Dialogs.showMessageDialog((String)TITLE, (String)("Updated " + nScores + " cores"));
            }
            return nScores > 0;
        }
        catch (IOException e) {
            Dialogs.showErrorMessage((String)TITLE, (String)e.getLocalizedMessage());
            return false;
        }
    }

    private static PathObjectHierarchy createPseudoHierarchy(TMAGrid grid) {
        PathObjectHierarchy hierarchy = new PathObjectHierarchy();
        hierarchy.setTMAGrid(grid);
        return hierarchy;
    }

    private static boolean handlePasteGrid(TMAGrid infoGrid) {
        logger.trace("Importing TMA grid from clipboard...");
        if (!Clipboard.getSystemClipboard().hasString()) {
            Dialogs.showErrorMessage((String)TITLE, (String)"No text on clipboard!");
            return false;
        }
        return TMADataImporter.handleImportGrid(infoGrid, Clipboard.getSystemClipboard().getString());
    }

    private static boolean handleLoadGridFromFile(TMAGrid infoGrid) {
        logger.trace("Importing TMA grid from file...");
        File file = FileChoosers.promptForFile((FileChooser.ExtensionFilter[])new FileChooser.ExtensionFilter[]{TMADataImporter.createTextFileExtensionFilter()});
        if (file == null) {
            return false;
        }
        try {
            Scanner scanner = new Scanner(file);
            scanner.useDelimiter("\\Z");
            String text = scanner.next();
            scanner.close();
            return TMADataImporter.handleImportGrid(infoGrid, text);
        }
        catch (FileNotFoundException e) {
            Dialogs.showErrorMessage((String)TITLE, (String)("File " + file.getName() + " could not be read"));
            return false;
        }
    }

    private static boolean handleImportGrid(TMAGrid infoGrid, String text) {
        int ri;
        ArrayList<String[]> rows = new ArrayList<String[]>();
        int nCols = -1;
        for (String row : GeneralTools.splitLines((String)text)) {
            String[] cols;
            if (row.contains("\t")) {
                cols = row.split("\t");
            } else {
                if (!row.contains(",")) break;
                cols = row.split(",");
            }
            if (cols.length > nCols) {
                nCols = cols.length;
            }
            rows.add(cols);
        }
        if (nCols < 0) {
            Dialogs.showErrorMessage((String)TITLE, (String)"Could not identify tab or comma delimited columns");
            return false;
        }
        int nRows = rows.size();
        if (!(nRows == infoGrid.getGridHeight() && nCols == infoGrid.getGridWidth() || nRows == infoGrid.getGridHeight() + 1 && nCols == infoGrid.getGridWidth() + 1)) {
            Dialogs.showErrorMessage((String)TITLE, (String)String.format("Grid sizes inconsistent: TMA grid is %d x %d, but text grid is %d x %d", infoGrid.getGridHeight(), infoGrid.getGridWidth(), nRows, nCols));
            return false;
        }
        boolean hasHeaders = nRows == infoGrid.getGridHeight() + 1;
        int n = ri = hasHeaders ? 1 : 0;
        while (ri < rows.size()) {
            int ci;
            String[] cols = (String[])rows.get(ri);
            int r = hasHeaders ? ri - 1 : ri;
            int n2 = ci = hasHeaders ? 1 : 0;
            while (ci < nCols) {
                int c;
                int n3 = c = hasHeaders ? ci - 1 : ci;
                if (ci >= cols.length) {
                    infoGrid.getTMACore(r, c).setCaseID(null);
                } else {
                    String id = cols[ci];
                    if (id.trim().length() == 0) {
                        id = null;
                    }
                    infoGrid.getTMACore(r, c).setCaseID(id);
                }
                if (hasHeaders) {
                    String header = ((String[])rows.get(ri))[0] + "-" + ((String[])rows.get(0))[ci];
                    infoGrid.getTMACore(r, c).setName(header);
                }
                ++ci;
            }
            ++ri;
        }
        return true;
    }

    static String getDisplayString(TMACoreObject core) {
        StringBuilder sb = new StringBuilder();
        sb.append(core.getName()).append("\n");
        if (core.getCaseID() != null) {
            sb.append(core.getCaseID());
        }
        sb.append("\n");
        sb.append("\n");
        return sb.toString();
    }

    static String getExtendedDescription(TMACoreObject core) {
        StringBuilder sb = new StringBuilder();
        sb.append("Name:\t");
        sb.append(core.getName());
        if (core.isMissing()) {
            sb.append(" (missing)\n");
        }
        sb.append("\n");
        sb.append("\n");
        for (Map.Entry entry : core.getMetadata().entrySet()) {
            sb.append((String)entry.getKey()).append("\t").append((String)entry.getValue()).append("\n");
        }
        for (String name : core.getMeasurementList().getNames()) {
            sb.append(name).append("\t").append(core.getMeasurementList().get(name)).append("\n");
        }
        return sb.toString();
    }

    private static class CoreInfoGrid
    implements TMAGrid {
        private static final long serialVersionUID = 1L;
        private TMAGrid grid;
        private Map<TMACoreObject, CoreInfo> coreMap = new LinkedHashMap<TMACoreObject, CoreInfo>();
        private List<CoreInfoRow> rows = new ArrayList<CoreInfoRow>();

        CoreInfoGrid(TMAGrid grid) {
            this.grid = grid;
            for (int y = 0; y < this.getGridHeight(); ++y) {
                CoreInfoRow row = new CoreInfoRow(this.getGridWidth());
                for (int x = 0; x < this.getGridWidth(); ++x) {
                    TMACoreObject core = grid.getTMACore(y, x);
                    CoreInfo coreInfo = new CoreInfo(core);
                    row.set(coreInfo, x);
                    this.coreMap.put(core, coreInfo);
                }
                this.rows.add(row);
            }
        }

        public List<CoreInfoRow> getRows() {
            return Collections.unmodifiableList(this.rows);
        }

        public void synchronizeTMAGridToInfo() {
            this.coreMap.values().forEach(c -> c.synchronizeCoreToFields());
        }

        public int nCores() {
            return this.grid.nCores();
        }

        public int getGridWidth() {
            return this.grid.getGridWidth();
        }

        public int getGridHeight() {
            return this.grid.getGridHeight();
        }

        public TMACoreObject getTMACore(String coreName) {
            return this.coreMap.get(this.grid.getTMACore(coreName));
        }

        public TMACoreObject getTMACore(int row, int col) {
            return this.coreMap.get(this.grid.getTMACore(row, col));
        }

        public List<TMACoreObject> getTMACoreList() {
            return new ArrayList<TMACoreObject>(this.coreMap.values());
        }
    }

    static class CoreInfoTableCell
    extends TableCell<CoreInfoRow, TMACoreObject> {
        private Tooltip tooltip = new Tooltip();

        CoreInfoTableCell() {
        }

        public void updateItem(TMACoreObject item, boolean empty) {
            super.updateItem((Object)item, empty);
            this.setWidth(150.0);
            this.setHeight(150.0);
            this.setMaxWidth(200.0);
            this.setMaxHeight(200.0);
            if (item == null || empty) {
                this.setText(null);
                this.setGraphic(null);
                this.setTooltip(null);
                return;
            }
            this.setTextFill((Paint)ColorToolsFX.getDisplayedColor((PathObject)item));
            this.setAlignment(Pos.CENTER);
            this.setTextAlignment(TextAlignment.CENTER);
            this.setContentDisplay(ContentDisplay.CENTER);
            this.setText(TMADataImporter.getDisplayString(item));
            this.tooltip.setText(TMADataImporter.getExtendedDescription(item));
            this.setTooltip(this.tooltip);
        }
    }

    static class CoreInfoRow {
        private CoreInfo[] list;

        CoreInfoRow(int capacity) {
            this.list = new CoreInfo[capacity];
        }

        public CoreInfo get(int col) {
            return this.list[col];
        }

        private void set(CoreInfo info, int col) {
            this.list[col] = info;
        }
    }

    static class CoreInfo
    extends TMACoreObject {
        private TMACoreObject core;

        public CoreInfo(TMACoreObject core) {
            this.core = core;
            this.synchronizeFieldsToCore();
        }

        private void synchronizeCoreToFields() {
            this.core.setMissing(this.isMissing());
            this.core.setName(this.getName());
            this.core.setCaseID(this.getCaseID());
            for (String string : this.getMeasurementList().getNames()) {
                this.core.getMeasurementList().put(string, this.getMeasurementList().get(string));
            }
            this.core.getMeasurementList().close();
            for (Map.Entry entry : this.getMetadata().entrySet()) {
                this.core.putMetadataValue((String)entry.getKey(), (String)entry.getValue());
            }
        }

        private void synchronizeFieldsToCore() {
            this.setMissing(this.core.isMissing());
            this.setName(this.core.getName());
            this.setCaseID(this.core.getCaseID());
        }
    }
}

