/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui.commands;

import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.awt.image.ImagingOpException;
import java.awt.image.Raster;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.geometry.Rectangle2D;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.control.Button;
import javafx.scene.control.ButtonType;
import javafx.scene.control.Label;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.TitledPane;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.GridPane;
import javafx.scene.paint.Color;
import javafx.scene.paint.Paint;
import org.apache.commons.math3.stat.correlation.PearsonsCorrelation;
import org.apache.commons.math3.stat.correlation.SpearmansCorrelation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.fx.dialogs.Dialogs;
import qupath.lib.analysis.algorithms.EstimateStainVectors;
import qupath.lib.awt.common.BufferedImageTools;
import qupath.lib.color.ColorDeconvolutionHelper;
import qupath.lib.color.ColorDeconvolutionStains;
import qupath.lib.color.ColorToolsAwt;
import qupath.lib.color.StainVector;
import qupath.lib.common.ColorTools;
import qupath.lib.common.GeneralTools;
import qupath.lib.gui.dialogs.ParameterPanelFX;
import qupath.lib.gui.tools.ColorToolsFX;
import qupath.lib.gui.tools.GuiTools;
import qupath.lib.images.ImageData;
import qupath.lib.objects.PathObject;
import qupath.lib.plugins.parameters.ParameterList;
import qupath.lib.regions.ImagePlane;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.ROIs;
import qupath.lib.roi.interfaces.ROI;

class EstimateStainVectorsCommand {
    private static final Logger logger = LoggerFactory.getLogger(EstimateStainVectorsCommand.class);
    static int MAX_PIXELS = 16000000;
    private static final String TITLE = "Estimate stain vectors";

    EstimateStainVectorsCommand() {
    }

    public static void promptToEstimateStainVectors(ImageData<BufferedImage> imageData) {
        double bMax;
        double gMax;
        double rMax;
        ROI roi;
        if (imageData == null) {
            GuiTools.showNoImageError(TITLE);
            return;
        }
        if (!imageData.isBrightfield() || imageData.getServer() == null || imageData.getServer().nChannels() != 3) {
            Dialogs.showErrorMessage((String)TITLE, (String)"No brightfield, 3-channel image selected!");
            return;
        }
        ColorDeconvolutionStains stains = imageData.getColorDeconvolutionStains();
        if (stains == null || !stains.getStain(3).isResidual()) {
            Dialogs.showErrorMessage((String)TITLE, (String)"Sorry, stain estimation is only possible for brightfield, 3-channel images with 2 stains");
            return;
        }
        PathObject pathObject = imageData.getHierarchy().getSelectionModel().getSelectedObject();
        ROI rOI = roi = pathObject == null ? null : pathObject.getROI();
        if (roi == null) {
            roi = ROIs.createRectangleROI((double)0.0, (double)0.0, (double)imageData.getServer().getWidth(), (double)imageData.getServer().getHeight(), (ImagePlane)ImagePlane.getDefaultPlane());
        }
        double downsample = Math.max(1.0, Math.sqrt(roi.getBoundsWidth() * roi.getBoundsHeight() / (double)MAX_PIXELS));
        RegionRequest request = RegionRequest.createInstance((String)imageData.getServerPath(), (double)downsample, (ROI)roi);
        BufferedImage img = null;
        try {
            img = (BufferedImage)imageData.getServer().readRegion(request);
        }
        catch (IOException e) {
            Dialogs.showErrorMessage((String)TITLE, (Throwable)e);
            logger.error("Unable to obtain pixels for {}", (Object)request, (Object)e);
            return;
        }
        try {
            img = EstimateStainVectors.smoothImage((BufferedImage)img);
        }
        catch (ImagingOpException e) {
            logger.warn("Unable to convolve image - will attempt stain estimation without smoothing");
            logger.debug(e.getMessage(), (Throwable)e);
        }
        if (BufferedImageTools.is8bitColorType((int)img.getType())) {
            int[] rgb = img.getRGB(0, 0, img.getWidth(), img.getHeight(), null, 0, img.getWidth());
            int[] rgbMode = EstimateStainVectors.getModeRGB((int[])rgb);
            rMax = rgbMode[0];
            gMax = rgbMode[1];
            bMax = rgbMode[2];
        } else {
            rMax = EstimateStainVectors.getMode((float[])ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)0), (int)256);
            gMax = EstimateStainVectors.getMode((float[])ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)1), (int)256);
            bMax = EstimateStainVectors.getMode((float[])ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)2), (int)256);
        }
        if (rMax != stains.getMaxRed() || gMax != stains.getMaxGreen() || bMax != stains.getMaxBlue()) {
            ButtonType response = Dialogs.showYesNoCancelDialog((String)TITLE, (String)String.format("Modal RGB values %s, %s, %s do not match current background values.\nDo you want to use the modal values?", GeneralTools.formatNumber((double)rMax, (int)2), GeneralTools.formatNumber((double)gMax, (int)2), GeneralTools.formatNumber((double)bMax, (int)2)));
            if (response == ButtonType.CANCEL) {
                return;
            }
            if (response == ButtonType.YES) {
                stains = stains.changeMaxValues(rMax, gMax, bMax);
                imageData.setColorDeconvolutionStains(stains);
            }
        }
        ColorDeconvolutionStains stainsUpdated = null;
        logger.info("Requesting region for stain vector estimation: {}", (Object)request);
        try {
            stainsUpdated = EstimateStainVectorsCommand.showStainEditor(img, stains);
        }
        catch (Exception e) {
            Dialogs.showErrorMessage((String)TITLE, (String)("Error with stain estimation: " + e.getLocalizedMessage()));
            logger.error(e.getMessage(), (Throwable)e);
            return;
        }
        if (!stains.equals((Object)stainsUpdated)) {
            String collectiveNameBefore = stainsUpdated.getName();
            Object suggestedName = collectiveNameBefore.endsWith("default") ? collectiveNameBefore.substring(0, collectiveNameBefore.lastIndexOf("default")) + "estimated" : collectiveNameBefore;
            String newName = Dialogs.showInputDialog((String)TITLE, (String)"Set name for stain vectors", (String)suggestedName);
            if (newName == null) {
                return;
            }
            if (!newName.isBlank()) {
                stainsUpdated = stainsUpdated.changeName(newName);
            }
            imageData.setColorDeconvolutionStains(stainsUpdated);
        }
    }

    public static ColorDeconvolutionStains showStainEditor(BufferedImage img, ColorDeconvolutionStains stains) {
        float[] blue;
        float[] green;
        float[] red;
        int[] rgb;
        if (BufferedImageTools.is8bitColorType((int)img.getType())) {
            int[] buf = img.getRGB(0, 0, img.getWidth(), img.getHeight(), null, 0, img.getWidth());
            rgb = EstimateStainVectors.subsample((int[])buf, (int)10000);
            red = ColorDeconvolutionHelper.getRedOpticalDensities((int[])rgb, (double)stains.getMaxRed(), null);
            green = ColorDeconvolutionHelper.getGreenOpticalDensities((int[])rgb, (double)stains.getMaxGreen(), null);
            blue = ColorDeconvolutionHelper.getBlueOpticalDensities((int[])rgb, (double)stains.getMaxBlue(), null);
        } else {
            red = ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)0, null);
            green = ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)1, null);
            blue = ColorDeconvolutionHelper.getPixels((Raster)img.getRaster(), (int)2, null);
            int n = red.length;
            rgb = new int[n];
            for (int i = 0; i < n; ++i) {
                int r = (int)GeneralTools.clipValue((double)((double)red[i] / stains.getMaxRed() * 255.0), (double)0.0, (double)255.0);
                int g = (int)GeneralTools.clipValue((double)((double)green[i] / stains.getMaxGreen() * 255.0), (double)0.0, (double)255.0);
                int b = (int)GeneralTools.clipValue((double)((double)blue[i] / stains.getMaxBlue() * 255.0), (double)0.0, (double)255.0);
                rgb[i] = ColorTools.packRGB((int)r, (int)g, (int)b);
            }
            ColorDeconvolutionHelper.convertToOpticalDensity((float[])red, (double)stains.getMaxRed());
            ColorDeconvolutionHelper.convertToOpticalDensity((float[])green, (double)stains.getMaxGreen());
            ColorDeconvolutionHelper.convertToOpticalDensity((float[])blue, (double)stains.getMaxBlue());
        }
        StainsWrapper stainsWrapper = new StainsWrapper(stains);
        Node panelRedGreen = EstimateStainVectorsCommand.createScatterPanel(new ScatterPlot(red, green, null, rgb), stainsWrapper, AxisColor.RED, AxisColor.GREEN);
        Node panelRedBlue = EstimateStainVectorsCommand.createScatterPanel(new ScatterPlot(red, blue, null, rgb), stainsWrapper, AxisColor.RED, AxisColor.BLUE);
        Node panelGreenBlue = EstimateStainVectorsCommand.createScatterPanel(new ScatterPlot(green, blue, null, rgb), stainsWrapper, AxisColor.GREEN, AxisColor.BLUE);
        GridPane panelPlots = new GridPane();
        panelPlots.setHgap(10.0);
        panelPlots.add(panelRedGreen, 0, 0);
        panelPlots.add(panelRedBlue, 1, 0);
        panelPlots.add(panelGreenBlue, 2, 0);
        panelPlots.setPadding(new Insets(0.0, 0.0, 10.0, 0.0));
        BorderPane panelSouth = new BorderPane();
        final TableView table = new TableView();
        table.getItems().setAll((Object[])new Integer[]{1, 2, 3});
        stainsWrapper.addStainListener(new StainChangeListener(){

            @Override
            public void stainChanged(StainsWrapper stainsWrapper) {
                table.refresh();
            }
        });
        TableColumn colName = new TableColumn("Name");
        colName.setCellValueFactory(v -> new SimpleStringProperty(stainsWrapper.getStains().getStain(((Integer)v.getValue()).intValue()).getName()));
        TableColumn colOrig = new TableColumn("Original");
        colOrig.setCellValueFactory(v -> new SimpleStringProperty(EstimateStainVectorsCommand.stainArrayAsString(Locale.getDefault(Locale.Category.FORMAT), stainsWrapper.getOriginalStains().getStain(((Integer)v.getValue()).intValue()), " | ", 3)));
        TableColumn colCurrent = new TableColumn("Current");
        colCurrent.setCellValueFactory(v -> new SimpleStringProperty(EstimateStainVectorsCommand.stainArrayAsString(Locale.getDefault(Locale.Category.FORMAT), stainsWrapper.getStains().getStain(((Integer)v.getValue()).intValue()), " | ", 3)));
        TableColumn colAngle = new TableColumn("Angle");
        colAngle.setCellValueFactory(v -> new SimpleStringProperty(GeneralTools.formatNumber((double)StainVector.computeAngle((StainVector)stainsWrapper.getOriginalStains().getStain(((Integer)v.getValue()).intValue()), (StainVector)stainsWrapper.getStains().getStain(((Integer)v.getValue()).intValue())), (int)2)));
        table.getColumns().addAll((Object[])new TableColumn[]{colName, colOrig, colCurrent, colAngle});
        table.setColumnResizePolicy(TableView.UNCONSTRAINED_RESIZE_POLICY);
        table.setPrefHeight(120.0);
        ParameterList params = new ParameterList().addDoubleParameter("minStainOD", "Min channel OD", 0.05, "", "Minimum staining OD - pixels with a lower OD in any channel (RGB) are ignored (default = 0.05)").addDoubleParameter("maxStainOD", "Max total OD", 1.0, "", "Maximum staining OD - more densely stained pixels are ignored (default = 1)").addDoubleParameter("ignorePercentage", "Ignore extrema", 1.0, "%", "Percentage of extreme pixels to ignore, to improve robustness in the presence of noise/other artefacts (default = 1)").addBooleanParameter("checkColors", "Exclude unrecognised colors (H&E only)", false, "Exclude unexpected colors (e.g. green) that are likely to be caused by artefacts and not true staining");
        Button btnAuto = new Button("Auto");
        btnAuto.setOnAction(e -> {
            double minOD = params.getDoubleParameterValue("minStainOD");
            double maxOD = params.getDoubleParameterValue("maxStainOD");
            double ignore = params.getDoubleParameterValue("ignorePercentage");
            boolean checkColors = params.getBooleanParameterValue("checkColors") != false && stainsWrapper.getOriginalStains().isH_E();
            ignore = Math.max(0.0, Math.min(ignore, 100.0));
            try {
                ColorDeconvolutionStains stainsNew = EstimateStainVectors.estimateStains((BufferedImage)img, (ColorDeconvolutionStains)stainsWrapper.getStains(), (double)minOD, (double)maxOD, (double)ignore, (boolean)checkColors);
                stainsWrapper.setStains(stainsNew);
            }
            catch (Exception e2) {
                Dialogs.showErrorMessage((String)TITLE, (Throwable)e2);
                logger.error(e2.getMessage(), (Throwable)e2);
            }
        });
        ParameterPanelFX panelParams = new ParameterPanelFX(params);
        BorderPane panelAuto = new BorderPane();
        panelAuto.setCenter((Node)panelParams.getPane());
        panelAuto.setBottom((Node)btnAuto);
        TitledPane titledStainVectors = new TitledPane("Stain vectors", (Node)table);
        titledStainVectors.setCollapsible(false);
        panelSouth.setCenter((Node)titledStainVectors);
        TitledPane titledAutoDetect = new TitledPane("Auto detect", (Node)panelAuto);
        titledAutoDetect.setCollapsible(false);
        panelSouth.setBottom((Node)titledAutoDetect);
        BorderPane panelMain = new BorderPane();
        panelMain.setCenter((Node)panelPlots);
        panelMain.setBottom((Node)panelSouth);
        if (Dialogs.builder().title("Visual Stain Editor").content((Node)panelMain).buttons(new ButtonType[]{ButtonType.OK, ButtonType.CANCEL}).showAndWait().orElse(ButtonType.CANCEL).equals(ButtonType.OK)) {
            return stainsWrapper.getStains();
        }
        stainsWrapper.resetStains();
        return stainsWrapper.getStains();
    }

    private static String stainArrayAsString(Locale locale, StainVector stain, String delimiter, int nDecimalPlaces) {
        return GeneralTools.arrayToString((Locale)locale, (double[])stain.getArray(), (String)delimiter, (int)nDecimalPlaces);
    }

    static Node createScatterPanel(ScatterPlot scatterPlot, StainsWrapper stainsWrapper, AxisColor axisX, AxisColor axisY) {
        StainScatterPanel panelScatter = new StainScatterPanel(scatterPlot, stainsWrapper, axisX, axisY);
        panelScatter.setWidth(200.0);
        panelScatter.setHeight(200.0);
        GridPane pane = new GridPane();
        Label y = new Label(axisY.toString());
        y.setAlignment(Pos.CENTER_LEFT);
        y.setRotate(-90.0);
        Group group = new Group();
        group.getChildren().add((Object)y);
        pane.add((Node)group, 0, 0);
        pane.add((Node)panelScatter, 1, 0);
        Label x = new Label(axisX.toString());
        x.setAlignment(Pos.CENTER);
        x.prefWidthProperty().bind((ObservableValue)panelScatter.widthProperty());
        pane.add((Node)x, 1, 1);
        return pane;
    }

    static class StainsWrapper {
        private List<StainChangeListener> listeners = new ArrayList<StainChangeListener>();
        private ColorDeconvolutionStains stainsOrig;
        private ColorDeconvolutionStains stains;

        StainsWrapper(ColorDeconvolutionStains stains) {
            this.stainsOrig = stains;
            this.stains = stains;
        }

        public ColorDeconvolutionStains getOriginalStains() {
            return this.stainsOrig;
        }

        public void addStainListener(StainChangeListener listener) {
            this.listeners.add(listener);
        }

        public void removeStainListener(StainChangeListener listener) {
            this.listeners.remove(listener);
        }

        public void changeStain(StainVector stainOld, StainVector stainNew) {
            int n = this.stains.getStainNumber(stainOld);
            if (n >= 1) {
                this.setStains(this.stains.changeStain(stainNew, n));
            }
        }

        public void setStains(ColorDeconvolutionStains stainsNew) {
            this.stains = stainsNew;
            for (StainChangeListener l : this.listeners) {
                l.stainChanged(this);
            }
        }

        public void resetStains() {
            this.setStains(this.getOriginalStains());
        }

        public ColorDeconvolutionStains getStains() {
            return this.stains;
        }
    }

    static class ScatterPlot {
        private static Color DEFAULT_COLOR_DRAW = ColorToolsFX.getColorWithOpacity(Color.RED, 0.1);
        private double[] x;
        private double[] y;
        private double minX;
        private double maxX;
        private double minY;
        private double maxY;
        private boolean correlationsCalculated = false;
        private double pearsonsCorrelation = Double.NaN;
        private double spearmansCorrelation = Double.NaN;
        private Color[] colorDraw = null;
        private Color[] colorFill = null;
        private boolean fillMarkers = false;
        private double markerSize = 3.0;

        public ScatterPlot(double[] x, double[] y) {
            this(x, y, null, null);
        }

        public ScatterPlot(float[] x, float[] y) {
            this(x, y, (int[])null, (int[])null);
        }

        public ScatterPlot(float[] x, float[] y, int[] colorDraw, int[] colorFill) {
            this(ScatterPlot.toDouble(x), ScatterPlot.toDouble(y), colorDraw, colorFill);
        }

        private void compute(double[] x, double[] y) {
            this.x = x;
            this.y = y;
            this.minX = Double.POSITIVE_INFINITY;
            this.maxX = Double.NEGATIVE_INFINITY;
            for (double v : x) {
                if (v > this.maxX) {
                    this.maxX = v;
                    continue;
                }
                if (!(v < this.minX)) continue;
                this.minX = v;
            }
            this.minY = Double.POSITIVE_INFINITY;
            this.maxY = Double.NEGATIVE_INFINITY;
            for (double v : y) {
                if (v > this.maxY) {
                    this.maxY = v;
                    continue;
                }
                if (!(v < this.minY)) continue;
                this.minY = v;
            }
        }

        private void setColors(int[] colorDraw, int[] colorFill) {
            int i;
            this.colorDraw = new Color[this.x.length];
            if (colorDraw == null || colorDraw.length == 0) {
                if (colorFill == null) {
                    Arrays.fill(this.colorDraw, DEFAULT_COLOR_DRAW);
                } else {
                    Arrays.fill(this.colorDraw, null);
                }
            } else if (colorDraw.length == 1) {
                Arrays.fill(this.colorDraw, ColorToolsAwt.getCachedColor((Integer)colorDraw[0], (ColorTools.alpha((int)colorDraw[0]) != 0 ? 1 : 0) != 0));
            } else {
                for (i = 0; i < Math.min(this.colorDraw.length, colorDraw.length); ++i) {
                    this.colorDraw[i] = ColorToolsFX.getCachedColor(colorDraw[i], ColorTools.alpha((int)colorFill[i]) != 0);
                }
            }
            this.colorFill = new Color[this.x.length];
            this.fillMarkers = true;
            if (colorFill == null || colorFill.length == 0) {
                this.fillMarkers = false;
            } else if (colorFill.length == 1) {
                Arrays.fill(this.colorFill, ColorToolsFX.getCachedColor(colorFill[0], false));
            } else {
                this.colorFill = new Color[this.x.length];
                for (i = 0; i < Math.min(this.colorFill.length, colorFill.length); ++i) {
                    this.colorFill[i] = ColorToolsFX.getCachedColor(colorFill[i], false);
                }
            }
        }

        static double[][] removeNaNs(double[] x, double[] y) {
            double[] x2 = new double[x.length];
            double[] y2 = new double[y.length];
            int k = 0;
            for (int i = 0; i < x.length; ++i) {
                double xx = x[i];
                double yy = y[i];
                if (Double.isNaN(xx) || Double.isNaN(yy)) continue;
                x2[k] = xx;
                y2[k] = yy;
                ++k;
            }
            if (k < x.length) {
                x2 = Arrays.copyOf(x2, k);
                y2 = Arrays.copyOf(y2, k);
            }
            return new double[][]{x2, y2};
        }

        private void calculateCorrelations() {
            double[][] denaned = ScatterPlot.removeNaNs(this.x, this.y);
            if (denaned[0].length > 0) {
                this.pearsonsCorrelation = new PearsonsCorrelation().correlation(denaned[0], denaned[1]);
                this.spearmansCorrelation = new SpearmansCorrelation().correlation(denaned[0], denaned[1]);
            }
            this.correlationsCalculated = true;
        }

        public double getPearsonsCorrelation() {
            if (!this.correlationsCalculated) {
                this.calculateCorrelations();
            }
            return this.pearsonsCorrelation;
        }

        public double getSpearmansCorrelation() {
            if (!this.correlationsCalculated) {
                this.calculateCorrelations();
            }
            return this.spearmansCorrelation;
        }

        public ScatterPlot(double[] x, double[] y, int[] colorDraw, int[] colorFill) {
            this.compute(x, y);
            this.setColors(colorDraw, colorFill);
        }

        public static double[] toDouble(float[] arr) {
            double[] arr2 = new double[arr.length];
            for (int i = 0; i < arr.length; ++i) {
                arr2[i] = arr[i];
            }
            return arr2;
        }

        public void setLimitsX(double minX, double maxX) {
            this.minX = (float)minX;
            this.maxX = (float)maxX;
        }

        public void setLimitsY(double minY, double maxY) {
            this.minY = (float)minY;
            this.maxY = (float)maxY;
        }

        public double getMinX() {
            return this.minX;
        }

        public double getMinY() {
            return this.minY;
        }

        public double getMaxX() {
            return this.maxX;
        }

        public double getMaxY() {
            return this.maxY;
        }

        public double getMarkerSize() {
            return this.markerSize;
        }

        public void setMarkerSize(double markerSize) {
            this.markerSize = markerSize;
        }

        public boolean getFillMarkers() {
            return this.fillMarkers;
        }

        public void setFillMarkers(boolean doFill) {
            this.fillMarkers = doFill;
        }

        public void drawPlot(GraphicsContext g, Rectangle2D region) {
            this.drawPlot(g, region, 10000);
        }

        public void drawPlot(GraphicsContext g, Rectangle2D region, int maxPoints) {
            g.save();
            g.beginPath();
            g.moveTo(region.getMinX(), region.getMinY());
            g.lineTo(region.getMaxX(), region.getMinY());
            g.lineTo(region.getMaxX(), region.getMaxY());
            g.lineTo(region.getMinX(), region.getMaxY());
            g.closePath();
            g.clip();
            double scaleX = region.getWidth() / (this.maxX - this.minX);
            double scaleY = region.getHeight() / (this.maxY - this.minY);
            g.setLineWidth(1.5);
            g.translate(region.getMinX(), region.getMinY());
            double increment = maxPoints < 0 || this.x.length <= maxPoints ? 1.0 : (double)this.x.length / (double)maxPoints;
            for (double i = 0.0; i < (double)this.x.length; i += increment) {
                Color cDraw;
                int ind = (int)i;
                double xx = this.x[ind];
                double yy = this.y[ind];
                double xo = (xx - this.minX) * scaleX - this.markerSize / 2.0;
                double yo = region.getHeight() - (yy - this.minY) * scaleY - this.markerSize / 2.0;
                Color color = cDraw = this.colorDraw == null ? null : this.colorDraw[ind];
                if (this.fillMarkers) {
                    Color cFill;
                    Color color2 = cFill = this.colorFill[ind] == null ? cDraw : this.colorFill[ind];
                    if (cFill != null) {
                        g.setFill((Paint)cFill);
                        g.fillOval(xo, yo, this.markerSize, this.markerSize);
                        if (cFill == cDraw) continue;
                    }
                }
                if (cDraw == null) continue;
                g.setStroke((Paint)cDraw);
                g.strokeOval(xo, yo, this.markerSize, this.markerSize);
            }
            g.restore();
        }
    }

    private static enum AxisColor {
        RED,
        GREEN,
        BLUE;


        public String toString() {
            return switch (this.ordinal()) {
                case 0 -> "Red";
                case 1 -> "Green";
                case 2 -> "Blue";
                default -> throw new IllegalArgumentException();
            };
        }
    }

    static interface StainChangeListener {
        public void stainChanged(StainsWrapper var1);
    }

    static class StainScatterPanel
    extends Canvas
    implements StainChangeListener {
        private AxisColor xAxis;
        private AxisColor yAxis;
        private ScatterPlot plot;
        private StainsWrapper stainsWrapper;
        private StainVector stainEditing = null;
        private double handleSize = 5.0;
        private int padding = 2;
        private boolean constrainX = false;

        public StainScatterPanel(ScatterPlot plot, StainsWrapper stainsWrapper, AxisColor xAxis, AxisColor yAxis) {
            this.plot = plot;
            this.stainsWrapper = stainsWrapper;
            this.stainsWrapper.addStainListener(this);
            this.xAxis = xAxis;
            this.yAxis = yAxis;
            this.plot.setLimitsX(0.0, 1.0);
            this.plot.setLimitsY(0.0, 1.0);
            this.addEventHandler(MouseEvent.ANY, new MouseListener());
            this.updatePlot();
            this.widthProperty().addListener(e -> this.updatePlot());
            this.heightProperty().addListener(e -> this.updatePlot());
        }

        private double componentXtoDataX(double x) {
            double scaleX = (this.getWidth() - (double)(this.padding * 2)) / (this.plot.getMaxX() - this.plot.getMinY());
            return (x - (double)this.padding) / scaleX + this.plot.getMinX();
        }

        private double componentYtoDataY(double y) {
            double scaleY = (this.getHeight() - (double)(this.padding * 2)) / (this.plot.getMaxY() - this.plot.getMinY());
            return ((double)this.padding - y) / scaleY + this.plot.getMaxX() + this.plot.getMinY();
        }

        private double dataXtoComponentX(double x) {
            double scaleX = (this.getWidth() - (double)(this.padding * 2)) / (this.plot.getMaxX() - this.plot.getMinY());
            return (x - this.plot.getMinX()) * scaleX + (double)this.padding;
        }

        private double dataYtoComponentY(double y) {
            double scaleY = (this.getHeight() - (double)(this.padding * 2)) / (this.plot.getMaxY() - this.plot.getMinY());
            return (this.plot.getMaxY() - (y - this.plot.getMinY())) * scaleY + (double)this.padding;
        }

        private Rectangle2D getRegion() {
            return new Rectangle2D((double)this.padding, (double)this.padding, this.getWidth() - (double)(this.padding * 2), this.getHeight() - (double)(this.padding * 2));
        }

        private StainVector grabHandle(double x, double y) {
            double d2;
            ColorDeconvolutionStains stains;
            ColorDeconvolutionStains colorDeconvolutionStains = stains = this.stainsWrapper == null ? null : this.stainsWrapper.getStains();
            if (stains == null) {
                return null;
            }
            StainVector s1 = stains.getStain(1);
            StainVector s2 = stains.getStain(2);
            double d1 = Point2D.distanceSq(x, y, this.dataXtoComponentX(this.getStainX(s1)), this.dataYtoComponentY(this.getStainY(s1)));
            if (d1 <= (d2 = Point2D.distanceSq(x, y, this.dataXtoComponentX(this.getStainX(s2)), this.dataYtoComponentY(this.getStainY(s2)))) && d1 <= this.handleSize * this.handleSize) {
                return s1;
            }
            if (d2 < d1 && d2 <= this.handleSize * this.handleSize) {
                return s2;
            }
            return null;
        }

        private double getStainX(StainVector stain) {
            switch (this.xAxis.ordinal()) {
                case 2: {
                    return stain.getBlue();
                }
                case 1: {
                    return stain.getGreen();
                }
                case 0: {
                    return stain.getRed();
                }
            }
            return Double.NaN;
        }

        private double getStainY(StainVector stain) {
            switch (this.yAxis.ordinal()) {
                case 2: {
                    return stain.getBlue();
                }
                case 1: {
                    return stain.getGreen();
                }
                case 0: {
                    return stain.getRed();
                }
            }
            return Double.NaN;
        }

        public boolean getConstrainX() {
            return this.constrainX;
        }

        public void setConstrainX(boolean constrainX) {
            this.constrainX = constrainX;
        }

        public void updatePlot() {
            GraphicsContext g2d = this.getGraphicsContext2D();
            g2d.setFill((Paint)Color.WHITE);
            g2d.clearRect(0.0, 0.0, this.getWidth(), this.getHeight());
            if (this.getWidth() < (double)(this.padding * 2) || this.getHeight() < (double)(this.padding * 2)) {
                return;
            }
            Rectangle2D region = this.getRegion();
            g2d.fillRect((double)this.padding, (double)this.padding, this.getWidth() - (double)(this.padding * 2), this.getHeight() - (double)(this.padding * 2));
            this.plot.drawPlot(g2d, region);
            g2d.setLineWidth(1.0);
            g2d.setStroke((Paint)Color.GRAY);
            g2d.strokeRect(region.getMinX(), region.getMinY(), region.getWidth(), region.getHeight());
            if (this.stainsWrapper == null) {
                return;
            }
            g2d.setLineWidth(3.0);
            ColorDeconvolutionStains stains = this.stainsWrapper.getStains();
            for (int i = 1; i <= 3; ++i) {
                StainVector stain = stains.getStain(i);
                if (stain.isResidual()) continue;
                double x2 = this.getStainX(stain);
                double y2 = this.getStainY(stain);
                if (Double.isNaN(x2) || Double.isNaN(y2)) continue;
                if (stain == this.stainEditing) {
                    g2d.setStroke((Paint)Color.YELLOW);
                } else {
                    g2d.setStroke((Paint)ColorToolsFX.getCachedColor(stain.getColor()));
                }
                g2d.strokeLine(this.dataXtoComponentX(0.0), this.dataYtoComponentY(0.0), this.dataXtoComponentX(x2), this.dataYtoComponentY(y2));
                g2d.strokeOval(this.dataXtoComponentX(x2) - this.handleSize / 2.0, this.dataYtoComponentY(y2) - this.handleSize / 2.0, this.handleSize, this.handleSize);
            }
        }

        private void handleMouseDragged(MouseEvent e) {
            if (this.stainEditing == null) {
                return;
            }
            double x = this.componentXtoDataX(e.getX());
            double y = this.componentYtoDataY(e.getY());
            double r = this.stainEditing.getRed();
            double g = this.stainEditing.getGreen();
            double b = this.stainEditing.getBlue();
            switch (this.yAxis.ordinal()) {
                case 2: {
                    b = y;
                    break;
                }
                case 1: {
                    g = y;
                    break;
                }
                case 0: {
                    r = y;
                }
            }
            if (!this.constrainX) {
                switch (this.xAxis.ordinal()) {
                    case 2: {
                        b = x;
                        break;
                    }
                    case 1: {
                        g = x;
                        break;
                    }
                    case 0: {
                        r = x;
                    }
                }
            }
            StainVector stainNew = StainVector.createStainVector((String)this.stainEditing.getName(), (double)r, (double)g, (double)b);
            this.stainsWrapper.changeStain(this.stainEditing, stainNew);
            this.stainEditing = stainNew;
            this.updatePlot();
        }

        void handleMousePressed(MouseEvent e) {
            this.stainEditing = this.grabHandle(e.getX(), e.getY());
            if (this.stainEditing != null) {
                logger.debug("Editing stain vector: " + String.valueOf(this.stainEditing));
            }
            this.updatePlot();
        }

        void handleMouseReleased(MouseEvent e) {
            if (this.stainEditing == null) {
                return;
            }
            logger.info("Updated stain vector: {}", (Object)this.stainEditing);
            this.stainEditing = null;
            this.updatePlot();
        }

        @Override
        public void stainChanged(StainsWrapper stainsWrapper) {
            this.updatePlot();
        }

        class MouseListener
        implements EventHandler<MouseEvent> {
            MouseListener() {
            }

            public void handle(MouseEvent event) {
                if (event.getEventType() == MouseEvent.MOUSE_DRAGGED) {
                    StainScatterPanel.this.handleMouseDragged(event);
                } else if (event.getEventType() == MouseEvent.MOUSE_PRESSED) {
                    StainScatterPanel.this.handleMousePressed(event);
                } else if (event.getEventType() == MouseEvent.MOUSE_RELEASED) {
                    StainScatterPanel.this.handleMouseReleased(event);
                }
            }
        }
    }
}

