/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.stream.Stream;
import javafx.beans.property.LongProperty;
import javafx.scene.Node;
import javafx.scene.control.ButtonType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.fx.dialogs.Dialogs;
import qupath.lib.common.Version;
import qupath.lib.gui.QuPathGUI;
import qupath.lib.gui.UpdateManagerContainer;
import qupath.lib.gui.commands.Commands;
import qupath.lib.gui.extensions.GitHubProject;
import qupath.lib.gui.extensions.UpdateChecker;
import qupath.lib.gui.localization.QuPathResources;
import qupath.lib.gui.prefs.PathPrefs;
import qupath.lib.gui.tools.GuiTools;

class UpdateManager {
    private static final Logger logger = LoggerFactory.getLogger(UpdateManager.class);
    private static final int HOURS_BETWEEN_AUTO_UPDATES = 12;
    private static final LongProperty lastUpdateCheck = PathPrefs.createPersistentPreference("lastUpdateCheck", -1L);
    private final QuPathGUI qupath;

    private UpdateManager(QuPathGUI qupath) {
        this.qupath = qupath;
    }

    public static UpdateManager create(QuPathGUI qupath) {
        return new UpdateManager(qupath);
    }

    public void runAutomaticUpdateCheck() {
        long lastUpdateCheckMillis;
        PathPrefs.AutoUpdateType checkType = (PathPrefs.AutoUpdateType)((Object)PathPrefs.autoUpdateCheckProperty().get());
        if (checkType == null || checkType == PathPrefs.AutoUpdateType.NONE) {
            logger.debug("No update check because of user preference ({})", (Object)checkType);
            return;
        }
        long currentTime = System.currentTimeMillis();
        double diffHours = (double)(currentTime - (lastUpdateCheckMillis = lastUpdateCheck.get())) / 3600000.0;
        if (diffHours < 12.0) {
            logger.debug("Skipping update check because it was already checked recently ({} hours ago, which is less than {} hours ago)", (Object)diffHours, (Object)12);
            return;
        }
        logger.debug("Automatic update check started - will search for {}", (Object)checkType);
        this.runUpdateCheckInBackground(checkType, false);
    }

    public void runManualUpdateCheck() {
        logger.debug("Manual update check started - will search for QuPath and extensions");
        this.runUpdateCheckInBackground(PathPrefs.AutoUpdateType.QUPATH_AND_EXTENSIONS, true);
    }

    private void runUpdateCheckInBackground(PathPrefs.AutoUpdateType checkType, boolean showDialogs) {
        this.qupath.getThreadPoolManager().submitShortTask(() -> this.doUpdateCheck(checkType, showDialogs));
    }

    private synchronized void doUpdateCheck(PathPrefs.AutoUpdateType updateCheckType, boolean showDialogs) {
        boolean result;
        UpdateManagerContainer updateManagerContainer;
        lastUpdateCheck.set(System.currentTimeMillis());
        List<UpdateManagerContainer.UpdateEntry> updateEntries = Stream.concat(this.getQuPathUpdate(updateCheckType).stream(), this.getExtensionUpdates(updateCheckType).stream()).toList();
        if (updateEntries.isEmpty()) {
            logger.info("No updates found");
            if (showDialogs) {
                Dialogs.showMessageDialog((String)QuPathResources.getString("UpdateManager.updateCheck"), (String)QuPathResources.getString("UpdateManager.noUpdatesFound"));
            }
            return;
        }
        try {
            updateManagerContainer = new UpdateManagerContainer(updateEntries);
        }
        catch (IOException e) {
            logger.error("Cannot create update manager window", (Throwable)e);
            if (showDialogs) {
                Dialogs.showErrorMessage((String)QuPathResources.getString("UpdateManager.updateCheck"), (String)QuPathResources.getString("UpdateManager.cannotCreateUpdateWindow"));
            }
            return;
        }
        boolean bl = result = new Dialogs.Builder().buttons(new ButtonType[]{ButtonType.OK}).title(QuPathResources.getString("UpdateManager.updateCheck")).headerText(QuPathResources.getString("UpdateManager.updatesAvailable")).content((Node)updateManagerContainer).resizable().showAndWait().orElse(ButtonType.CANCEL) == ButtonType.OK;
        if (result && updateManagerContainer.getSelectedUpdateType() != null) {
            PathPrefs.autoUpdateCheckProperty().set((Object)updateManagerContainer.getSelectedUpdateType());
        }
    }

    private Optional<UpdateManagerContainer.UpdateEntry> getQuPathUpdate(PathPrefs.AutoUpdateType updateCheckType) {
        Version qupathVersion = QuPathGUI.getVersion();
        if (qupathVersion != null && qupathVersion != Version.UNKNOWN && List.of(PathPrefs.AutoUpdateType.QUPATH_ONLY, PathPrefs.AutoUpdateType.QUPATH_AND_EXTENSIONS).contains((Object)updateCheckType)) {
            GitHubProject.GitHubRepo gitHubProject = GitHubProject.GitHubRepo.create("QuPath", "qupath", "qupath");
            logger.debug("Update check for {}", (Object)gitHubProject.getUrlString());
            try {
                UpdateChecker.ReleaseVersion latestRelease = UpdateChecker.checkForUpdate(gitHubProject);
                if (latestRelease != null && latestRelease.getVersion() != Version.UNKNOWN && qupathVersion.compareTo(latestRelease.getVersion()) < 0) {
                    logger.info("Found newer release for {} ({} -> {})", new Object[]{gitHubProject.getName(), qupathVersion, latestRelease.getVersion()});
                    return Optional.of(new UpdateManagerContainer.UpdateEntry(gitHubProject.getName(), qupathVersion.toString(), latestRelease.getVersion().toString(), latestRelease.getUri() == null ? () -> {} : () -> GuiTools.browseURI(latestRelease.getUri()), latestRelease.getUri() == null ? QuPathResources.getString("UpdateManager.noAvailableUrl") : latestRelease.getUri().toString()));
                }
                if (latestRelease != null) {
                    logger.info("No newer release for {} (current {} vs upstream {})", new Object[]{gitHubProject.getName(), qupathVersion, latestRelease.getVersion()});
                }
            }
            catch (Exception e) {
                logger.warn("Update check failed for {}", (Object)gitHubProject, (Object)e);
            }
        }
        return Optional.empty();
    }

    private List<UpdateManagerContainer.UpdateEntry> getExtensionUpdates(PathPrefs.AutoUpdateType updateCheckType) {
        if (List.of(PathPrefs.AutoUpdateType.QUPATH_AND_EXTENSIONS, PathPrefs.AutoUpdateType.EXTENSIONS_ONLY).contains((Object)updateCheckType)) {
            try {
                return ((List)QuPathGUI.getExtensionCatalogManager().getAvailableUpdates().get()).stream().map(extensionUpdate -> new UpdateManagerContainer.UpdateEntry(extensionUpdate.extensionName(), extensionUpdate.currentVersion(), extensionUpdate.newVersion(), () -> Commands.showInstalledExtensions(this.qupath), QuPathResources.getString("UpdateManager.openExtensionManager"))).toList();
            }
            catch (InterruptedException | ExecutionException e) {
                logger.warn("Cannot check updates on extensions", (Throwable)e);
            }
        }
        return List.of();
    }
}

