/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.gui;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.ServiceConfigurationError;
import java.util.ServiceLoader;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.ext.extensionmanager.core.ExtensionCatalogManager;
import qupath.fx.dialogs.Dialogs;
import qupath.fx.utils.FXUtils;
import qupath.lib.common.Version;
import qupath.lib.gui.QuPathGUI;
import qupath.lib.gui.extensions.QuPathExtension;
import qupath.lib.gui.localization.QuPathResources;
import qupath.lib.images.servers.ImageServerBuilder;
import qupath.lib.images.servers.ImageServerProvider;

class ExtensionLoader {
    private static final Logger logger = LoggerFactory.getLogger(ExtensionLoader.class);
    private final Set<Class<? extends QuPathExtension>> loadedExtensions = new HashSet<Class<? extends QuPathExtension>>();
    private final ClassLoader extensionClassLoader;
    private final QuPathGUI quPathGUI;

    private ExtensionLoader(ExtensionCatalogManager extensionCatalogManager, QuPathGUI quPathGUI) {
        this.extensionClassLoader = extensionCatalogManager.getExtensionClassLoader();
        this.quPathGUI = quPathGUI;
        this.loadExtensions(false);
        extensionCatalogManager.addOnJarLoadedRunnable(() -> this.loadExtensions(true));
    }

    public static void loadFromManager(ExtensionCatalogManager extensionCatalogManager, QuPathGUI quPathGUI) {
        new ExtensionLoader(extensionCatalogManager, quPathGUI);
    }

    private synchronized void loadExtensions(boolean showNotifications) {
        try {
            for (QuPathExtension extension : ServiceLoader.load(QuPathExtension.class, this.extensionClassLoader)) {
                if (this.loadedExtensions.contains(extension.getClass())) continue;
                this.loadedExtensions.add(extension.getClass());
                FXUtils.runOnApplicationThread(() -> this.loadExtension(extension, showNotifications));
            }
            this.loadServerBuilders(showNotifications);
        }
        catch (ServiceConfigurationError e) {
            logger.debug("Error while loading extension", (Throwable)e);
        }
    }

    private void loadExtension(QuPathExtension extension, boolean showNotifications) {
        try {
            long startTime = System.currentTimeMillis();
            extension.installExtension(this.quPathGUI);
            long endTime = System.currentTimeMillis();
            logger.info("Loaded extension {} version {} ({} ms)", new Object[]{extension.getName(), extension.getVersion(), endTime - startTime});
            if (showNotifications) {
                Dialogs.showInfoNotification((String)QuPathResources.getString("ExtensionLoader.extensionLoaded"), (String)extension.getName());
            }
        }
        catch (Exception | LinkageError e) {
            if (showNotifications) {
                Dialogs.showErrorNotification((String)QuPathResources.getString("ExtensionLoader.extensionError"), (String)MessageFormat.format(QuPathResources.getString("ExtensionLoader.unableToLoad"), extension.getName()));
            }
            logger.error("Error loading extension {}:\n{}{}", new Object[]{extension.getName(), ExtensionLoader.getCompatibilityErrorMessage(extension), ExtensionLoader.getDeleteExtensionMessage(extension), e});
            this.quPathGUI.getCommonActions().SHOW_LOG.handle(null);
        }
    }

    private void loadServerBuilders(boolean showNotifications) {
        List<String> previousServerBuilderNames = ImageServerProvider.getInstalledImageServerBuilders().stream().map(ImageServerBuilder::getName).toList();
        ImageServerProvider.setServiceLoader(ServiceLoader.load(ImageServerBuilder.class, this.extensionClassLoader));
        List<String> newServerBuildersNames = ImageServerProvider.getInstalledImageServerBuilders().stream().map(ImageServerBuilder::getName).filter(name -> !previousServerBuilderNames.contains(name)).toList();
        if (!newServerBuildersNames.isEmpty()) {
            logger.info("Loaded image servers {}", newServerBuildersNames);
            if (showNotifications) {
                for (String builderName : newServerBuildersNames) {
                    Dialogs.showInfoNotification((String)QuPathResources.getString("ExtensionLoader.imageServerLoaded"), (String)builderName);
                }
            }
        }
    }

    private static String getCompatibilityErrorMessage(QuPathExtension extension) {
        Version compatibleQuPathVersion;
        Version qupathVersion = QuPathGUI.getVersion();
        if (!Objects.equals(qupathVersion, compatibleQuPathVersion = extension.getQuPathVersion())) {
            return "";
        }
        if (compatibleQuPathVersion == null || Version.UNKNOWN.equals((Object)compatibleQuPathVersion)) {
            return String.format("QuPath version for which the '%s' was written is unknown!", extension.getName());
        }
        if (compatibleQuPathVersion.getMajor() == qupathVersion.getMajor() && compatibleQuPathVersion.getMinor() == qupathVersion.getMinor()) {
            return String.format("'%s' reports that it is compatible with QuPath %s; the current QuPath version is %s.", extension.getName(), compatibleQuPathVersion, qupathVersion);
        }
        return String.format("'%s' was written for QuPath %s but current version is %s.", extension.getName(), compatibleQuPathVersion, qupathVersion);
    }

    private static String getDeleteExtensionMessage(QuPathExtension extension) {
        try {
            return String.format("It is recommended that you delete %s and restart QuPath.", URLDecoder.decode(extension.getClass().getProtectionDomain().getCodeSource().getLocation().toExternalForm(), StandardCharsets.UTF_8));
        }
        catch (Exception e) {
            logger.debug("Error finding code source {}", (Object)e.getLocalizedMessage(), (Object)e);
            return "";
        }
    }
}

