/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.writers.ome;

import java.awt.image.BufferedImage;
import java.io.File;
import java.nio.channels.ClosedByInterruptException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.stage.FileChooser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.fx.dialogs.Dialogs;
import qupath.fx.dialogs.FileChoosers;
import qupath.lib.common.GeneralTools;
import qupath.lib.common.ThreadTools;
import qupath.lib.gui.QuPathGUI;
import qupath.lib.gui.prefs.PathPrefs;
import qupath.lib.gui.tools.GuiTools;
import qupath.lib.gui.viewer.QuPathViewer;
import qupath.lib.images.ImageData;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.writers.ome.OMEPyramidWriter;
import qupath.lib.objects.PathObject;
import qupath.lib.plugins.parameters.ParameterList;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.RegionRequest;
import qupath.lib.roi.interfaces.ROI;

public class OMEPyramidWriterCommand
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(OMEPyramidWriterCommand.class);
    private static ObjectProperty<OMEPyramidWriter.CompressionType> defaultPyramidCompression = PathPrefs.createPersistentPreference((String)"ome-pyramid-compression", (Enum)OMEPyramidWriter.CompressionType.DEFAULT, OMEPyramidWriter.CompressionType.class);
    private static IntegerProperty defaultTileSize = PathPrefs.createPersistentPreference((String)"ome-pyramid-tile-size", (int)256);
    private static IntegerProperty minSizeForTiling = PathPrefs.createPersistentPreference((String)"ome-pyramid-min-size-for-tiling", (int)4096);
    private static IntegerProperty scaledDownsample = PathPrefs.createPersistentPreference((String)"ome-pyramid-scaled-downsample", (int)4);
    private static BooleanProperty parallelizeTiling = PathPrefs.createPersistentPreference((String)"ome-pyramid-parallelize", (boolean)true);
    private static BooleanProperty allZ = PathPrefs.createPersistentPreference((String)"ome-pyramid-all-z", (boolean)true);
    private static BooleanProperty allT = PathPrefs.createPersistentPreference((String)"ome-pyramid-all-t", (boolean)true);
    private QuPathGUI qupath;
    private ExecutorService pool;
    private Future<?> currentTask;

    public static OMEPyramidWriter.CompressionType getDefaultPyramidCompression() {
        return (OMEPyramidWriter.CompressionType)((Object)defaultPyramidCompression.get());
    }

    public static int getDefaultTileSize() {
        return defaultTileSize.get();
    }

    public static int getMinSizeForTiling() {
        return minSizeForTiling.get();
    }

    public OMEPyramidWriterCommand(QuPathGUI qupath) {
        this.qupath = qupath;
    }

    @Override
    public void run() {
        File fileOutput;
        int height;
        int width;
        if (this.currentTask != null && !this.currentTask.isDone()) {
            if (!Dialogs.showConfirmDialog((String)"OME Pyramid writer", (String)"Do you want to stop the current export?")) {
                return;
            }
            this.currentTask.cancel(true);
        }
        QuPathViewer viewer = this.qupath.getViewer();
        int zPos = viewer.getZPosition();
        int tPos = viewer.getTPosition();
        ImageData imageData = viewer.getImageData();
        if (imageData == null) {
            GuiTools.showNoImageError((String)"OME Pyramid writer");
            return;
        }
        ImageServer server = imageData.getServer();
        PathObject selected = imageData.getHierarchy().getSelectionModel().getSelectedObject();
        ImageRegion region = null;
        if (selected == null || !selected.hasROI() || !selected.getROI().isArea()) {
            width = server.getWidth();
            height = server.getHeight();
        } else {
            region = ImageRegion.createInstance((ROI)selected.getROI());
            width = region.getWidth();
            height = region.getHeight();
        }
        OMEPyramidWriter.CompressionType compression = OMEPyramidWriterCommand.getDefaultPyramidCompression();
        List<String> compatibleCompression = Arrays.stream(OMEPyramidWriter.CompressionType.values()).filter(c -> c.supportsImage(server)).map(c -> c.toFriendlyString()).toList();
        if (!compatibleCompression.contains(compression.toFriendlyString())) {
            compression = OMEPyramidWriter.CompressionType.DEFAULT;
        }
        ParameterList params = new ParameterList().addChoiceParameter("compression", "Compression type", (Object)compression.toFriendlyString(), compatibleCompression).addIntParameter("scaledDownsample", "Pyramidal downsample", scaledDownsample.get(), "", 1.0, 8.0, "Amount to downsample each consecutive pyramidal level; use 1 to indicate the image should not be pyramidal").addIntParameter("tileSize", "Tile size", OMEPyramidWriterCommand.getDefaultTileSize(), "px", "Tile size for export (should be between 128 and 8192)").addBooleanParameter("parallelize", "Parallelize export", parallelizeTiling.get(), "Export image tiles in parallel - this should be faster, best keep it on unless you encounter export problems").addBooleanParameter("allZ", "All z-slices", allZ.get(), "Include all z-slices in the stack").addBooleanParameter("allT", "All timepoints", allT.get(), "Include all timepoints in the time-series");
        boolean singleTile = server.getTileRequestManager().getTileRequests(RegionRequest.createInstance((ImageServer)server)).size() == 1;
        params.setHiddenParameters(server.nZSlices() == 1, new String[]{"allZ"});
        params.setHiddenParameters(server.nTimepoints() == 1, new String[]{"allT"});
        params.setHiddenParameters(singleTile, new String[]{"tileSize", "parallelize"});
        if (!GuiTools.showParameterDialog((String)"Export OME-TIFF", (ParameterList)params)) {
            return;
        }
        compression = OMEPyramidWriter.CompressionType.fromFriendlyString((String)params.getChoiceParameterValue("compression"));
        defaultPyramidCompression.set((Object)compression);
        int downsampleScale = params.getIntParameterValue("scaledDownsample");
        scaledDownsample.set(downsampleScale);
        int tileSize = params.getIntParameterValue("tileSize");
        boolean parallelize = params.getBooleanParameterValue("parallelize");
        if (!singleTile) {
            tileSize = GeneralTools.clipValue((int)tileSize, (int)128, (int)8192);
            defaultTileSize.set(tileSize);
            parallelizeTiling.set(parallelize);
        }
        boolean doAllZ = false;
        boolean doAllT = false;
        if (server.nZSlices() > 1) {
            doAllZ = params.getBooleanParameterValue("allZ");
            allZ.set(doAllZ);
        }
        if (server.nTimepoints() > 1) {
            doAllT = params.getBooleanParameterValue("allT");
            allT.set(doAllT);
        }
        OMEPyramidWriter.Builder builder = new OMEPyramidWriter.Builder((ImageServer<BufferedImage>)server);
        if (region != null) {
            builder = builder.region(region);
        } else {
            if (server.nZSlices() > 1 && !doAllZ) {
                builder.zSlice(zPos);
            }
            if (server.nTimepoints() > 1 && !doAllT) {
                builder.timePoint(tPos);
            }
        }
        builder.compression(compression);
        if (downsampleScale <= 1 || (double)Math.max(width, height) / server.getDownsampleForResolution(0) < (double)minSizeForTiling.get()) {
            builder.downsamples(server.getDownsampleForResolution(0));
        } else {
            builder.scaledDownsampling(server.getDownsampleForResolution(0), downsampleScale);
        }
        builder = singleTile ? builder.tileSize(width, height) : builder.tileSize(tileSize).parallelize(parallelize);
        if (server.nZSlices() > 1 && doAllZ) {
            builder.allZSlices();
        }
        if (server.nTimepoints() > 1 && doAllT) {
            builder.allTimePoints();
        }
        if ((fileOutput = FileChoosers.promptToSaveFile((String)"Write pyramid", null, (FileChooser.ExtensionFilter[])new FileChooser.ExtensionFilter[]{FileChoosers.createExtensionFilter((String)"OME TIFF pyramid", (String[])new String[]{".ome.tif"})})) == null) {
            return;
        }
        String name = fileOutput.getName();
        if (name.endsWith(".tif") && !name.endsWith(".ome.tif")) {
            fileOutput = new File(fileOutput.getParentFile(), name.substring(0, name.length() - 4) + ".ome.tif");
        }
        OMEPyramidWriter.OMEPyramidSeries writer = builder.build();
        if (this.pool == null) {
            this.pool = Executors.newSingleThreadExecutor(ThreadTools.createThreadFactory((String)"ome-pyramid-export", (boolean)false));
        }
        this.currentTask = this.pool.submit(new WriterTask(OMEPyramidWriter.createWriter(writer), fileOutput.getAbsolutePath()));
    }

    static class WriterTask
    implements Runnable {
        private OMEPyramidWriter writer;
        private String path;

        WriterTask(OMEPyramidWriter writer, String path) {
            this.writer = writer;
            this.path = path;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                Dialogs.showInfoNotification((String)"OME Pyramid writer", (String)("Exporting to " + this.path + " - \nplease keep QuPath running until export is complete!"));
                long startTime = System.currentTimeMillis();
                this.writer.writeImage(this.path);
                long endTime = System.currentTimeMillis();
                logger.info(String.format("OME TIFF export to {} complete in %.1f seconds", (double)(endTime - startTime) / 1000.0), (Object)this.path);
                Dialogs.showInfoNotification((String)"OME Pyramid writer", (String)"OME TIFF export complete!");
            }
            catch (ClosedByInterruptException e) {
                logger.warn("OME Pyramid writer closed by interrupt (possibly due to user cancelling it)", (Throwable)e);
            }
            catch (Exception e) {
                Dialogs.showErrorMessage((String)"OME Pyramid writer", (Throwable)e);
                logger.error(e.getMessage(), (Throwable)e);
            }
            finally {
                this.writer = null;
            }
        }

        public String getPath() {
            return this.path;
        }
    }
}

