/*
 * Decompiled with CFR 0.152.
 */
package qupath.lib.images.writers.ome;

import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import loci.formats.FormatException;
import loci.formats.FormatWriter;
import loci.formats.IFormatWriter;
import loci.formats.ImageWriter;
import loci.formats.MetadataTools;
import loci.formats.codec.CodecOptions;
import loci.formats.meta.IMetadata;
import loci.formats.meta.IPyramidStore;
import loci.formats.meta.MetadataRetrieve;
import loci.formats.out.OMETiffWriter;
import loci.formats.out.PyramidOMETiffWriter;
import loci.formats.out.TiffWriter;
import loci.formats.tiff.IFD;
import ome.units.UNITS;
import ome.units.quantity.Length;
import ome.xml.model.enums.DimensionOrder;
import ome.xml.model.primitives.Color;
import ome.xml.model.primitives.PositiveInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import qupath.lib.color.ColorModelFactory;
import qupath.lib.common.ColorTools;
import qupath.lib.images.servers.ImageChannel;
import qupath.lib.images.servers.ImageServer;
import qupath.lib.images.servers.ImageServerMetadata;
import qupath.lib.images.servers.ImageServers;
import qupath.lib.images.servers.PixelCalibration;
import qupath.lib.images.servers.PixelType;
import qupath.lib.images.servers.ServerTools;
import qupath.lib.images.servers.TileRequest;
import qupath.lib.regions.ImageRegion;
import qupath.lib.regions.RegionRequest;

public class OMEPyramidWriter {
    private static Logger logger = LoggerFactory.getLogger(OMEPyramidWriter.class);
    private static int DEFAULT_TILE_SIZE = 512;
    private static int MIN_SIZE_FOR_TILING = DEFAULT_TILE_SIZE * 8;
    private static int MIN_SIZE_FOR_PYRAMID = MIN_SIZE_FOR_TILING * 2;
    private boolean keepExisting = false;
    private List<OMEPyramidSeries> series = new ArrayList<OMEPyramidSeries>();

    private OMEPyramidWriter() {
    }

    private OMEPyramidWriter(Collection<OMEPyramidSeries> series) {
        this.series.addAll(series);
    }

    public void writeImage(String path) throws FormatException, IOException {
        IMetadata meta = MetadataTools.createOMEXMLMetadata();
        File file = new File(path);
        if (file.exists() && !this.keepExisting) {
            logger.warn("Deleting existing file {}", (Object)path);
            if (!file.delete()) {
                throw new IOException("Unable to delete " + file.getAbsolutePath());
            }
        }
        try (ImageWriter writer = new ImageWriter();){
            boolean bigTiff = false;
            boolean noBigTiff = false;
            long nPixelBytes = 0L;
            for (int s2 = 0; s2 < this.series.size(); ++s2) {
                OMEPyramidSeries temp = this.series.get(s2);
                if (!(bigTiff |= Boolean.TRUE.equals(temp.bigTiff)) && !noBigTiff && Boolean.FALSE.equals(temp.bigTiff)) {
                    noBigTiff = true;
                }
                for (double d : temp.downsamples) {
                    nPixelBytes = (long)((double)nPixelBytes + (double)((long)Math.ceil((double)temp.width / d)) * Math.ceil((double)temp.height / d) * (double)temp.channels.length * (double)temp.getExportPixelType().getBytesPerPixel() * (double)(temp.tEnd - temp.tStart) * (double)(temp.zEnd - temp.zStart));
                }
                temp.initializeMetadata(meta, s2);
            }
            writer.setWriteSequentially(true);
            writer.setMetadataRetrieve((MetadataRetrieve)meta);
            IFormatWriter wrappedWriter = writer.getWriter(path);
            if (wrappedWriter instanceof TiffWriter) {
                TiffWriter tiffWriter = (TiffWriter)wrappedWriter;
                if (bigTiff) {
                    logger.debug("Setting bigtiff to true");
                    tiffWriter.setBigTiff(true);
                } else if (noBigTiff) {
                    logger.debug("Setting bigtiff to false");
                    tiffWriter.setBigTiff(false);
                    tiffWriter.setCanDetectBigTiff(false);
                } else {
                    long bigTiffBytes = 2043650047L;
                    if (nPixelBytes >= bigTiffBytes) {
                        logger.info(String.format("Setting to big tiff (estimated %.2f MB", (double)nPixelBytes / 1048576.0));
                        tiffWriter.setBigTiff(true);
                    }
                }
            }
            this.series.stream().map(s -> s.codecOptions).filter(Objects::nonNull).findAny().ifPresent(arg_0 -> ((ImageWriter)writer).setCodecOptions(arg_0));
            writer.setId(path);
            for (int s3 = 0; s3 < this.series.size(); ++s3) {
                OMEPyramidSeries temp = this.series.get(s3);
                logger.info("Writing {} to {} (series {}/{})", new Object[]{ServerTools.getDisplayableImageName(temp.getOriginalServer()), path, s3 + 1, this.series.size()});
                temp.writeSeries(writer.getWriter(), meta, s3);
            }
        }
    }

    static int[] ensureIntArray(Object array, int length) {
        if (!(array instanceof int[]) || ((int[])array).length != length) {
            return new int[length];
        }
        return (int[])array;
    }

    static float[] ensureFloatArray(Object array, int length) {
        if (!(array instanceof float[]) || ((float[])array).length != length) {
            return new float[length];
        }
        return (float[])array;
    }

    static double[] ensureDoubleArray(Object array, int length) {
        if (!(array instanceof double[]) || ((double[])array).length != length) {
            return new double[length];
        }
        return (double[])array;
    }

    public static OMEPyramidWriter createWriter(OMEPyramidSeries ... series) {
        return new OMEPyramidWriter(Arrays.asList(series));
    }

    public static OMEPyramidWriter createWriter(Collection<OMEPyramidSeries> series) {
        return new OMEPyramidWriter(series);
    }

    static Collection<String> getAvailableCompressionTypes() {
        return Arrays.asList(TiffWriter.COMPRESSION_UNCOMPRESSED, TiffWriter.COMPRESSION_JPEG, TiffWriter.COMPRESSION_J2K, TiffWriter.COMPRESSION_J2K_LOSSY, TiffWriter.COMPRESSION_LZW, TiffWriter.COMPRESSION_ZLIB);
    }

    static String getUncompressedType() {
        return TiffWriter.COMPRESSION_UNCOMPRESSED;
    }

    static boolean isLossyCompressionType(String type) {
        return Arrays.asList(TiffWriter.COMPRESSION_JPEG, TiffWriter.COMPRESSION_J2K_LOSSY).contains(type);
    }

    static CompressionType getDefaultLosslessCompressionType(ImageServer<BufferedImage> server) {
        if (server.getPixelType() == PixelType.UINT8) {
            return CompressionType.LZW;
        }
        return CompressionType.ZLIB;
    }

    static CompressionType getDefaultLossyCompressionType(ImageServer<BufferedImage> server) {
        if (server.isRGB()) {
            return CompressionType.JPEG;
        }
        if (server.getPixelType().getBitsPerPixel() <= 16) {
            return CompressionType.J2K_LOSSY;
        }
        return OMEPyramidWriter.getDefaultLosslessCompressionType(server);
    }

    public static void writeImage(ImageServer<BufferedImage> server, String path) throws FormatException, IOException {
        OMEPyramidWriter.writeImage(server, path, null);
    }

    public static void writeImage(ImageServer<BufferedImage> server, String path, CompressionType compression) throws FormatException, IOException {
        OMEPyramidWriter.writeImage(server, path, compression, (ImageRegion)RegionRequest.createInstance(server), true, true);
    }

    public static void writeImage(ImageServer<BufferedImage> server, String path, CompressionType compression, ImageRegion region) throws FormatException, IOException {
        if (region == null) {
            OMEPyramidWriter.writeImage(server, path, compression);
            return;
        }
        OMEPyramidWriter.writeImage(server, path, compression, region, false, false);
    }

    public static void writeImage(ImageServer<BufferedImage> server, String path, CompressionType compression, ImageRegion region, boolean allZ, boolean allT) throws FormatException, IOException {
        int maxDimension;
        if (region == null) {
            OMEPyramidWriter.writeImage(server, path, compression);
            return;
        }
        Builder builder = new Builder(server).compression(compression == null ? CompressionType.DEFAULT : compression).parallelize().region(region);
        double downsample = server.getDownsampleForResolution(0);
        if (region instanceof RegionRequest) {
            downsample = ((RegionRequest)region).getDownsample();
        }
        if ((maxDimension = (int)((double)Math.max(region.getWidth(), region.getHeight()) / downsample)) > MIN_SIZE_FOR_PYRAMID) {
            builder.scaledDownsampling(downsample, 4.0);
        } else {
            builder.downsamples(downsample);
        }
        if (maxDimension > MIN_SIZE_FOR_TILING) {
            builder.tileSize(DEFAULT_TILE_SIZE);
        }
        if (allZ) {
            builder.allZSlices();
        }
        if (allT) {
            builder.allTimePoints();
        }
        builder.build().writeSeries(path);
    }

    public static class OMEPyramidSeries {
        private ImageServer<BufferedImage> serverOriginal;
        private ImageServer<BufferedImage> serverPyramidalized;
        private PixelType exportPixelType;
        private String name;
        private double[] downsamples;
        private int tileWidth;
        private int tileHeight;
        private int x;
        private int y;
        private int width;
        private int height;
        private int zStart = 0;
        private int zEnd = 0;
        private int tStart = 0;
        private int tEnd = 0;
        private int[] channels;
        private ByteOrder endian = ByteOrder.BIG_ENDIAN;
        private int parallelThreads = 1;
        private Boolean bigTiff;
        private ChannelExportType channelExportType = ChannelExportType.DEFAULT;
        private CompressionType compression = CompressionType.DEFAULT;
        private CodecOptions codecOptions;
        private static int[] RGB_CHANNEL_ARRAY = new int[]{0, 1, 2};
        private ThreadLocal<Object> pixelBuffer = new ThreadLocal();

        private OMEPyramidSeries() {
        }

        void initializeMetadata(IMetadata meta, int series) throws IOException {
            meta.setImageID("Image:" + series, series);
            meta.setPixelsID("Pixels:" + series, series);
            if (this.name != null) {
                meta.setImageName(this.name, series);
            }
            meta.setPixelsBigEndian(Boolean.valueOf(ByteOrder.BIG_ENDIAN.equals(this.endian)), series);
            meta.setPixelsDimensionOrder(DimensionOrder.XYCZT, series);
            PixelType pixelType = this.getExportPixelType();
            switch (pixelType) {
                case INT8: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.INT8, series);
                    break;
                }
                case UINT8: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.UINT8, series);
                    break;
                }
                case INT16: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.INT16, series);
                    break;
                }
                case UINT16: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.UINT16, series);
                    break;
                }
                case INT32: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.INT32, series);
                    break;
                }
                case UINT32: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.UINT32, series);
                    break;
                }
                case FLOAT32: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.FLOAT, series);
                    break;
                }
                case FLOAT64: {
                    meta.setPixelsType(ome.xml.model.enums.PixelType.DOUBLE, series);
                    break;
                }
                default: {
                    throw new IOException("Cannot convert pixel type value of " + String.valueOf(pixelType) + " into a valid OME PixelType");
                }
            }
            meta.setPixelsSizeX(new PositiveInteger(Integer.valueOf((int)((double)this.width / this.downsamples[0]))), series);
            meta.setPixelsSizeY(new PositiveInteger(Integer.valueOf((int)((double)this.height / this.downsamples[0]))), series);
            int sizeZ = this.zEnd - this.zStart;
            int sizeT = this.tEnd - this.tStart;
            if (sizeZ <= 0) {
                throw new IllegalArgumentException("Need to specify positive z-slice range (non-inclusive): requested start " + this.zStart + " and end " + this.zEnd);
            }
            if (sizeT <= 0) {
                throw new IllegalArgumentException("Need to specify positive time point range (non-inclusive): requested start " + this.tStart + " and end " + this.tEnd);
            }
            meta.setPixelsSizeZ(new PositiveInteger(Integer.valueOf(sizeZ)), series);
            meta.setPixelsSizeT(new PositiveInteger(Integer.valueOf(sizeT)), series);
            int nSamples = 1;
            int nChannels = this.channels.length;
            boolean isRGB = this.doExportRGB();
            boolean isInterleaved = false;
            if (this.channelExportType == ChannelExportType.DEFAULT) {
                this.channelExportType = isRGB ? ChannelExportType.INTERLEAVED : ChannelExportType.PLANAR;
            }
            switch (this.channelExportType.ordinal()) {
                case 3: {
                    if (nChannels > 1) {
                        logger.warn("Exporting channels to individual images not yet supported! Will use the default...");
                    }
                }
                case 2: {
                    break;
                }
                default: {
                    isInterleaved = nChannels > 1;
                    nSamples = nChannels;
                }
            }
            if (this.channels.length <= 0) {
                throw new IllegalArgumentException("No channels specified for export!");
            }
            ImageServer<BufferedImage> serverOriginal = this.getOriginalServer();
            meta.setPixelsSizeC(new PositiveInteger(Integer.valueOf(nChannels)), series);
            if (isRGB) {
                meta.setChannelID("Channel:0", series, 0);
                meta.setPixelsInterleaved(Boolean.valueOf(isInterleaved), series);
                meta.setChannelSamplesPerPixel(new PositiveInteger(Integer.valueOf(nSamples)), series, 0);
            } else {
                meta.setChannelSamplesPerPixel(new PositiveInteger(Integer.valueOf(nSamples)), series, 0);
                meta.setPixelsInterleaved(Boolean.valueOf(isInterleaved), series);
                for (int c = 0; c < nChannels; ++c) {
                    meta.setChannelID("Channel:0:" + c, series, c);
                    ImageChannel channel = serverOriginal.getChannel(this.channels[c]);
                    Integer color = channel.getColor();
                    meta.setChannelColor(new Color(ColorTools.red((int)color), ColorTools.green((int)color), ColorTools.blue((int)color), 0), series, c);
                    meta.setChannelName(channel.getName(), series, c);
                }
            }
            PixelCalibration cal = serverOriginal.getPixelCalibration();
            if (cal.hasPixelSizeMicrons()) {
                meta.setPixelsPhysicalSizeX(new Length((Number)(cal.getPixelWidthMicrons() * this.downsamples[0]), UNITS.MICROMETER), series);
                meta.setPixelsPhysicalSizeY(new Length((Number)(cal.getPixelHeightMicrons() * this.downsamples[0]), UNITS.MICROMETER), series);
            }
            if (!Double.isNaN(cal.getZSpacingMicrons())) {
                meta.setPixelsPhysicalSizeZ(new Length((Number)cal.getZSpacingMicrons(), UNITS.MICROMETER), series);
            }
            if (this.tEnd - this.tStart > 1) {
                logger.warn("I can't currently export time series calibration information, sorry");
            }
            ImageServer<BufferedImage> exportServer = this.getExportServer();
            boolean isCropped = this.x != 0 || this.y != 0 || this.width != exportServer.getWidth() || this.height != exportServer.getHeight();
            for (int level = 0; level < this.downsamples.length; ++level) {
                double d = this.downsamples[level];
                int w = (int)((double)this.width / d);
                int h = (int)((double)this.height / d);
                int exportLevel = ServerTools.getPreferredResolutionLevel(exportServer, (double)d);
                if (!isCropped && exportServer.getDownsampleForResolution(exportLevel) == d) {
                    w = exportServer.getMetadata().getLevel(exportLevel).getWidth();
                    h = exportServer.getMetadata().getLevel(exportLevel).getHeight();
                }
                logger.debug("Setting resolution {}: {} x {}", new Object[]{level, w, h});
                ((IPyramidStore)meta).setResolutionSizeX(new PositiveInteger(Integer.valueOf(w)), series, level);
                ((IPyramidStore)meta).setResolutionSizeY(new PositiveInteger(Integer.valueOf(h)), series, level);
            }
        }

        PixelType getExportPixelType() {
            return this.exportPixelType == null ? this.serverOriginal.getPixelType() : this.exportPixelType;
        }

        boolean doExportRGB() {
            return this.serverOriginal.isRGB() && this.getExportPixelType() == PixelType.UINT8 && Arrays.equals(this.channels, RGB_CHANNEL_ARRAY);
        }

        @Deprecated
        public void writePyramid(String path) throws FormatException, IOException {
            OMEPyramidWriter writer = new OMEPyramidWriter();
            writer.series.add(this);
            writer.writeImage(path);
        }

        public void writeSeries(String path) throws FormatException, IOException {
            OMEPyramidWriter writer = new OMEPyramidWriter();
            writer.series.add(this);
            writer.writeImage(path);
        }

        @Deprecated
        public void writePyramid(PyramidOMETiffWriter writer, IMetadata meta, int series) throws FormatException, IOException {
            this.writeSeries((IFormatWriter)writer, meta, series);
        }

        private static boolean isTiffWriter(IFormatWriter writer) {
            while (writer instanceof ImageWriter) {
                writer = ((ImageWriter)writer).getWriter();
            }
            return writer instanceof TiffWriter;
        }

        public void writeSeries(IFormatWriter writer, IMetadata meta, int series) throws FormatException, IOException {
            boolean isTiled;
            List<Object> supportedCompression;
            while (writer instanceof ImageWriter) {
                writer = ((ImageWriter)writer).getWriter();
            }
            boolean isRGB = this.doExportRGB();
            int nChannels = (Integer)meta.getPixelsSizeC(series).getValue();
            int nSamples = (Integer)meta.getChannelSamplesPerPixel(series, 0).getValue();
            int sizeZ = (Integer)meta.getPixelsSizeZ(series).getValue();
            int sizeT = (Integer)meta.getPixelsSizeT(series).getValue();
            int width = (Integer)meta.getPixelsSizeX(series).getValue();
            int height = (Integer)meta.getPixelsSizeY(series).getValue();
            int nPlanes = nChannels / nSamples * sizeZ * sizeT;
            ImageServer<BufferedImage> server = this.getExportServer();
            String compressionString = this.compression.getOMEString(server);
            String[] compressionTypesArray = writer.getCompressionTypes();
            List<Object> list = supportedCompression = compressionTypesArray == null ? Collections.emptyList() : Arrays.asList(compressionTypesArray);
            if (!this.compression.supportsImage(server) || !supportedCompression.contains(compressionString)) {
                if (OMEPyramidSeries.isTiffWriter(writer)) {
                    compressionString = CompressionType.DEFAULT.getOMEString(server);
                    logger.warn("Requested compression {} incompatible with current image, will use {} instead", (Object)this.compression.getOMEString(server), (Object)compressionString);
                } else {
                    compressionString = null;
                    if (this.compression != CompressionType.DEFAULT) {
                        logger.warn("Requested compression {} incompatible with current image", (Object)compressionString);
                    }
                }
            }
            if (compressionString != null) {
                logger.info("Setting series {} compression to {}", (Object)series, (Object)compressionString);
                writer.setCompression(compressionString);
            }
            writer.setInterleaved(meta.getPixelsInterleaved(series).booleanValue());
            int tileWidth = this.tileWidth;
            int tileHeight = this.tileHeight;
            boolean bl = isTiled = tileWidth > 0 && tileHeight > 0;
            if (isTiled) {
                tileWidth = writer.setTileSizeX(tileWidth);
                tileHeight = writer.setTileSizeY(tileHeight);
                if (this.tileWidth != tileWidth || this.tileHeight != tileHeight) {
                    logger.warn("Requested tile size {}x{}, tile size accepted by image writer {}x{}", new Object[]{this.tileWidth, this.tileHeight, tileWidth, tileHeight});
                }
            }
            if (server.getMetadata().getChannelType() == ImageServerMetadata.ChannelType.CLASSIFICATION) {
                try {
                    writer.setColorModel((ColorModel)ColorModelFactory.getIndexedClassificationColorModel((Map)server.getMetadata().getClassificationLabels()));
                }
                catch (Exception e) {
                    logger.warn("Error setting classification color model: {}", (Object)e.getLocalizedMessage());
                }
            }
            writer.setSeries(series);
            boolean isTiff = writer instanceof TiffWriter;
            HashMap<Integer, IFD> map = new HashMap<Integer, IFD>();
            writer.setSeries(series);
            for (int level = 0; level < this.downsamples.length; ++level) {
                writer.setResolution(level);
                if (isTiff) {
                    map.clear();
                    for (int i2 = 0; i2 < nPlanes; ++i2) {
                        IFD ifd = new IFD();
                        if (isTiled) {
                            ifd.put((Object)322, (Object)tileWidth);
                            ifd.put((Object)323, (Object)tileHeight);
                        }
                        if (nSamples > 1 && !isRGB) {
                            ifd.put((Object)338, (Object)new short[nSamples - 1]);
                        }
                        map.put(i2, ifd);
                    }
                }
                double d = this.downsamples[level];
                int w = width;
                int h = height;
                if (meta instanceof IPyramidStore && level > 0) {
                    w = (Integer)((IPyramidStore)meta).getResolutionSizeX(series, level).getValue();
                    h = (Integer)((IPyramidStore)meta).getResolutionSizeY(series, level).getValue();
                }
                int tInc = this.tEnd >= this.tStart ? 1 : -1;
                int zInc = this.zEnd >= this.zStart ? 1 : -1;
                int effectiveSizeC = nChannels / nSamples;
                AtomicInteger count = new AtomicInteger(0);
                int ti = 0;
                for (int t = this.tStart; t < this.tEnd; t += tInc) {
                    int zi = 0;
                    for (int z = this.zStart; z < this.zEnd; z += zInc) {
                        ArrayList<TileRequest> tiles = new ArrayList<TileRequest>();
                        int levelTemp = ServerTools.getPreferredResolutionLevel(server, (double)d);
                        if (d == server.getDownsampleForResolution(levelTemp) && this.x == 0 && this.y == 0 && w == server.getMetadata().getLevel(levelTemp).getWidth() && h == server.getMetadata().getLevel(levelTemp).getHeight() && tileWidth == server.getMetadata().getPreferredTileWidth() && tileHeight == server.getMetadata().getPreferredTileHeight()) {
                            logger.debug("Using tile requests directly for level {}", (Object)level);
                            logger.trace("Tiled level: {} ({})", (Object)level, (Object)server.getMetadata().getLevel(level));
                            int thisZ = z;
                            int thisT = t;
                            server.getTileRequestManager().getTileRequestsForLevel(levelTemp).stream().filter(tile -> tile.getZ() == thisZ && tile.getT() == thisT).forEachOrdered(tiles::add);
                        } else {
                            for (int yy = 0; yy < h; yy += tileHeight) {
                                int hh = Math.min(h - yy, tileHeight);
                                for (int xx = 0; xx < w; xx += tileWidth) {
                                    int ww = Math.min(w - xx, tileWidth);
                                    ImageRegion region = ImageRegion.createInstance((int)xx, (int)yy, (int)ww, (int)hh, (int)z, (int)t);
                                    tiles.add(TileRequest.createInstance((String)server.getPath(), (int)level, (double)d, (ImageRegion)region));
                                }
                            }
                        }
                        int total = tiles.size() * (this.tEnd - this.tStart) * (this.zEnd - this.zStart);
                        if (z == this.zStart && t == this.tStart) {
                            logger.info("Writing resolution {} of {} (downsample={}, {} tiles)", new Object[]{level + 1, this.downsamples.length, d, total});
                        }
                        TileRequest firstTile = (TileRequest)tiles.remove(0);
                        int inc = total > 1000 ? 20 : 10;
                        Set keyCounts = IntStream.range(1, inc).mapToObj(i -> (int)Math.round((double)total / (double)inc * (double)i)).collect(Collectors.toCollection(() -> new HashSet()));
                        keyCounts.add(total - 1);
                        for (int ci = 0; ci < effectiveSizeC; ++ci) {
                            int[] nArray;
                            IFD ifd;
                            long planeStartTime = System.currentTimeMillis();
                            count.set(0);
                            final int plane = ti * sizeZ * effectiveSizeC + zi * effectiveSizeC + ci;
                            IFD iFD = ifd = isTiff ? (IFD)map.get(plane) : null;
                            if (effectiveSizeC == this.channels.length) {
                                int[] nArray2 = new int[1];
                                nArray = nArray2;
                                nArray2[0] = this.channels[ci];
                            } else {
                                nArray = this.channels;
                            }
                            int[] localChannels = nArray;
                            logger.info("Writing plane {}/{}", (Object)(plane + 1), (Object)nPlanes);
                            this.writeRegion(writer, plane, ifd, server, firstTile, isRGB, localChannels);
                            if (tiles.isEmpty()) continue;
                            if (ci > 0 || level > 0) {
                                logger.trace("Reversing list if {} regions", (Object)tiles.size());
                                Collections.reverse(tiles);
                            }
                            final IFormatWriter localWriter = writer;
                            List<1> tasks = tiles.stream().map(tile -> new Runnable(){
                                final /* synthetic */ ImageServer val$server;
                                final /* synthetic */ TileRequest val$tile;
                                final /* synthetic */ boolean val$isRGB;
                                final /* synthetic */ int[] val$localChannels;
                                final /* synthetic */ double val$d;
                                final /* synthetic */ AtomicInteger val$count;
                                final /* synthetic */ int val$total;
                                final /* synthetic */ Set val$keyCounts;
                                {
                                    this.val$server = imageServer;
                                    this.val$tile = tileRequest;
                                    this.val$isRGB = bl;
                                    this.val$localChannels = nArray;
                                    this.val$d = d;
                                    this.val$count = atomicInteger;
                                    this.val$total = n2;
                                    this.val$keyCounts = set;
                                }

                                /*
                                 * WARNING - Removed try catching itself - possible behaviour change.
                                 */
                                @Override
                                public void run() {
                                    try {
                                        if (Thread.currentThread().isInterrupted()) {
                                            return;
                                        }
                                        this.writeRegion(localWriter, plane, ifd, (ImageServer<BufferedImage>)this.val$server, this.val$tile, this.val$isRGB, this.val$localChannels);
                                    }
                                    catch (Exception e) {
                                        logger.error(String.format("Error writing %s (downsample=%.2f)", this.val$tile.toString(), this.val$d), (Throwable)e);
                                    }
                                    finally {
                                        int localCount = this.val$count.incrementAndGet();
                                        if (this.val$total > 20 && this.val$keyCounts.size() > 1 && this.val$keyCounts.contains(localCount)) {
                                            double percentage = (double)localCount * 100.0 / (double)this.val$total;
                                            logger.info("Written {}% tiles", (Object)Math.round(percentage));
                                        }
                                    }
                                }
                            }).toList();
                            if (this.parallelThreads > 1) {
                                ExecutorService pool = Executors.newWorkStealingPool(this.parallelThreads);
                                for (1 task : tasks) {
                                    pool.submit(task);
                                }
                                pool.shutdown();
                                try {
                                    pool.awaitTermination(tiles.size(), TimeUnit.MINUTES);
                                    logger.info("Plane written in {} ms", (Object)(System.currentTimeMillis() - planeStartTime));
                                    continue;
                                }
                                catch (InterruptedException e) {
                                    logger.warn("OME-TIFF export interrupted!");
                                    pool.shutdownNow();
                                    throw new IOException("Error writing regions", e);
                                }
                            }
                            for (1 task : tasks) {
                                if (Thread.currentThread().isInterrupted()) {
                                    throw new IOException("Interrupted writing regions!");
                                }
                                task.run();
                            }
                            logger.info("Plane written in {} ms", (Object)(System.currentTimeMillis() - planeStartTime));
                        }
                        ++zi;
                    }
                    ++ti;
                }
            }
            logger.trace("Image count: {}", (Object)meta.getImageCount());
            if (writer instanceof FormatWriter) {
                logger.trace("Plane count: {}", (Object)((TiffWriter)writer).getPlaneCount());
            }
            logger.trace("Resolution count: {}", (Object)writer.getResolutionCount());
        }

        private ImageServer<BufferedImage> getOriginalServer() {
            return this.serverOriginal;
        }

        private ImageServer<BufferedImage> getExportServer() {
            return this.serverPyramidalized == null ? this.getOriginalServer() : this.serverPyramidalized;
        }

        private void writeRegion(IFormatWriter writer, int plane, IFD ifd, ImageServer<BufferedImage> server, TileRequest tile, boolean isRGB, int[] channels) throws FormatException, IOException {
            RegionRequest request = tile.getRegionRequest().translate(this.x, this.y);
            BufferedImage img = (BufferedImage)server.readRegion(request);
            PixelType pixelType = this.getExportPixelType();
            int bytesPerPixel = pixelType.getBytesPerPixel();
            int nChannels = channels.length;
            if (img == null) {
                byte[] zeros = new byte[tile.getTileWidth() * tile.getTileHeight() * bytesPerPixel * nChannels];
                if (writer instanceof TiffWriter) {
                    ((TiffWriter)writer).saveBytes(plane, zeros, ifd, tile.getTileX(), tile.getTileY(), tile.getTileWidth(), tile.getTileHeight());
                } else {
                    writer.saveBytes(plane, zeros, tile.getTileX(), tile.getTileY(), tile.getTileWidth(), tile.getTileHeight());
                }
                return;
            }
            int ww = img.getWidth();
            int hh = img.getHeight();
            ByteBuffer buf = ByteBuffer.allocate(ww * hh * bytesPerPixel * nChannels).order(this.endian);
            if (isRGB) {
                int[] rgba;
                Object pixelBuffer = this.getPixelBuffer(ww * hh, pixelType);
                if (!(pixelBuffer instanceof int[])) {
                    pixelBuffer = null;
                }
                for (int val : rgba = img.getRGB(0, 0, ww, hh, (int[])pixelBuffer, 0, ww)) {
                    buf.put((byte)ColorTools.red((int)val));
                    buf.put((byte)ColorTools.green((int)val));
                    buf.put((byte)ColorTools.blue((int)val));
                }
            } else {
                for (int ci = 0; ci < channels.length; ++ci) {
                    int c = channels[ci];
                    int ind = ci * bytesPerPixel;
                    this.channelToBuffer(img.getRaster(), c, buf, ind, channels.length * bytesPerPixel, pixelType);
                }
            }
            if (writer instanceof TiffWriter) {
                ((TiffWriter)writer).saveBytes(plane, buf.array(), ifd, tile.getTileX(), tile.getTileY(), ww, hh);
            } else {
                writer.saveBytes(plane, buf.array(), tile.getTileX(), tile.getTileY(), ww, hh);
            }
        }

        boolean channelToBuffer(WritableRaster raster, int c, ByteBuffer buf, int startInd, int inc, PixelType pixelType) {
            int ind = startInd;
            int ww = raster.getWidth();
            int hh = raster.getHeight();
            int n = ww * hh;
            Object pixelBuffer = this.getPixelBuffer(n, pixelType);
            switch (pixelType) {
                case INT8: 
                case UINT8: 
                case INT16: 
                case UINT16: 
                case INT32: 
                case UINT32: {
                    int[] pixelsInt;
                    int[] nArray = pixelsInt = pixelBuffer instanceof int[] ? (int[])pixelBuffer : null;
                    if (pixelsInt == null || pixelsInt.length < n) {
                        pixelsInt = new int[n];
                    }
                    pixelsInt = raster.getSamples(0, 0, ww, hh, c, pixelsInt);
                    if (pixelType.getBitsPerPixel() == 8) {
                        for (int i = 0; i < n; ++i) {
                            buf.put(ind, (byte)pixelsInt[i]);
                            ind += inc;
                        }
                    } else if (pixelType.getBitsPerPixel() == 16) {
                        for (int i = 0; i < n; ++i) {
                            buf.putShort(ind, (short)pixelsInt[i]);
                            ind += inc;
                        }
                    } else if (pixelType.getBitsPerPixel() == 32) {
                        for (int i = 0; i < n; ++i) {
                            buf.putInt(ind, pixelsInt[i]);
                            ind += inc;
                        }
                    }
                    return true;
                }
                case FLOAT32: {
                    float[] pixelsFloat;
                    float[] fArray = pixelsFloat = pixelBuffer instanceof float[] ? (float[])pixelBuffer : null;
                    if (pixelsFloat == null || pixelsFloat.length < n) {
                        pixelsFloat = new float[n];
                    }
                    pixelsFloat = raster.getSamples(0, 0, ww, hh, c, pixelsFloat);
                    for (int i = 0; i < n; ++i) {
                        buf.putFloat(ind, pixelsFloat[i]);
                        ind += inc;
                    }
                    return true;
                }
                case FLOAT64: {
                    double[] pixelsDouble;
                    double[] dArray = pixelsDouble = pixelBuffer instanceof double[] ? (double[])pixelBuffer : null;
                    if (pixelsDouble == null || pixelsDouble.length < n) {
                        pixelsDouble = new double[n];
                    }
                    pixelsDouble = raster.getSamples(0, 0, ww, hh, c, pixelsDouble);
                    for (int i = 0; i < n; ++i) {
                        buf.putDouble(ind, pixelsDouble[i]);
                        ind += inc;
                    }
                    return true;
                }
            }
            logger.warn("Cannot convert to buffer - unknown pixel type {}", (Object)pixelType);
            return false;
        }

        Object getPixelBuffer(int length, PixelType pixelType) {
            Object originalBuffer = this.pixelBuffer.get();
            Object[] updatedBuffer = null;
            switch (pixelType) {
                case FLOAT32: {
                    updatedBuffer = OMEPyramidWriter.ensureFloatArray(originalBuffer, length);
                    break;
                }
                case FLOAT64: {
                    updatedBuffer = OMEPyramidWriter.ensureDoubleArray(originalBuffer, length);
                    break;
                }
                default: {
                    updatedBuffer = OMEPyramidWriter.ensureIntArray(originalBuffer, length);
                }
            }
            if (updatedBuffer != originalBuffer) {
                this.pixelBuffer.set(updatedBuffer);
            }
            return updatedBuffer;
        }
    }

    public static enum CompressionType {
        UNCOMPRESSED,
        DEFAULT,
        JPEG,
        J2K,
        J2K_LOSSY,
        LZW,
        ZLIB;


        public String getOMEString(ImageServer<?> server) {
            switch (this.ordinal()) {
                case 3: {
                    return OMETiffWriter.COMPRESSION_J2K;
                }
                case 4: {
                    return OMETiffWriter.COMPRESSION_J2K_LOSSY;
                }
                case 2: {
                    return OMETiffWriter.COMPRESSION_JPEG;
                }
                case 0: {
                    return OMETiffWriter.COMPRESSION_UNCOMPRESSED;
                }
                case 5: {
                    return OMETiffWriter.COMPRESSION_LZW;
                }
                case 6: {
                    return OMETiffWriter.COMPRESSION_ZLIB;
                }
            }
            if (server.isRGB() && server.nResolutions() > 1) {
                return OMETiffWriter.COMPRESSION_JPEG;
            }
            if (server.getPixelType() == PixelType.UINT8) {
                return OMETiffWriter.COMPRESSION_LZW;
            }
            return OMETiffWriter.COMPRESSION_ZLIB;
        }

        public boolean supportsImage(ImageServer<?> server) {
            return this.supportsImage(server.getPixelType(), server.nChannels(), server.isRGB());
        }

        public boolean supportsImage(PixelType pixelType, int nChannels, boolean isRGB) {
            switch (this.ordinal()) {
                case 2: {
                    return isRGB || nChannels >= 1 && pixelType == PixelType.UINT8;
                }
                case 3: 
                case 4: {
                    return pixelType.getBytesPerPixel() <= 2;
                }
                case 0: 
                case 1: 
                case 5: 
                case 6: {
                    return true;
                }
            }
            return false;
        }

        public String toFriendlyString() {
            switch (this.ordinal()) {
                case 1: {
                    return "Default (lossless or lossy)";
                }
                case 3: {
                    return "JPEG-2000 (lossless)";
                }
                case 4: {
                    return "JPEG-2000 (lossy)";
                }
                case 2: {
                    return "JPEG (lossy)";
                }
                case 0: {
                    return "Uncompressed";
                }
                case 5: {
                    return "LZW (lossless)";
                }
                case 6: {
                    return "ZLIB (lossless)";
                }
            }
            throw new IllegalArgumentException("Unknown compression type: " + String.valueOf((Object)this));
        }

        public static CompressionType fromFriendlyString(String friendlyCompression) {
            for (CompressionType compression : CompressionType.values()) {
                if (!friendlyCompression.equals(compression.toFriendlyString())) continue;
                return compression;
            }
            throw new IllegalArgumentException("Unknown compression type: " + friendlyCompression);
        }
    }

    public static class Builder {
        private OMEPyramidSeries series = new OMEPyramidSeries();

        public Builder(ImageServer<BufferedImage> server) {
            this.series.serverOriginal = server;
            this.series.x = 0;
            this.series.y = 0;
            this.series.width = server.getWidth();
            this.series.height = server.getHeight();
            this.series.downsamples = server.getPreferredDownsamples();
            if (server.getMetadata().getPreferredTileWidth() >= server.getWidth() && server.getMetadata().getPreferredTileHeight() >= server.getHeight()) {
                this.series.tileWidth = server.getMetadata().getPreferredTileWidth();
                this.series.tileHeight = server.getMetadata().getPreferredTileHeight();
            } else {
                this.series.tileWidth = 256;
                this.series.tileHeight = 256;
            }
            this.series.zStart = 0;
            this.series.zEnd = server.nZSlices();
            this.series.tStart = 0;
            this.series.tEnd = server.nTimepoints();
            this.series.channels = server.getMetadata().getChannelType() == ImageServerMetadata.ChannelType.CLASSIFICATION ? new int[]{0} : IntStream.range(0, server.nChannels()).toArray();
        }

        public Builder channelsPlanar() {
            this.series.channelExportType = ChannelExportType.PLANAR;
            return this;
        }

        public Builder channelsInterleaved() {
            this.series.channelExportType = ChannelExportType.INTERLEAVED;
            return this;
        }

        public Builder channelsImages() {
            this.series.channelExportType = ChannelExportType.IMAGES;
            return this;
        }

        public Builder bigTiff() {
            this.series.bigTiff = Boolean.TRUE;
            return this;
        }

        public Builder bigTiff(boolean doBigTiff) {
            this.series.bigTiff = doBigTiff;
            return this;
        }

        public Builder compression(CompressionType compression) {
            this.series.compression = compression;
            return this;
        }

        public Builder lossyCompression() {
            this.series.compression = OMEPyramidWriter.getDefaultLossyCompressionType(this.series.serverOriginal);
            return this;
        }

        public Builder losslessCompression() {
            this.series.compression = OMEPyramidWriter.getDefaultLosslessCompressionType(this.series.serverOriginal);
            return this;
        }

        public Builder uncompressed() {
            this.series.compression = CompressionType.UNCOMPRESSED;
            return this;
        }

        public Builder parallelize() {
            return this.parallelize(true);
        }

        public Builder parallelize(boolean doParallel) {
            return this.parallelize(doParallel ? 4 : 1);
        }

        public Builder parallelize(int nThreads) {
            this.series.parallelThreads = nThreads;
            return this;
        }

        public Builder pixelType(PixelType exportPixelType) {
            this.series.exportPixelType = exportPixelType;
            return this;
        }

        public Builder pixelType(String exportPixelType) {
            if (exportPixelType == null) {
                return this.pixelType((PixelType)null);
            }
            try {
                this.series.exportPixelType = PixelType.valueOf((String)exportPixelType.toUpperCase());
            }
            catch (Exception e) {
                logger.warn("{} is not a valid pixel type! Supported values are {}", (Object)exportPixelType, Arrays.asList(PixelType.values()));
            }
            return this;
        }

        public Builder allZSlices() {
            return this.zSlices(0, this.series.serverOriginal.nZSlices());
        }

        public Builder zSlice(int z) {
            return this.zSlices(z, z + 1);
        }

        public Builder zSlices(int zStart, int zEnd) {
            if (zStart < 0) {
                logger.warn("First z-slice (" + zStart + ") is out of bounds. Will use 0 instead.");
                zStart = 0;
            }
            if (zEnd > this.series.serverOriginal.nZSlices()) {
                logger.warn("Last z-slice (" + zEnd + ") is out of bounds. Will use " + this.series.serverOriginal.nZSlices() + " instead.");
                zEnd = this.series.serverOriginal.nZSlices();
            }
            this.series.zStart = zStart;
            this.series.zEnd = zEnd;
            return this;
        }

        public Builder timePoint(int t) {
            return this.timePoints(t, t + 1);
        }

        public Builder allTimePoints() {
            return this.timePoints(0, this.series.serverOriginal.nTimepoints());
        }

        public Builder timePoints(int tStart, int tEnd) {
            if (tStart < 0) {
                logger.warn("First timepoint (" + tStart + ") is out of bounds. Will use 0 instead.");
                tStart = 0;
            }
            if (tEnd > this.series.serverOriginal.nTimepoints()) {
                logger.warn("Last timepoint (" + tEnd + ") is out of bounds. Will use " + this.series.serverOriginal.nTimepoints() + " instead.");
                tEnd = this.series.serverOriginal.nTimepoints();
            }
            this.series.tStart = tStart;
            this.series.tEnd = tEnd;
            return this;
        }

        public Builder name(String name) {
            this.series.name = name;
            return this;
        }

        public Builder region(int x, int y, int width, int height) {
            this.series.x = x;
            this.series.y = y;
            this.series.width = width;
            this.series.height = height;
            return this;
        }

        public Builder region(ImageRegion region) {
            return this.region(region.getX(), region.getY(), region.getWidth(), region.getHeight()).zSlice(region.getZ()).timePoint(region.getT());
        }

        public Builder tileSize(int tileSize) {
            return this.tileSize(tileSize, tileSize);
        }

        public Builder tileSize(int tileWidth, int tileHeight) {
            this.series.tileWidth = tileWidth;
            this.series.tileHeight = tileHeight;
            return this;
        }

        public Builder downsamples(double ... downsamples) {
            this.series.downsamples = downsamples;
            return this;
        }

        public Builder dyadicDownsampling() {
            return this.scaledDownsampling(1.0, 2.0);
        }

        public Builder scaledDownsampling(double scale) {
            return this.scaledDownsampling(1.0, scale);
        }

        public Builder scaledDownsampling(double minDownsample, double scale) {
            ArrayList<Double> downsampleList = new ArrayList<Double>();
            double nextDownsample = minDownsample;
            do {
                downsampleList.add(nextDownsample);
            } while ((int)((double)this.series.width / (nextDownsample *= scale)) > this.series.tileWidth && (int)((double)this.series.height / nextDownsample) > this.series.tileHeight);
            this.series.downsamples = downsampleList.stream().mapToDouble(d -> d).toArray();
            return this;
        }

        public Builder channels(int ... channels) {
            this.series.channels = channels;
            return this;
        }

        public Builder codecOptions(CodecOptions codecOptions) {
            this.series.codecOptions = codecOptions;
            return this;
        }

        public OMEPyramidSeries build() {
            int lastDownsample;
            int nChannels = this.series.serverOriginal.nChannels();
            if ((nChannels == 2 || nChannels > 3) && this.series.channelExportType == ChannelExportType.INTERLEAVED && this.series.compression == CompressionType.JPEG) {
                logger.warn("JPEG compression not possible in INTERLEAVED mode for nChannel = 2 or > 3.Will be corrected to PLANAR");
                logger.warn("CompressionType will be changed to PLANAR");
                this.series.channelExportType = ChannelExportType.PLANAR;
            }
            Arrays.sort(this.series.downsamples);
            for (lastDownsample = 1; lastDownsample < this.series.downsamples.length && (double)this.series.width / this.series.downsamples[lastDownsample] > 16.0 && (double)this.series.height / this.series.downsamples[lastDownsample] > 16.0; ++lastDownsample) {
            }
            if (lastDownsample < this.series.downsamples.length) {
                this.series.downsamples = Arrays.copyOf(this.series.downsamples, lastDownsample);
            }
            if (this.series.downsamples.length > 1 && (this.series.serverOriginal.nResolutions() == 1 || this.series.serverOriginal.getDownsampleForResolution(0) < this.series.downsamples[0])) {
                logger.info("Creating pyramidal server");
                this.series.serverPyramidalized = ImageServers.pyramidalizeTiled(this.series.serverOriginal, (int)this.series.tileWidth, (int)this.series.tileHeight, (double[])this.series.downsamples);
            }
            return this.series;
        }
    }

    public static enum ChannelExportType {
        DEFAULT,
        INTERLEAVED,
        PLANAR,
        IMAGES;

    }
}

